/*
 * Copyright 2020-2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef __AudioIO_h__
#define __AudioIO_h__

#ifdef __cplusplus
extern "C"{
#endif


#define NUM_I2S_INPUT  4
#define NUM_I2S_OUTPUT 2

//part 1 --- SAI
//4038_4000 for SAI1; 4038_8000 for SAI2; 4038_C000 for SAI3 --- only use
#define SAI1_BaseRegAddr 0x40384000
#define SAI2_BaseRegAddr 0x40388000
#define SAI3_BaseRegAddr 0x4038c000


#define     __I     volatile const       /*!< Defines 'read only' permissions */
#define     __O     volatile             /*!< Defines 'write only' permissions */
#define     __IO    volatile             /*!< Defines 'read / write' permissions */

typedef struct {
    __I  uint32_t VERID;                             /**< Version ID Register, offset: 0x0 */
    __I  uint32_t PARAM;                             /**< Parameter Register, offset: 0x4 */
    __IO uint32_t TCSR;                              /**< SAI Transmit Control Register, offset: 0x8 */
    __IO uint32_t TCR1;                              /**< SAI Transmit Configuration 1 Register, offset: 0xC */
    __IO uint32_t TCR2;                              /**< SAI Transmit Configuration 2 Register, offset: 0x10 */
    __IO uint32_t TCR3;                              /**< SAI Transmit Configuration 3 Register, offset: 0x14 */
    __IO uint32_t TCR4;                              /**< SAI Transmit Configuration 4 Register, offset: 0x18 */
    __IO uint32_t TCR5;                              /**< SAI Transmit Configuration 5 Register, offset: 0x1C */
    __O  uint32_t TDR[4];                            /**< SAI Transmit Data Register, array offset: 0x20, array step: 0x4 */
            uint8_t RESERVED_0[16];     //not useful
    __I  uint32_t TFR[4];                            /**< SAI Transmit FIFO Register, array offset: 0x40, array step: 0x4 */
            uint8_t RESERVED_1[16];     //not useful
    __IO uint32_t TMR;                               /**< SAI Transmit Mask Register, offset: 0x60 */
            uint8_t RESERVED_2[36];     //not useful
    __IO uint32_t RCSR;                              /**< SAI Receive Control Register, offset: 0x88 */
    __IO uint32_t RCR1;                              /**< SAI Receive Configuration 1 Register, offset: 0x8C */
    __IO uint32_t RCR2;                              /**< SAI Receive Configuration 2 Register, offset: 0x90 */
    __IO uint32_t RCR3;                              /**< SAI Receive Configuration 3 Register, offset: 0x94 */
    __IO uint32_t RCR4;                              /**< SAI Receive Configuration 4 Register, offset: 0x98 */
    __IO uint32_t RCR5;                              /**< SAI Receive Configuration 5 Register, offset: 0x9C */
    __I  uint32_t RDR[4];                            /**< SAI Receive Data Register, array offset: 0xA0, array step: 0x4 */
            uint8_t RESERVED_3[16];     //not useful
    __I  uint32_t RFR[4];                            /**< SAI Receive FIFO Register, array offset: 0xC0, array step: 0x4 */
            uint8_t RESERVED_4[16];     //not useful
    __IO uint32_t RMR;                               /**< SAI Receive Mask Register, offset: 0xE0 */
} SAII2S_RegStructType;



#define DMA_MUX_BaseRegAddr 0x400EC000
typedef struct {
    __IO uint32_t CHCFG[32];                         /**< Channel 0 Configuration Register..Channel 31 Configuration Register, array offset: 0x0, array step: 0x4 */
} DMAMUX_RegStructType;


#define DMA_BaseRegAddr 0x400E8000
typedef struct
{
    //__IO uint32_t SADDR;     /*!< SADDR register, used to save source address */
    //__IO uint16_t SOFF;      /*!< SOFF register, save offset bytes every transfer */
    //__IO uint16_t ATTR;      /*!< ATTR register, source/destination transfer size and modulo */
    //__IO uint32_t NBYTES;    /*!< Nbytes register, minor loop length in bytes */
    //__IO uint32_t SLAST;     /*!< SLAST register */
    //__IO uint32_t DADDR;     /*!< DADDR register, used for destination address */
    //__IO uint16_t DOFF;      /*!< DOFF register, used for destination offset */
    //__IO uint16_t CITER;     /*!< CITER register, current minor loop numbers, for unfinished minor loop.*/
    //__IO uint32_t DLAST_SGA; /*!< DLASTSGA register, next stcd address used in scatter-gather mode */
    //__IO uint16_t CSR;       /*!< CSR register, for TCD control status */
    //__IO uint16_t BITER;     /*!< BITER register, begin minor loop count. */

    __IO uint32_t SADDR;                         /**< TCD Source Address, array offset: 0x1000, array step: 0x20 */
    __IO uint16_t SOFF;                          /**< TCD Signed Source Address Offset, array offset: 0x1004, array step: 0x20 */
    __IO uint16_t ATTR;                          /**< TCD Transfer Attributes, array offset: 0x1006, array step: 0x20 */
    union
    {                                            /* offset: 0x1008, array step: 0x20 */
        __IO uint32_t NBYTES_MLNO;               /**< TCD Minor Byte Count (Minor Loop Mapping Disabled), array offset: 0x1008, array step: 0x20 */
        __IO uint32_t NBYTES_MLOFFNO;            /**< TCD Signed Minor Loop Offset (Minor Loop Mapping Enabled and Offset Disabled), array offset: 0x1008, array step: 0x20 */
        __IO uint32_t NBYTES_MLOFFYES;           /**< TCD Signed Minor Loop Offset (Minor Loop Mapping and Offset Enabled), array offset: 0x1008, array step: 0x20 */
    };
    __IO uint32_t SLAST;                         /**< TCD Last Source Address Adjustment, array offset: 0x100C, array step: 0x20 */
    __IO uint32_t DADDR;                         /**< TCD Destination Address, array offset: 0x1010, array step: 0x20 */
    __IO uint16_t DOFF;                          /**< TCD Signed Destination Address Offset, array offset: 0x1014, array step: 0x20 */
    union
    {                                            /* offset: 0x1016, array step: 0x20 */
        __IO uint16_t CITER_ELINKNO;             /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x1016, array step: 0x20 */
        __IO uint16_t CITER_ELINKYES;            /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x1016, array step: 0x20 */
    };
    __IO uint32_t DLAST_SGA;                     /**< TCD Last Destination Address Adjustment/Scatter Gather Address, array offset: 0x1018, array step: 0x20 */
    __IO uint16_t CSR;                           /**< TCD Control and Status, array offset: 0x101C, array step: 0x20 */
    union
    {                                            /* offset: 0x101E, array step: 0x20 */
    __IO uint16_t BITER_ELINKNO;                 /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x101E, array step: 0x20 */
    __IO uint16_t BITER_ELINKYES;                /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x101E, array step: 0x20 */
    };

} DMA_TCD_Type;

typedef struct {
    __IO uint32_t CR;                                /**< Control Register, offset: 0x0 */
    __I  uint32_t ES;                                /**< Error Status Register, offset: 0x4 */
            uint8_t RESERVED_0[4];     //not useful
    __IO uint32_t ERQ;                               /**< Enable Request Register, offset: 0xC */
            uint8_t RESERVED_1[4];     //not useful
    __IO uint32_t EEI;                               /**< Enable Error Interrupt Register, offset: 0x14 */
    __O  uint8_t CEEI;                               /**< Clear Enable Error Interrupt Register, offset: 0x18 */
    __O  uint8_t SEEI;                               /**< Set Enable Error Interrupt Register, offset: 0x19 */
    __O  uint8_t CERQ;                               /**< Clear Enable Request Register, offset: 0x1A */
    __O  uint8_t SERQ;                               /**< Set Enable Request Register, offset: 0x1B */
    __O  uint8_t CDNE;                               /**< Clear DONE Status Bit Register, offset: 0x1C */
    __O  uint8_t SSRT;                               /**< Set START Bit Register, offset: 0x1D */
    __O  uint8_t CERR;                               /**< Clear Error Register, offset: 0x1E */
    __O  uint8_t CINT;                               /**< Clear Interrupt Request Register, offset: 0x1F */
            uint8_t RESERVED_2[4];     //not useful
    __IO uint32_t INT;                               /**< Interrupt Request Register, offset: 0x24 */
            uint8_t RESERVED_3[4];     //not useful
    __IO uint32_t ERR;                               /**< Error Register, offset: 0x2C */
            uint8_t RESERVED_4[4];     //not useful
    __I  uint32_t HRS;                               /**< Hardware Request Status Register, offset: 0x34 */
            uint8_t RESERVED_5[12];    //not useful
    __IO uint32_t EARS;                              /**< Enable Asynchronous Request in Stop Register, offset: 0x44 */
            uint8_t RESERVED_6[184];   //not useful
    __IO uint8_t DCHPRI3;                            /**< Channel n Priority Register, offset: 0x100 */
    __IO uint8_t DCHPRI2;                            /**< Channel n Priority Register, offset: 0x101 */
    __IO uint8_t DCHPRI1;                            /**< Channel n Priority Register, offset: 0x102 */
    __IO uint8_t DCHPRI0;                            /**< Channel n Priority Register, offset: 0x103 */
    __IO uint8_t DCHPRI7;                            /**< Channel n Priority Register, offset: 0x104 */
    __IO uint8_t DCHPRI6;                            /**< Channel n Priority Register, offset: 0x105 */
    __IO uint8_t DCHPRI5;                            /**< Channel n Priority Register, offset: 0x106 */
    __IO uint8_t DCHPRI4;                            /**< Channel n Priority Register, offset: 0x107 */
    __IO uint8_t DCHPRI11;                           /**< Channel n Priority Register, offset: 0x108 */
    __IO uint8_t DCHPRI10;                           /**< Channel n Priority Register, offset: 0x109 */
    __IO uint8_t DCHPRI9;                            /**< Channel n Priority Register, offset: 0x10A */
    __IO uint8_t DCHPRI8;                            /**< Channel n Priority Register, offset: 0x10B */
    __IO uint8_t DCHPRI15;                           /**< Channel n Priority Register, offset: 0x10C */
    __IO uint8_t DCHPRI14;                           /**< Channel n Priority Register, offset: 0x10D */
    __IO uint8_t DCHPRI13;                           /**< Channel n Priority Register, offset: 0x10E */
    __IO uint8_t DCHPRI12;                           /**< Channel n Priority Register, offset: 0x10F */
    __IO uint8_t DCHPRI19;                           /**< Channel n Priority Register, offset: 0x110 */
    __IO uint8_t DCHPRI18;                           /**< Channel n Priority Register, offset: 0x111 */
    __IO uint8_t DCHPRI17;                           /**< Channel n Priority Register, offset: 0x112 */
    __IO uint8_t DCHPRI16;                           /**< Channel n Priority Register, offset: 0x113 */
    __IO uint8_t DCHPRI23;                           /**< Channel n Priority Register, offset: 0x114 */
    __IO uint8_t DCHPRI22;                           /**< Channel n Priority Register, offset: 0x115 */
    __IO uint8_t DCHPRI21;                           /**< Channel n Priority Register, offset: 0x116 */
    __IO uint8_t DCHPRI20;                           /**< Channel n Priority Register, offset: 0x117 */
    __IO uint8_t DCHPRI27;                           /**< Channel n Priority Register, offset: 0x118 */
    __IO uint8_t DCHPRI26;                           /**< Channel n Priority Register, offset: 0x119 */
    __IO uint8_t DCHPRI25;                           /**< Channel n Priority Register, offset: 0x11A */
    __IO uint8_t DCHPRI24;                           /**< Channel n Priority Register, offset: 0x11B */
    __IO uint8_t DCHPRI31;                           /**< Channel n Priority Register, offset: 0x11C */
    __IO uint8_t DCHPRI30;                           /**< Channel n Priority Register, offset: 0x11D */
    __IO uint8_t DCHPRI29;                           /**< Channel n Priority Register, offset: 0x11E */
    __IO uint8_t DCHPRI28;                           /**< Channel n Priority Register, offset: 0x11F */
            uint8_t RESERVED_7[3808];  //not useful

    DMA_TCD_Type TCD[32];                           // --- offset: 0x1000 bytes
} DMA_RegStructType;


//in this program, actual I2S in is: 3 rx lines: SAI1_Rx0,1,2.
//in this program, actual I2S out is: 2 rx lines: SAI1_Tx0,1

extern uint32_t I2SDmaIntrCnt;
extern volatile int DmaAudioRxBufIsReady;



extern int UsbDnOneFrameAudioBuf_L[AudioFrameSizeInSamplePerCh];
extern int UsbDnOneFrameAudioBuf_R[AudioFrameSizeInSamplePerCh];


extern int32_t *I2SInputPtr[NUM_I2S_INPUT*2];
extern int32_t *I2SOutputPtr[NUM_I2S_OUTPUT*2];

extern volatile unsigned char MainAlgorithmProcessingIsAllowed;

extern void ClearI2SDmaDualTxBuf(void);
extern void ClearI2SDmaDualRxBuf(void);

extern void EnableSai1TxAndRxDmaRequest(uint8_t NeedToEnable);
extern void InitDmaForSaiAudio(void);
extern void InitSai1I2S(void);
extern void StopSai1I2S(void);

extern void AudioRxTxIOProcess(void);
extern void AudioRxTxDmaIntrProcess(void);



#ifdef __cplusplus
}
#endif

#endif


