/*
 * Copyright (c) 2015 - 2016, Freescale Semiconductor, Inc.
 * Copyright 2016 - 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"
#include "usb_device_descriptor.h"
#include "audio_unified.h"


#define USB_STRING_NONE  0x00U
/*******************************************************************************
 * Variables
 ******************************************************************************/
/* hid mouse endpoint information */
usb_device_endpoint_struct_t g_UsbDeviceHidEndpoints[USB_HID_CONSUMER_CONTROL_ENDPOINT_COUNT] = {
    {
        /* HID consumer control interrupt IN pipe */
        USB_HID_CONSUMER_CONTROL_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), USB_ENDPOINT_INTERRUPT,
        FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE,
    },
};
/* HID consumer control interface information */
usb_device_interface_struct_t g_UsbDeviceHidInterface[] = {{
    0U, /* The alternate setting of the interface */
    {
        USB_HID_CONSUMER_CONTROL_ENDPOINT_COUNT, /* Endpoint count */
        g_UsbDeviceHidEndpoints,         /* Endpoints handle */
    },
    NULL,
}};

usb_device_interfaces_struct_t g_UsbDeviceHidInterfaces[USB_HID_CONSUMER_CONTROL_INTERFACE_COUNT] = {{
    USB_HID_CONSUMER_CONTROL_CLASS,           /* HID consumer control class code */
    USB_HID_CONSUMER_CONTROL_SUBCLASS,        /* HID consumer control subclass code */
    USB_HID_CONSUMER_CONTROL_PROTOCOL,        /* HID consumer control protocol code */
    USB_HID_CONSUMER_CONTROL_INTERFACE_INDEX, /* The interface number of the HID consumer control */
    g_UsbDeviceHidInterface,          /* Interfaces handle */
    sizeof(g_UsbDeviceHidInterface) / sizeof(usb_device_interface_struct_t),
}};

usb_device_interface_list_t g_UsbDeviceHidInterfaceList[USB_DEVICE_CONFIGURATION_COUNT] = {
    {
        USB_HID_CONSUMER_CONTROL_INTERFACE_COUNT, /* The interface count of the HID consumer control */
        g_UsbDeviceHidInterfaces,         /* The interfaces handle */
    },
};
usb_device_class_struct_t g_UsbDeviceHidConsumerControlClass = {
    g_UsbDeviceHidInterfaceList,    /* The interface list of the HID consumer control */
    kUSB_DeviceClassTypeHid,        /* The HID class type */
    USB_DEVICE_CONFIGURATION_COUNT, /* The configuration count */
};

/* Audio generator entity struct */
usb_device_audio_entity_struct_t g_UsbDeviceAudioRecorderEntity[] = {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    {
        USB_AUDIO_CONTROL_RECORDER_CLOCK_SOURCE_ENTITY_ID,
        USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_CLOCK_SOURCE_UNIT,
        0U,
    },
#endif
    {
        USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID,
        USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL,
        0U,
    },
    {
        USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID,
        USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT,
        0U,
    },
    {
        USB_AUDIO_RECORDER_CONTROL_OUTPUT_TERMINAL_ID,
        USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL,
        0U,
    },
};

/* Audio speaker entity information */
usb_device_audio_entities_struct_t g_UsbDeviceAudioRecorderEntities = {
    g_UsbDeviceAudioRecorderEntity,
    sizeof(g_UsbDeviceAudioRecorderEntity) / sizeof(usb_device_audio_entity_struct_t),
};


/* Audio generator stream endpoint information */
usb_device_endpoint_struct_t g_UsbDeviceAudioRecorderEndpoints[USB_AUDIO_RECORDER_STREAM_ENDPOINT_COUNT] = {
    /* Audio generator ISO IN pipe */
    {
        USB_AUDIO_RECORDER_STREAM_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        USB_ENDPOINT_ISOCHRONOUS,
        FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS *AUDIO_IN_FORMAT_SIZE,
        FS_ISO_IN_ENDP_INTERVAL,
    },
};

/* Audio speaker control endpoint information */
usb_device_endpoint_struct_t g_UsbDeviceAudioChatControlEndpoints[USB_AUDIO_CONTROL_ENDPOINT_COUNT] = {{
    USB_AUDIO_CHAT_CONTROL_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), USB_ENDPOINT_INTERRUPT,
    FS_AUDIO_INTERRUPT_IN_PACKET_SIZE,
}};

/* Audio speaker control interface information */
usb_device_interface_struct_t g_UsbDeviceAudioRecorderControInterface[] = {{
    USB_AUDIO_CONTROL_INTERFACE_ALTERNATE_0,
    {
        USB_AUDIO_CONTROL_ENDPOINT_COUNT,
        g_UsbDeviceAudioChatControlEndpoints,
    },
    &g_UsbDeviceAudioRecorderEntities,
}};



/* Audio speaker stream interface information */
usb_device_interface_struct_t g_UsbDeviceAudioRecStreamInterface[] = {
    {
        USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_0,
        {
            0U,
            NULL,
        },
        NULL,
    },
    {
        USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_1,
        {
            USB_AUDIO_RECORDER_STREAM_ENDPOINT_COUNT,
            g_UsbDeviceAudioRecorderEndpoints,
        },
        NULL,
    },
};


/* Define interfaces for audio speaker */
usb_device_interfaces_struct_t g_UsbDeviceAudioRecorderInterfaces[2] = {
    {
        USB_AUDIO_CLASS,                         /* Audio class code */
        USB_SUBCLASS_AUDIOCONTROL,               /* Audio control subclass code */
        USB_AUDIO_PROTOCOL,                      /* Audio protocol code */
        USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,       /* The interface number of the Audio control */
        g_UsbDeviceAudioRecorderControInterface, /* The handle of Audio control */
        sizeof(g_UsbDeviceAudioRecorderControInterface) / sizeof(usb_device_interface_struct_t),
    },
    {
        USB_AUDIO_CLASS,                           /* Audio class code */
        USB_SUBCLASS_AUDIOSTREAM,                  /* Audio stream subclass code */
        USB_AUDIO_PROTOCOL,                        /* Audio protocol code */
        USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX, /* The interface number of the Audio control */
        g_UsbDeviceAudioRecStreamInterface,        /* The handle of Audio stream */
        sizeof(g_UsbDeviceAudioRecStreamInterface) / sizeof(usb_device_interface_struct_t),
    }

};


usb_device_interface_list_t g_UsbDeviceAudioInterfaceListRecorder[USB_DEVICE_CONFIGURATION_COUNT] = {
    {
        2,
        g_UsbDeviceAudioRecorderInterfaces,
    },
};



usb_device_class_struct_t g_UsbDeviceAudioClassRecorder = {
    g_UsbDeviceAudioInterfaceListRecorder,
    kUSB_DeviceClassTypeAudio,
    USB_DEVICE_CONFIGURATION_COUNT,
};


usb_device_audio_entity_struct_t g_UsbDeviceAudioChatSpeakerEntity[] = {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    {
        USB_AUDIO_CONTROL_CHAT_SPEAKER_CLOCK_SOURCE_ENTITY_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_CLOCK_SOURCE_UNIT, 0U,
    },
#endif
    {
        USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, 0U,
    },
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    {
        USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, 0U,
    },
#endif
    {
        USB_AUDIO_CHAT_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, 0U,
    },
};

/* Audio speaker entity information */
usb_device_audio_entities_struct_t g_UsbDeviceAudioChatSpeakerEntities = {
    g_UsbDeviceAudioChatSpeakerEntity, sizeof(g_UsbDeviceAudioChatSpeakerEntity) / sizeof(usb_device_audio_entity_struct_t),
};

/* Audio speaker control interface information */
usb_device_interface_struct_t g_UsbDeviceAudioChatSpeakerControInterface[] = {{
    0U,
    {
        USB_AUDIO_CONTROL_ENDPOINT_COUNT, g_UsbDeviceAudioChatControlEndpoints,
    },
    &g_UsbDeviceAudioChatSpeakerEntities,
}};


usb_device_endpoint_struct_t g_UsbDeviceAudioChatSpeakerEndpoints[USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT] = {
    /* Audio generator ISO OUT pipe */
    {
        USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT | (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        USB_ENDPOINT_ISOCHRONOUS, (FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE), /* The max
                                                    packet size should be increased otherwise if host send data larger
                                                    than max packet size will cause DMA error. */
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    {
        USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        USB_ENDPOINT_ISOCHRONOUS, FS_ISO_FEEDBACK_ENDP_PACKET_SIZE,
    }
#endif
};


usb_device_interface_struct_t g_UsbDeviceAudioChatSpeakerStreamInterface[] = {
    {
        0U,
        {
            0U, NULL,
        },
        NULL,
    },
    {
        1U,
        {
            USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT, g_UsbDeviceAudioChatSpeakerEndpoints,
        },
        NULL,
    },
};

usb_device_interfaces_struct_t g_UsbDeviceAudioChatSpeakerInterfaces[2] = {
    {
        USB_AUDIO_CLASS,                                    /* Audio class code */
        USB_SUBCLASS_AUDIOCONTROL,                          /* Audio control subclass code */
        USB_AUDIO_PROTOCOL,                                 /* Audio protocol code */
        USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,             /* The interface number of the Audio control */
        g_UsbDeviceAudioChatSpeakerControInterface,         /* The handle of Audio control */
        sizeof(g_UsbDeviceAudioChatSpeakerControInterface) / sizeof(usb_device_interfaces_struct_t),
    },
    {
        USB_AUDIO_CLASS,                                    /* Audio class code */
        USB_SUBCLASS_AUDIOSTREAM,                           /* Audio stream subclass code */
        USB_AUDIO_PROTOCOL,                                 /* Audio protocol code */
        USB_AUDIO_CHAT_SPEAKER_STREAM_INTERFACE_INDEX,      /* The interface number of the Audio control */
        g_UsbDeviceAudioChatSpeakerStreamInterface,         /* The handle of Audio stream */
        sizeof(g_UsbDeviceAudioChatSpeakerStreamInterface) / sizeof(usb_device_interfaces_struct_t),
    }
};

/* Define configurations for audio speaker */
usb_device_interface_list_t g_UsbDeviceAudioInterfaceListChatSpeaker[USB_DEVICE_CONFIGURATION_COUNT] = {
    {
        2, g_UsbDeviceAudioChatSpeakerInterfaces,
    },
};

/* Define class information for audio speaker */
usb_device_class_struct_t g_UsbDeviceAudioClassChatSpeaker = {
    g_UsbDeviceAudioInterfaceListChatSpeaker, kUSB_DeviceClassTypeAudio, USB_DEVICE_CONFIGURATION_COUNT,
};

usb_device_endpoint_struct_t g_UsbDeviceAudioGameSpeakerEndpoints[USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT] = {
    /* Audio generator ISO OUT pipe */
    {
        USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT | (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        USB_ENDPOINT_ISOCHRONOUS, (FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE), /* The max
                                                    packet size should be increased otherwise if host send data larger
                                                    than max packet size will cause DMA error. */
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    {
        USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        USB_ENDPOINT_ISOCHRONOUS, FS_ISO_FEEDBACK_ENDP_PACKET_SIZE,
    }
#endif
};

usb_device_interface_struct_t g_UsbDeviceAudioGameSpeakerStreamInterface[] = {
    {
        0U,
        {
            0U, NULL,
        },
        NULL,
    },
    {
        1U,
        {
            USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT, g_UsbDeviceAudioGameSpeakerEndpoints,
        },
        NULL,
    },
};

/* Audio speaker control endpoint information */
usb_device_endpoint_struct_t g_UsbDeviceAudioGameControlEndpoints[USB_AUDIO_CONTROL_ENDPOINT_COUNT] = {{
    USB_AUDIO_GAME_CONTROL_ENDPOINT | (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), USB_ENDPOINT_INTERRUPT,
    FS_AUDIO_INTERRUPT_IN_PACKET_SIZE,
}};

usb_device_audio_entity_struct_t g_UsbDeviceAudioGameSpeakerEntity[] = {
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    {
        USB_AUDIO_CONTROL_GAME_SPEAKER_CLOCK_SOURCE_ENTITY_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_CLOCK_SOURCE_UNIT, 0U,
    },
#endif
    {
        USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, 0U,
    },
#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
    {
        USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, 0U,
    },
#endif
    {
        USB_AUDIO_GAME_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID, USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, 0U,
    },
};

/* Audio speaker entity information */
usb_device_audio_entities_struct_t g_UsbDeviceAudioGameSpeakerEntities = {
    g_UsbDeviceAudioGameSpeakerEntity, sizeof(g_UsbDeviceAudioGameSpeakerEntity) / sizeof(usb_device_audio_entity_struct_t),
};
/* Audio speaker control interface information */
usb_device_interface_struct_t g_UsbDeviceAudioGameSpeakerControInterface[] = {{
    0U,
    {
        USB_AUDIO_CONTROL_ENDPOINT_COUNT, g_UsbDeviceAudioGameControlEndpoints,
    },
    &g_UsbDeviceAudioGameSpeakerEntities,
}};

usb_device_interfaces_struct_t g_UsbDeviceAudioGameSpeakerInterfaces[2] = {
    {
        USB_AUDIO_CLASS,                                    /* Audio class code */
        USB_SUBCLASS_AUDIOCONTROL,                          /* Audio control subclass code */
        USB_AUDIO_PROTOCOL,                                 /* Audio protocol code */
        USB_AUDIO_GAME_CONTROL_INTERFACE_INDEX,             /* The interface number of the Audio control */
        g_UsbDeviceAudioGameSpeakerControInterface,         /* The handle of Audio control */
        sizeof(g_UsbDeviceAudioGameSpeakerControInterface) / sizeof(usb_device_interfaces_struct_t),
    },
    {
        USB_AUDIO_CLASS,                                    /* Audio class code */
        USB_SUBCLASS_AUDIOSTREAM,                           /* Audio stream subclass code */
        USB_AUDIO_PROTOCOL,                                 /* Audio protocol code */
        USB_AUDIO_GAME_SPEAKER_STREAM_INTERFACE_INDEX,      /* The interface number of the Audio control */
        g_UsbDeviceAudioGameSpeakerStreamInterface,         /* The handle of Audio stream */
        sizeof(g_UsbDeviceAudioGameSpeakerStreamInterface) / sizeof(usb_device_interfaces_struct_t),
    }
};

/* Define configurations for audio speaker */
usb_device_interface_list_t g_UsbDeviceAudioInterfaceListGameSpeaker[USB_DEVICE_CONFIGURATION_COUNT] = {
    {
        2, g_UsbDeviceAudioGameSpeakerInterfaces,
    },
};

/* Define class information for audio speaker */
usb_device_class_struct_t g_UsbDeviceAudioClassGameSpeaker = {
    g_UsbDeviceAudioInterfaceListGameSpeaker, kUSB_DeviceClassTypeAudio, USB_DEVICE_CONFIGURATION_COUNT,
};

/* App specific structs */
#define USB_HEADSET_AUDIO_CONTROL_INTERFACE_LENGTH \
    (sizeof(usb_descriptor_class_specific_ac_interface_headset_t) + \
     sizeof(usb_class_audio_control_ifo_descriptor_group_t) + \
     sizeof(usb_class_audio_control_io_descriptor_group_t))
#define USB_HEADSET_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH  { USB_SHORT_GET_LOW(USB_HEADSET_AUDIO_CONTROL_INTERFACE_LENGTH), \
                                                            USB_SHORT_GET_HIGH(USB_HEADSET_AUDIO_CONTROL_INTERFACE_LENGTH) }
#define USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_LENGTH \
    (sizeof(usb_descriptor_class_specific_ac_interface_headphones_t) + \
     sizeof(usb_class_audio_control_io_descriptor_group_t))
#define USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH  { USB_SHORT_GET_LOW(USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_LENGTH), \
                                                               USB_SHORT_GET_HIGH(USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_LENGTH) }

typedef struct _usb_class_audio_headset_device_descriptor {
    usb_descriptor_interface_association_t iadAudio;
    usb_descriptor_interface_t control;
    usb_descriptor_class_specific_ac_interface_headset_t controlSub;

    usb_class_audio_control_ifo_descriptor_group_t controlRec;
    usb_class_audio_control_io_descriptor_group_t controlSpkr;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
#else
    usb_descriptor_ac_interrupt_endpoint_t controlInterruptEndpoint; 
#endif

    usb_class_audio_stream_descriptor_group_t streamRec;
    usb_class_audio_stream_descriptor_group_t streamSpkr;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    usb_descriptor_as_iso_sync_endpoint_t feedbackEndpointSpkr;
#endif
} usb_class_audio_headset_device_descriptor_t;

typedef struct _usb_class_audio_headphones_device_descriptor {
    usb_descriptor_interface_association_t iadAudio;
    usb_descriptor_interface_t control;
    usb_descriptor_class_specific_ac_interface_headphones_t controlSub;

    usb_class_audio_control_io_descriptor_group_t controlSpkr;
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
#else
    usb_descriptor_ac_interrupt_endpoint_t controlInterruptEndpoint;
#endif

    usb_class_audio_stream_descriptor_group_t streamSpkr;
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    usb_descriptor_as_iso_sync_endpoint_t feedbackEndpointSpkr;
#endif
} usb_class_audio_headphones_device_descriptor_t;

typedef struct _usb_composite_device_descriptor {
    usb_descriptor_configuration_t deviceConfig;
    usb_class_audio_headset_device_descriptor_t audioChat;
	usb_class_audio_headphones_device_descriptor_t audioGame;
    usb_class_hid_descriptor_t hid;
    //usb_class_cdc_vcom_descriptor_t vcomHci;
#if SERIAL_PORT_TYPE_USBCDC
    usb_class_cdc_vcom_descriptor_t vcomLog;
#endif
} usb_composite_device_descriptor_t;


/* Define string descriptor */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString0[] = {
    2U + 2U,
    USB_DESCRIPTOR_TYPE_STRING,
    0x09U,
    0x04U,
};

USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString1[] = {
    2U + 2U * 18U, USB_DESCRIPTOR_TYPE_STRING,
    'N',           0x00U,
    'X',           0x00U,
    'P',           0x00U,
    ' ',           0x00U,
    'S',           0x00U,
    'E',           0x00U,
    'M',           0x00U,
    'I',           0x00U,
    'C',           0x00U,
    'O',           0x00U,
    'N',           0x00U,
    'D',           0x00U,
    'U',           0x00U,
    'C',           0x00U,
    'T',           0x00U,
    'O',           0x00U,
    'R',           0x00U,
    'S',           0x00U,
};

USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString2[] = {
    2U + 2U * 14U, USB_DESCRIPTOR_TYPE_STRING,
    'N',           0x00U,
    'X',           0x00U,
    'P',           0x00U,
    ' ',           0x00U,
    'D',           0x00U,
    'o',           0x00U,
    'n',           0x00U,
    'g',           0x00U,
    'l',           0x00U,
    'e',           0x00U,
    ' ',           0x00U,
    'm',           0x00U,
    'i',           0x00U,
    'x',           0x00U
};

#define USB_STRING_INDEX_AUDIO_CHAT  0x03U
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString3[] = {
    2U + 2U * 21U, USB_DESCRIPTOR_TYPE_STRING,
    'N',           0x00U,
    'X',           0x00U,
    'P',           0x00U,
    ' ',           0x00U,
    'D',           0x00U,
    'o',           0x00U,
    'n',           0x00U,
    'g',           0x00U,
    'l',           0x00U,
    'e',           0x00U,
    ' ',           0x00U,
    'm',           0x00U,
    'i',           0x00U,
    'x',           0x00U,
    ' ',           0x00U,
    '(',           0x00U,
    'C',           0x00U,
    'h',           0x00U,
    'a',           0x00U,
    't',           0x00U,
    ')',           0x00U
};

#define USB_STRING_INDEX_AUDIO_GAME  0x04U
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString4[] = {
    2U + 2U * 21U, USB_DESCRIPTOR_TYPE_STRING,
    'N',           0x00U,
    'X',           0x00U,
    'P',           0x00U,
    ' ',           0x00U,
    'D',           0x00U,
    'o',           0x00U,
    'n',           0x00U,
    'g',           0x00U,
    'l',           0x00U,
    'e',           0x00U,
    ' ',           0x00U,
    'm',           0x00U,
    'i',           0x00U,
    'x',           0x00U,
    ' ',           0x00U,
    '(',           0x00U,
    'G',           0x00U,
    'a',           0x00U,
    'm',           0x00U,
    'e',           0x00U,
    ')',           0x00U,
};

#define USB_STRING_INDEX_VCOM  0x05U
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceString5[] = {
    2U + 2U * 8U, USB_DESCRIPTOR_TYPE_STRING,
    'U',           0x00U,
    'S',           0x00U,
    'B',           0x00U,
    ' ',           0x00U,
    'V',           0x00U,
    'C',           0x00U,
    'O',           0x00U,
    'M',           0x00U,
};

uint32_t g_UsbDeviceStringDescriptorLength[USB_DEVICE_STRING_COUNT] = {
    sizeof(g_UsbDeviceString0),
    sizeof(g_UsbDeviceString1),
    sizeof(g_UsbDeviceString2),
    sizeof(g_UsbDeviceString3),
    sizeof(g_UsbDeviceString4),
    sizeof(g_UsbDeviceString5),
};

uint8_t *g_UsbDeviceStringDescriptorArray[USB_DEVICE_STRING_COUNT] = {
    g_UsbDeviceString0,
    g_UsbDeviceString1,
    g_UsbDeviceString2,
    g_UsbDeviceString3,
    g_UsbDeviceString4,
    g_UsbDeviceString5,
};

usb_language_t g_UsbDeviceLanguage[USB_DEVICE_LANGUAGE_COUNT] = {{
    g_UsbDeviceStringDescriptorArray,
    g_UsbDeviceStringDescriptorLength,
    (uint16_t)0x0409U,
}};

usb_language_list_t g_UsbDeviceLanguageList = {
    g_UsbDeviceString0,
    sizeof(g_UsbDeviceString0),
    g_UsbDeviceLanguage,
    USB_DEVICE_LANGUAGE_COUNT,
};

/* Define device descriptor */
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)

uint8_t g_UsbDeviceDescriptor[] = {
    USB_DESCRIPTOR_LENGTH_DEVICE, /* Size of this descriptor in bytes */
    USB_DESCRIPTOR_TYPE_DEVICE,   /* DEVICE Descriptor Type */
    USB_SHORT_GET_LOW(USB_DEVICE_SPECIFIC_BCD_VERSION),
    USB_SHORT_GET_HIGH(USB_DEVICE_SPECIFIC_BCD_VERSION), /* USB Specification Release Number in
                                                            Binary-Coded Decimal (i.e., 2.10 is 210H). */
    USB_DEVICE_CLASS,                                    /* Class code (assigned by the USB-IF). */
    USB_DEVICE_SUBCLASS,                                 /* Subclass code (assigned by the USB-IF). */
    USB_DEVICE_PROTOCOL,                                 /* Protocol code (assigned by the USB-IF). */
    USB_CONTROL_MAX_PACKET_SIZE,                         /* Maximum packet size for endpoint zero
                                                            (only 8, 16, 32, or 64 are valid) */
    USB_SHORT_GET_LOW(USB_DEVICE_VID),
    USB_SHORT_GET_HIGH(USB_DEVICE_VID), /* Vendor ID (assigned by the USB-IF) */
    USB_SHORT_GET_LOW(USB_DEVICE_PID),
    USB_SHORT_GET_HIGH(USB_DEVICE_PID), /* Product ID (assigned by the manufacturer) */
    USB_SHORT_GET_LOW(USB_DEVICE_DEMO_BCD_VERSION),
    USB_SHORT_GET_HIGH(USB_DEVICE_DEMO_BCD_VERSION), /* Device release number in binary-coded decimal */
    0x01U,                                           /* Index of string descriptor describing manufacturer */
    0x02U,                                           /* Index of string descriptor describing product */
    0x00U,                                           /* Index of string descriptor describing the
                                                        device's serial number */
    USB_DEVICE_CONFIGURATION_COUNT,                  /* Number of possible configurations */
};

USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceHidConsumerControlReportDescriptor[USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT] = {
    0x05, 0x0c,                    /* USAGE_PAGE (Consumer Devices) */
    0x09, 0x01,                    /* USAGE (Consumer Control) */
    0xa1, 0x01,                    /* COLLECTION (Application) */
    0x15, 0x00,                    /*   LOGICAL_MINIMUM (0) */
    0x25, 0x01,                    /*   LOGICAL_MAXIMUM (1) */
    0x05, 0x0c,                    /*   USAGE_PAGE (Consumer Devices) */
    0x09, 0xcd,                    /*   USAGE (Play/Pause) */
    0x09, 0xe9,                    /*   USAGE (Volume Up) */
    0x09, 0xea,                    /*   USAGE (Volume Down) */
    0x09, 0xb5,                    /*   USAGE (Scan Next Track) */
    0x09, 0xb6,                    /*   USAGE (Scan Previous Track) */
    0x09, 0xb7,                    /*   USAGE (Stop) */
    0x95, 0x06,                    /*   REPORT_COUNT (6) */
    0x75, 0x01,                    /*   REPORT_SIZE (1) */
    0x81, 0x02,                    /*   INPUT (Data,Var,Abs) */
    0x75, 0x02,                    /*   REPORT_SIZE (2) */
    0x95, 0x01,                    /*   REPORT_COUNT (1) */
    0x81, 0x03,                    /*   INPUT (Cnst,Var,Abs) */
    0xc0                           /* END_COLLECTION */
};

USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
usb_composite_device_descriptor_t g_UsbDeviceConfigurationDescriptor = {
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
      .deviceConfig = {
        .bLength = USB_DESCRIPTOR_LENGTH_CONFIGURE, /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_CONFIGURE,   /* CONFIGURATION Descriptor Type */
        .wTotalLength = {USB_SHORT_GET_LOW(sizeof(g_UsbDeviceConfigurationDescriptor)),
                         USB_SHORT_GET_HIGH(sizeof(g_UsbDeviceConfigurationDescriptor))}, /* Total length of data returned for this configuration. */
        .bNumInterfaces = USB_DEVICE_INTERFACE_COUNT,          /* Number of interfaces supported by this configuration */
        .bConfigurationValue = USB_COMPOSITE_CONFIGURE_INDEX,  /* Value to use as an argument to the
                                                                  SetConfiguration() request to select this configuration */
        .iConfiguration = USB_STRING_NONE,                     /* Index of string descriptor describing this configuration */
        .bmAttributes = (USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_D7_MASK) |
            (USB_DEVICE_CONFIG_SELF_POWER << USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_SELF_POWERED_SHIFT) |
            (USB_DEVICE_CONFIG_REMOTE_WAKEUP << USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_REMOTE_WAKEUP_SHIFT),
                                            /* Configuration characteristics
                                               D7: Reserved (set to one)
                                               D6: Self-powered
                                               D5: Remote Wakeup
                                               D4...0: Reserved (reset to zero)
                                            */
        .bMaxPower = 0xFA,                  /* Maximum power consumption of the USB
                                            * device from the bus in this specific
                                            * configuration when the device is fully
                                            * operational. Expressed in 2 mA units
                                            *  (i.e., 50 = 100 mA).
                                            */
    },
/* -------------------------------------------------------Audio chat------------------------------------------------------- */
    .audioChat.iadAudio = {
        .bLength = USB_IAD_DESC_SIZE,                                 /* Size of this descriptor in bytes*/
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* Constant name specifying INTERFACE ASSOCIATION Descriptor*/
        .bFirstInterface = USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,    /* Interface number of the first interface that is associated with this function.*/
        .bInterfaceCount = USB_AUDIO_CHAT_COMPOSITE_INTERFACE_COUNT,  /* Number of contiguous interfaces that are associated with this function.*/
        .bFunctionClass = USB_AUDIO_CLASS,                            /* The function belongs to the Audio Control Device Class  */
        .bFunctionSubClass = 0x00,               /* Subclass code (assigned by USB-IF).*/
        .bFunctionProtocol = 0x20,                                    /* Protocol code (assigned by USB-IF).*/
        .iFunction = USB_STRING_INDEX_AUDIO_CHAT,                          /* The Function string descriptor index */
    },
    .audioChat.control = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,   /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting for the interface identified in the prior field */
        .bNumEndpoints = 0x00U,                                       /* Number of endpoints used by this interface (excluding endpoint zero). */
        .bInterfaceClass = USB_AUDIO_CLASS,                           /*The interface implements the Audio Interface class  */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOCONTROL,              /*The interface implements the AUDIOCONTROL Subclass  */
        .bInterfaceProtocol = USB_AUDIO_PROTOCOL,                     /*The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.controlSub = {
        .bLength = sizeof(usb_descriptor_class_specific_ac_interface_headset_t),   /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,         /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_HEADER, /* HEADER descriptor subtype   */
        .bcdADC = {0x00U, 0x02U},                                          /* Audio Device compliant to the USB Audio specification version 1.00  */
        .bCategory = 0x04,                                                 /* Undefied(0x00) : Indicating the primary use of this audio function   */
        .wTotalLength = USB_HEADSET_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH, /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                                                       Includes the combined length of this descriptor header and all Unit and Terminal descriptors. */
        .bmControls  = 0x00,
    },
    
    .audioChat.controlRec.clockSource = {
      .bLength = 0x08,
      .bDescriptorType = 0x24,
      .bDescriptorSubType = 0x0A,
      .bClockID = 0x10,
      .bmAttrButes = 0x01,
      .bmControls = 0x07,
      .bAssocTerminal = 0x00,
      .iClockSource = 0x00,    
    },
    .audioChat.controlRec.inputTerminal = {
        /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID,               /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = HEADSET_TERMINAL_TYPE,                                    /* Headset terminaltype.  */
        /*
         * bi-directional terminal types (such as headset)
         * require two terminals (input + output).
         * They need to be tied together with the bAssocTerminal field
         * We link this recording input channel to the headset output channel here
         */
        .bAssocTerminal = USB_AUDIO_CHAT_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,
        .bCSourceID = 0x10,  
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS, /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .bmChannelConfig = {0x03U, 0x00U, 0x00, 0x00},     /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,     /* Index of a string descriptor, describing the name of the first logical channel.*/
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,         /* Index of a string descriptor, describing the Input Terminal.*/
    },
    .audioChat.controlRec.featureUnit = {
       /* Audio Class Specific type of Feature Unit */
       .bLength = USB_AUDIO_FEATURE_UNIT_ONLY_DESC_SIZE(2, 4),                  /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
       .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,               /* CS_INTERFACE descriptor type.*/
       .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, /* FEATURE_UNIT descriptor subtype.*/
       .bUnitId = USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID,                   /* Constant uniquely identifying the Unit within the audio function.
                                                                                   This value is used in all requests to address this Unit.*/
       .bSourceID = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID,               /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
       .bmaControls0 = {0x0f, 0x00, 0x00, 0x00},                                     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                                                                   D0: Mute, D1: Volume, ...*/
       .bmaControls1 = {0x00, 0x00, 0x00, 0x00},
       .bmaControls2 = {0x00, 0x00, 0x00, 0x00},
       .iFeature = USB_STRING_NONE,                                             /* Index of a string descriptor, describing this Feature Unit.*/
    },
    .audioChat.controlRec.outputTerminal = {
        /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_RECORDER_CONTROL_OUTPUT_TERMINAL_ID,               /* Constant uniquely identifying the Terminal within
                                                                                       the audio function*/
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface */
        .bAssocTerminal = 0x00U,                      /* This Output Terminal has no association   */
        .bSourceID = USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID, /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .bCSourceID = 0X10,
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,                            /* Index of a string descriptor, describing the Output Terminal.  */
    },
    
    .audioChat.controlSpkr.clockSource = {
      .bLength = 0x08,
      .bDescriptorType = 0x24,
      .bDescriptorSubType = 0x0A,
      .bClockID = 0x20,
      .bmAttrButes = 0x01,
      .bmControls = 0x07,
      .bAssocTerminal = 0x00,
      .iClockSource = 0x00,    
    },
    
    .audioChat.controlSpkr.inputTerminal = {
        /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface .  */
        .bAssocTerminal = 0x00U,                      /* This Input Terminal has no association  */
        .bCSourceID = 0X20,
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,         /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .bmChannelConfig = {0x03U, 0x00U, 0x00, 0x00},             /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,             /* Index of a string descriptor, describing the name of the first logical channel.*/
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,                 /* Index of a string descriptor, describing the Input Terminal.*/
                                           
    },
    
    .audioChat.controlSpkr.featureUnit = {
       /* Audio Class Specific type of Feature Unit */
       .bLength = USB_AUDIO_FEATURE_UNIT_ONLY_DESC_SIZE(2, 4),                  /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
       .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,               /* CS_INTERFACE descriptor type.*/
       .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, /* FEATURE_UNIT descriptor subtype.*/
       .bUnitId = USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID,                   /* Constant uniquely identifying the Unit within the audio function.
                                                                                   This value is used in all requests to address this Unit.*/
       .bSourceID = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,               /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
       .bmaControls0 = {0x0f, 0x00, 0x00, 0x00},                                     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                                                                   D0: Mute, D1: Volume, ...*/
       .bmaControls1 = {0x00, 0x00, 0x00, 0x00},
       .bmaControls2 = {0x00, 0x00, 0x00, 0x00},
       .iFeature = USB_STRING_NONE,                                             /* Index of a string descriptor, describing this Feature Unit.*/
    },
    
    .audioChat.controlSpkr.outputTerminal = {
        /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_CHAT_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within
                                                                                     the audio function*/
        .wTerminalType = HEADSET_TERMINAL_TYPE,                                     /* Headset/Headphones terminaltype. */
        .bAssocTerminal = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID, /* See info in recorder audio class type for more info
                                                                           on the bAssocTerminal and why this is needed */
        .bSourceID = USB_AUDIO_CHAT_SPEAKER_CONTROL_FEATURE_UNIT_ID,    /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .bCSourceID = 0x20,
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,                                   /* Index of a string descriptor, describing the Output Terminal.  */
    },
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
#else
    .audioChat.controlInterruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_AC_INTERRUPT_ENDPOINT,                         /* Size of this descriptor, in bytes: 9U */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                /* ENDPOINT descriptor type */
        .bEndpointAddress = USB_AUDIO_CHAT_CONTROL_ENDPOINT |
                           (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* Endpoint address */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                         /* Transfer type */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE),
                           USB_SHORT_GET_HIGH(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE)},      /* Max Packet Size */
        .bInterval = FS_AUDIO_INTERRUPT_IN_INTERVAL,                                    /* Interval */
        .bRefresh = 0x00U,                                                              /* Unused */
        .bSynchAddress = 0x00U,                                                         /* no sychronization endpoint */
    },
#endif
    .audioChat.streamRec.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                    /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,              /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX, /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                    /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                        /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                            /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                   /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                 /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamRec.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                    /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,              /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX, /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                    /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = 0x01U,                                        /* The number of endpoints used by this interface is 1 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                            /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                   /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                 /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamRec.asInterface = {
        /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = 0x10,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_RECORDER_CONTROL_OUTPUT_TERMINAL_ID,       /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bmControls = 0x00,
        .bFormatType = 0x01,
        .bmFormats = {0x01, 0x00, 0x000, 0x00},                               /* Bit0: PCM = 1 */
        .bNrChannels = 0x01,
        .bmChannelConfig = {0x03, 0x00, 0x00, 0x00},
        .iChannelNames = 0x00,       
    },
    .audioChat.streamRec.audioFormat = {
        /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = 0x06,                          /* bLength (11) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bSubslotSize = 0x02,
        .bBitResolution = 0x10,
    },

#if USB_DEVICE_AUDIO_USE_SYNC_MODE

    .audioChat.streamRec.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_RECORDER_STREAM_ENDPOINT |
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),  /* IN endpoint address */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE)},
        .bInterval = FS_ISO_IN_ENDP_INTERVAL, /* The polling interval value is every 1 Frames. If Hi-Speed, every 1 uFrames*/
    },

#else
    .audioChat.streamRec.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_RECORDER_STREAM_ENDPOINT |
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),  /* IN endpoint address */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | (0x2 << 4U),                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_FORMAT_IN_CHANNELS * AUDIO_RX_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_FORMAT_IN_CHANNELS * AUDIO_RX_FORMAT_SIZE)},
        .bInterval = FS_ISO_IN_ENDP_INTERVAL, /* The polling interval value is every 1 Frames. If Hi-Speed, every 1 uFrames*/
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = 0x00U,                                                           /* no sychronization endpoint */
    },
#endif

    .audioChat.streamRec.specificIsoEndpoint = {
        /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_CLASS_SPECIFIC_ENDPOINT_LENGTH,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00,
        .bmControls = 0x00,
        .bLockDelayUnits = 0x00,
        .wLockDelay = {0x00, 0x00},
    },
    .audioChat.streamSpkr.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_CHAT_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                         /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                             /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                        /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamSpkr.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_CHAT_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                         /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT,           /* The number of endpoints used by this interface is 2 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                        /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamSpkr.asInterface = {        
         /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = 0x10,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,       /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bmControls = 0x00,
        .bFormatType = 0x01,
        .bmFormats = {0x01, 0x00, 0x000, 0x00},                               /* Bit0: PCM = 1 */
        .bNrChannels = 0x02,
        .bmChannelConfig = {0x03, 0x00, 0x00, 0x00},
        .iChannelNames = 0x00,    
    },
    .audioChat.streamSpkr.audioFormat = {
         /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = 0x06,                          /* bLength (6) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bSubslotSize = AUDIO_OUT_FORMAT_SIZE,
        .bBitResolution = AUDIO_OUT_FORMAT_BITS,
    },

#if USB_DEVICE_AUDIO_USE_SYNC_MODE
    .audioChat.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                                  /* Descriptor size is 7 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
    },
#else
    .audioChat.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT |
                         (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),     /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#endif

    .audioChat.streamSpkr.specificIsoEndpoint = {       
         /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_CLASS_SPECIFIC_ENDPOINT_LENGTH,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00,
        .bmControls = 0x00,
        .bLockDelayUnits = 0x00,
        .wLockDelay = {0x00, 0x00},
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    .audioChat.feedbackEndpointSpkr = {
        /* Endpoint 3 Feedback ENDPOINT */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                 /* bLength */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                 /* bDescriptorType */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT |   /* This is an IN endpoint */
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | 0x10,          /*  Types - Transfer: ISOCHRONOUS Sync: Async Usage: Feedback EP */
        .wMaxPacketSize = {FS_ISO_FEEDBACK_ENDP_PACKET_SIZE, 0x00},      /* wMaxPacketSize */
        .bInterval = 0x01,                                               /* interval polling(2^x ms) */
        .bRefresh = 0x05,                                                /* bRefresh(32ms)  */
        .bSynchAddress = 0x00,                                           /* unused */
    },
#endif
/* -------------------------------------------------------Audio game------------------------------------------------------- */
    .audioGame.iadAudio = {
        .bLength = USB_IAD_DESC_SIZE,                                 /* Size of this descriptor in bytes*/
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* Constant name specifying INTERFACE ASSOCIATION Descriptor*/
        .bFirstInterface = USB_AUDIO_GAME_CONTROL_INTERFACE_INDEX,    /* Interface number of the first interface that is associated with this function.*/
        .bInterfaceCount = USB_AUDIO_GAME_COMPOSITE_INTERFACE_COUNT,  /* Number of contiguous interfaces that are associated with this function.*/
        .bFunctionClass = USB_AUDIO_CLASS,                            /* The function belongs to the Audio Control Device Class  */
        .bFunctionSubClass = USB_SUBCLASS_AUDIOCONTROL,               /* Subclass code (assigned by USB-IF).*/
        .bFunctionProtocol = 0x20,                                    /* Protocol code (assigned by USB-IF).*/
        .iFunction = USB_STRING_INDEX_AUDIO_GAME,                     /* The Function string descriptor index */
    },
    .audioGame.control = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_AUDIO_GAME_CONTROL_INTERFACE_INDEX,   /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting for the interface identified in the prior field */
        .bNumEndpoints = 0x00U,                                       /* Number of endpoints used by this interface (excluding endpoint zero). */
        .bInterfaceClass = USB_AUDIO_CLASS,                           /*The interface implements the Audio Interface class  */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOCONTROL,              /*The interface implements the AUDIOCONTROL Subclass  */
        .bInterfaceProtocol = 0x20U,                                  /*The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.controlSub = {          
        .bLength = sizeof(usb_descriptor_class_specific_ac_interface_headphones_t),   /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,         /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_HEADER, /* HEADER descriptor subtype   */
        .bcdADC = {0x00U, 0x02U},                                          /* Audio Device compliant to the USB Audio specification version 1.00  */
        .bCategory = 0x04,                                                 /* Undefied(0x00) : Indicating the primary use of this audio function   */
        .wTotalLength = USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH, /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                                                       Includes the combined length of this descriptor header and all Unit and Terminal descriptors. */
        .bmControls  = 0x00,
    },
    
    .audioGame.controlSpkr.clockSource = {
      .bLength = 0x08,
      .bDescriptorType = 0x24,
      .bDescriptorSubType = 0x0A,
      .bClockID = 0x30,
      .bmAttrButes = 0x01,
      .bmControls = 0x07,
      .bAssocTerminal = 0x00,
      .iClockSource = 0x00,    
    },
    
    .audioGame.controlSpkr.inputTerminal = {       
         /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface .  */
        .bAssocTerminal = 0x00U,                      /* This Input Terminal has no association  */
        .bCSourceID = 0X30,
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,         /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .bmChannelConfig = {0x03U, 0x00U, 0x00, 0x00},             /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,             /* Index of a string descriptor, describing the name of the first logical channel.*/
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,                 /* Index of a string descriptor, describing the Input Terminal.*/
    },
    
    .audioGame.controlSpkr.featureUnit = {
       /* Audio Class Specific type of Feature Unit */
       .bLength = USB_AUDIO_FEATURE_UNIT_ONLY_DESC_SIZE(2, 4),                  /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
       .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,               /* CS_INTERFACE descriptor type.*/
       .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, /* FEATURE_UNIT descriptor subtype.*/
       .bUnitId = USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID,                   /* Constant uniquely identifying the Unit within the audio function.
                                                                                   This value is used in all requests to address this Unit.*/
       .bSourceID = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,               /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
       .bmaControls0 = {0x0f, 0x00, 0x00, 0x00},                                     /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                                                                   D0: Mute, D1: Volume, ...*/
       .bmaControls1 = {0x00, 0x00, 0x00, 0x00},
       .bmaControls2 = {0x00, 0x00, 0x00, 0x00},
       .iFeature = USB_STRING_NONE,                                             /* Index of a string descriptor, describing this Feature Unit.*/
    },
    .audioGame.controlSpkr.outputTerminal = {        
         /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_GAME_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within
                                                                                     the audio function*/
        .wTerminalType = HEADPHONES_TERMINAL_TYPE,                                     /* Headset/Headphones terminaltype. */
        .bAssocTerminal = 0x00U,                                           /* See info in recorder audio class type for more info
                                                                           on the bAssocTerminal and why this is needed */
        .bSourceID = USB_AUDIO_GAME_SPEAKER_CONTROL_FEATURE_UNIT_ID,    /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .bCSourceID = 0x30,
        .bmControls = {0x00, 0x00},
        .iTerminal = USB_STRING_NONE,                                   /* Index of a string descriptor, describing the Output Terminal.  */
    },
#if USB_DEVICE_CONFIG_AUDIO_CLASS_2_0
#else
    .audioGame.controlInterruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_AC_INTERRUPT_ENDPOINT,                         /* Size of this descriptor, in bytes: 9U */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                /* ENDPOINT descriptor type */
        .bEndpointAddress = USB_AUDIO_GAME_CONTROL_ENDPOINT |
                           (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* Endpoint address */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                         /* Transfer type */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE),
                           USB_SHORT_GET_HIGH(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE)},      /* Max Packet Size */
        .bInterval = FS_AUDIO_INTERRUPT_IN_INTERVAL,                                    /* Interval */
        .bRefresh = 0x00U,                                                              /* Unused */
        .bSynchAddress = 0x00U,                                                         /* no sychronization endpoint */
    },
#endif
    .audioGame.streamSpkr.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_GAME_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                         /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                             /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                        /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.streamSpkr.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_GAME_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                         /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT,           /* The number of endpoints used by this interface is 2 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x20U,                                        /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.streamSpkr.asInterface = {      
         /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = 0x10,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,       /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bmControls = 0x00,
        .bFormatType = 0x01,
        .bmFormats = {0x01, 0x00, 0x000, 0x00},                               /* Bit0: PCM = 1 */
        .bNrChannels = 0x02,
        .bmChannelConfig = {0x03, 0x00, 0x00, 0x00},
        .iChannelNames = 0x00,    
    },
    .audioGame.streamSpkr.audioFormat = {        
         /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = 0x06,                          /* bLength (6) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bSubslotSize = AUDIO_OUT_FORMAT_SIZE,
        .bBitResolution = AUDIO_OUT_FORMAT_BITS,
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
    .audioGame.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
					        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
    },
#else
    .audioGame.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT |
                         (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),     /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#endif
    .audioGame.streamSpkr.specificIsoEndpoint = {        
         /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_CLASS_SPECIFIC_ENDPOINT_LENGTH,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00,
        .bmControls = 0x00,
        .bLockDelayUnits = 0x00,
        .wLockDelay = {0x00, 0x00},
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    .audioGame.feedbackEndpointSpkr = {
        /* Endpoint 3 Feedback ENDPOINT */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,            /* bLength */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,            /* bDescriptorType */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT |   /* This is an IN endpoint */
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | 0x10,          /*  Types - Transfer: ISOCHRONOUS Sync: Async Usage: Feedback EP */
        .wMaxPacketSize = {FS_ISO_FEEDBACK_ENDP_PACKET_SIZE, 0x00},      /* wMaxPacketSize */
        .bInterval = 0x01,                                               /* interval polling(2^x ms) */
        .bRefresh = 0x05,                                                /* bRefresh(32ms)  */
        .bSynchAddress = 0x00,                                           /* unused */
    },
#endif
/* ---------------------------------------------------------HID--------------------------------------------------------- */
    .hid.consumerControlInterface = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_HID_CONSUMER_CONTROL_INTERFACE_INDEX, /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting
                                                                         for the interface identified in the prior field */
        .bNumEndpoints = USB_HID_CONSUMER_CONTROL_ENDPOINT_COUNT,     /* Number of endpoints used by this
                                                                         interface (excluding endpoint zero). */
        .bInterfaceClass = USB_HID_CONSUMER_CONTROL_CLASS,            /* Class code (assigned by the USB-IF). */
        .bInterfaceSubClass = USB_HID_CONSUMER_CONTROL_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
        .bInterfaceProtocol = USB_HID_CONSUMER_CONTROL_PROTOCOL,      /* Protocol code (assigned by the USB). */
        .iInterface = USB_STRING_NONE,                                /* Index of string descriptor describing this interface */
    },
    .hid.hidReport= {
        .bLength = USB_DESCRIPTOR_LENGTH_HID,                                        /* Numeric expression that is the total size of the HID descriptor. */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_HID,                                  /* Constant name specifying type of HID descriptor. */
        .bcdHID = {0x00U, 0x01U},                                                    /* Numeric expression identifying the HID Class Specification release. */
        .bCountryCode = USB_HID_COUNTRY_CODE,                                        /* Numeric expression identifying country code of the localized hardware */
        .bNumDescriptors = USB_HID_CLASS_DESCRIPTOR_COUNT,                           /* Numeric expression specifying the number of class descriptors(at least one report descriptor) */
        .bHidDescriptorType = USB_DESCRIPTOR_TYPE_HID_REPORT,                        /* Constant name identifying type of class descriptor. */
        .wDescriptorLength =
            {USB_SHORT_GET_LOW(USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT),
             USB_SHORT_GET_HIGH(USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT)}, /* Numeric expression that is the total size of the Report descriptor. */
    },
    .hid.interruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                                /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                          /* ENDPOINT Descriptor Type */
        .bEndpointAddress = USB_HID_CONSUMER_CONTROL_ENDPOINT | (USB_IN << 7),                    /* The address of the endpoint on the USB device described by this descriptor. */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                                   /* This field describes the endpoint's attributes */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
                           USB_SHORT_GET_HIGH(FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
        .bInterval = FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL,                               /* Interval for polling endpoint for data transfers. */
    },
		
//		/* -------------------------------------------------------VCOM HCI-------------------------------------------------------- */
//    .vcomHci.iad = {
//        /* Interface Association Descriptor */
//        .bLength =           USB_IAD_DESC_SIZE,                         /* Size of this descriptor in bytes */
//        .bDescriptorType =   USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* INTERFACE_ASSOCIATION Descriptor Type  */
//        .bFirstInterface =   USB_CDC_VCOM_CIC_INTERFACE_INDEX,          /* The first interface number associated with this function
//                                                                           in other words: the interface number of the VCOM class */
//        .bInterfaceCount =   USB_CDC_VCOM_INTERFACE_COUNT,              /* The number of contiguous interfaces associated with this function */
//        /* The function belongs to the Communication Device/Interface Class  */
//        .bFunctionClass =    USB_CDC_VCOM_CIC_CLASS,
//        .bFunctionSubClass = USB_CDC_VCOM_CIC_SUBCLASS,
//        .bFunctionProtocol = 0x00,                      /* The function uses the No class specific protocol required Protocol  */
//        .iFunction =         USB_STRING_INDEX_VCOM,     /* The Function string descriptor index */
//
//    },
//    .vcomHci.cic = {
//        /* Interface Descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,           /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,     /* INTERFACE Descriptor Type */
//        .bInterfaceNumber = USB_CDC_VCOM_CIC_INTERFACE_INDEX, /* Number of this interface. Zero-based value identifying the index in the array of
//                                                                 concurrent interfaces supported by this configuration. */
//        .bAlternateSetting =  0x00,                           /* Value used to select this alternate setting
//                                                                 for the interface identified in the prior field */
//        .bNumEndpoints = USB_CDC_VCOM_CIC_ENDPOINT_COUNT,     /* Number of endpoints used by this interface (excluding endpoint zero). */
//        .bInterfaceClass = USB_CDC_VCOM_CIC_CLASS,            /* Class code (assigned by the USB-IF). */
//        .bInterfaceSubClass = USB_CDC_VCOM_CIC_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
//        .bInterfaceProtocol = USB_CDC_VCOM_CIC_PROTOCOL,      /* Protocol code (assigned by the USB). */
//        .iInterface = USB_STRING_NONE,                        /* Index of string descriptor describing this interface */
//    },
//    .vcomHci.functionalHeader = {
//        /* CDC Class-Specific descriptor */
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_HEADER_FUNC, /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE,  /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_HEADER_FUNC_DESC,
//        .bcdUSB = {0x10, 0x01}, /* USB Class Definitions for Communications the Communication specification version 1.10 */
//    },
//    .vcomHci.callMgmt = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_CALL_MANAG, /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE, /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_CALL_MANAGEMENT_FUNC_DESC,
//        .bmCapabilities = 0x01, /*Bit 0: Whether device handle call management itself 1,
//                                  Bit 1: Whether device can send/receive call management information over a Data Class Interface 0 */
//        .bDataInterface = 0x01, /* Indicates multiplexed commands are handled via data interface */
//    },
//    .vcomHci.abstractControl = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_ABSTRACT,   /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE, /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_ABSTRACT_CONTROL_FUNC_DESC,
//        .bmCapabilities = 0x06, /* Bit 0: Whether device supports the request combination of Set_Comm_Feature, Clear_Comm_Feature, and
//                 Get_Comm_Feature 0, Bit 1: Whether device supports the request combination of Set_Line_Coding,
//                 Set_Control_Line_State, Get_Line_Coding, and the notification Serial_State 1, Bit ...  */
//    },
//    .vcomHci.unionInterface = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_UNION_FUNC,   /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE,   /* CS_INTERFACE Descriptor Type */
//        .bDescriptorSubtype = USB_CDC_UNION_FUNC_DESC,
//        .bControlInterface = USB_CDC_VCOM_CIC_INTERFACE_INDEX,     /* The interface number of the Communications or Data Class interface  */
//        .bSubordinateInterface = USB_CDC_VCOM_DIC_INTERFACE_INDEX, /* Interface number of subordinate interface in the Union  */
//    },
////    .vcomHci.interruptEndpoint = {
////        /*Notification Endpoint descriptor */
////        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                    /* Size of this descriptor in bytes */
////        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                              /* ENDPOINT Descriptor Type */
////        .bEndpointAddress = USB_CDC_VCOM_CIC_INTERRUPT_IN_ENDPOINT | (USB_IN << 7U),  /* The address of the endpoint on the USB device described by this descriptor. */
////        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                       /* This field describes the endpoint's attributes */
////        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_INTERRUPT_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
////                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_INTERRUPT_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
////        .bInterval = FS_CDC_VCOM_INTERRUPT_IN_INTERVAL,                               /* Interval for polling endpoint for data transfers. */
////    },
//    .vcomHci.dic = {
//        /* Data Interface Descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,           /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,     /* INTERFACE Descriptor Type */
//        .bInterfaceNumber = USB_CDC_VCOM_DIC_INTERFACE_INDEX, /* Number of this interface. Zero-based value identifying the index in the array of
//                                                                 concurrent interfaces supported by this configuration. */
//        .bAlternateSetting = 0x00,                            /* Value used to select this alternate setting
//                                                                 for the interface identified in the prior field */
//        .bNumEndpoints = USB_CDC_VCOM_DIC_ENDPOINT_COUNT,     /* Number of endpoints used by this interface (excluding endpoint zero). */
//        .bInterfaceClass = USB_CDC_VCOM_DIC_CLASS,            /* Class code (assigned by the USB-IF). */
//        .bInterfaceSubClass = USB_CDC_VCOM_DIC_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
//        .bInterfaceProtocol = USB_CDC_VCOM_DIC_PROTOCOL,      /* Protocol code (assigned by the USB). */
//        .iInterface = USB_STRING_NONE,                        /* Index of string descriptor describing this interface */
//    },
//    .vcomHci.bulkIn = {
//        /*Bulk IN Endpoint descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                               /* Size of this descriptor in bytes */
//        .bDescriptorType =  USB_DESCRIPTOR_TYPE_ENDPOINT,                        /* ENDPOINT Descriptor Type */
//        .bEndpointAddress = USB_CDC_VCOM_DIC_BULK_IN_ENDPOINT | (USB_IN << 7U),  /* The address of the endpoint on the USB device described by this descriptor. */
//        .bmAttributes = USB_ENDPOINT_BULK,                                       /* This field describes the endpoint's attributes */
//        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_BULK_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
//                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_BULK_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
//        .bInterval = 0x00,                                                       /* The polling interval value is every 0 Frames */
//    },
//    .vcomHci.bulkOut = {
//         /*Bulk OUT Endpoint descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                          /* ENDPOINT Descriptor Type */
//        .bEndpointAddress = USB_CDC_VCOM_DIC_BULK_OUT_ENDPOINT | (USB_OUT << 7U), /* The address of the endpoint on the USB device described by this descriptor. */
//        .bmAttributes = USB_ENDPOINT_BULK,                                        /* This field describes the endpoint's attributes */
//        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_BULK_OUT_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
//                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_BULK_OUT_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
//        .bInterval = 0x00,                                                        /* The polling interval value is every 0 Frames */
//    },
#else
    .deviceConfig = {
        .bLength = USB_DESCRIPTOR_LENGTH_CONFIGURE, /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_CONFIGURE,   /* CONFIGURATION Descriptor Type */
        .wTotalLength = {USB_SHORT_GET_LOW(sizeof(g_UsbDeviceConfigurationDescriptor)),
                         USB_SHORT_GET_HIGH(sizeof(g_UsbDeviceConfigurationDescriptor))}, /* Total length of data returned for this configuration. */
        .bNumInterfaces = USB_DEVICE_INTERFACE_COUNT,          /* Number of interfaces supported by this configuration */
        .bConfigurationValue = USB_COMPOSITE_CONFIGURE_INDEX,  /* Value to use as an argument to the
                                                                  SetConfiguration() request to select this configuration */
        .iConfiguration = USB_STRING_NONE,                     /* Index of string descriptor describing this configuration */
        .bmAttributes = (USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_D7_MASK) |
            (USB_DEVICE_CONFIG_SELF_POWER << USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_SELF_POWERED_SHIFT) |
            (USB_DEVICE_CONFIG_REMOTE_WAKEUP << USB_DESCRIPTOR_CONFIGURE_ATTRIBUTE_REMOTE_WAKEUP_SHIFT),
                                            /* Configuration characteristics
                                               D7: Reserved (set to one)
                                               D6: Self-powered
                                               D5: Remote Wakeup
                                               D4...0: Reserved (reset to zero)
                                            */
        .bMaxPower = USB_DEVICE_MAX_POWER, /* Maximum power consumption of the USB
                                            * device from the bus in this specific
                                            * configuration when the device is fully
                                            * operational. Expressed in 2 mA units
                                            *  (i.e., 50 = 100 mA).
                                            */
    },
/* -------------------------------------------------------Audio chat------------------------------------------------------- */
    .audioChat.iadAudio = {
        .bLength = USB_IAD_DESC_SIZE,                                 /* Size of this descriptor in bytes*/
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* Constant name specifying INTERFACE ASSOCIATION Descriptor*/
        .bFirstInterface = USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,    /* Interface number of the first interface that is associated with this function.*/
        .bInterfaceCount = USB_AUDIO_CHAT_COMPOSITE_INTERFACE_COUNT,  /* Number of contiguous interfaces that are associated with this function.*/
        .bFunctionClass = USB_AUDIO_CLASS,                            /* The function belongs to the Audio Control Device Class  */
        .bFunctionSubClass = USB_SUBCLASS_AUDIOCONTROL,               /* Subclass code (assigned by USB-IF).*/
        .bFunctionProtocol = 0x00,                                    /* Protocol code (assigned by USB-IF).*/
        .iFunction = USB_STRING_INDEX_AUDIO_CHAT,                          /* The Function string descriptor index */
    },
    .audioChat.control = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_AUDIO_CHAT_CONTROL_INTERFACE_INDEX,   /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting for the interface identified in the prior field */
        .bNumEndpoints = 0x01U,                                       /* Number of endpoints used by this interface (excluding endpoint zero). */
        .bInterfaceClass = USB_AUDIO_CLASS,                           /*The interface implements the Audio Interface class  */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOCONTROL,              /*The interface implements the AUDIOCONTROL Subclass  */
        .bInterfaceProtocol = 0x00U,                                  /*The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.controlSub = {
        .bLength = sizeof(usb_descriptor_class_specific_ac_interface_headset_t),   /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,         /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_HEADER, /* HEADER descriptor subtype   */
        .bcdADC = {0x00U, 0x01U},                                          /* Audio Device compliant to the USB Audio specification version 1.00  */
        .wTotalLength = USB_HEADSET_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH, /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                                                             Includes the combined length of this descriptor header and all Unit and Terminal descriptors. */
        .bInCollection = HEADSET_AIC_COUNT,      /* The number of AudioStreaming and MIDIStreaming interfaces in the Audio Interface Collection to which this
                                                    AudioControl interface belongs   */
        .baInterfaceNr = USB_AUDIO_CHAT_STREAM_INTERFACE_INDEX_LIST
    },
    .audioChat.controlRec.inputTerminal = {
        /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID,               /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = HEADSET_TERMINAL_TYPE,                                    /* Headset terminaltype.  */
        /*
         * bi-directional terminal types (such as headset)
         * require two terminals (input + output).
         * They need to be tied together with the bAssocTerminal field
         * We link this recording input channel to the headset output channel here
         */
        .bAssocTerminal = USB_AUDIO_CHAT_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS, /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .wChannelConfig = {0x03U, 0x00U},     /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,     /* Index of a string descriptor, describing the name of the first logical channel.*/
        .iTerminal = USB_STRING_NONE,         /* Index of a string descriptor, describing the Input Terminal.*/
    },
    .audioChat.controlRec.featureUnit = {
       /* Audio Class Specific type of Feature Unit */
       .bLength = USB_AUDIO_FEATURE_UNIT_ONLY_DESC_SIZE(2, 1),                  /* Size of this descriptor, in bytes: 7+(ch+1)*n.*/
       .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,               /* CS_INTERFACE descriptor type.*/
       .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_FEATURE_UNIT, /* FEATURE_UNIT descriptor subtype.*/
       .bUnitId = USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID,                   /* Constant uniquely identifying the Unit within the audio function.
                                                                                   This value is used in all requests to address this Unit.*/
       .bSourceID = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID,               /* ID of the Unit or Terminal to which this Feature Unit is connected.*/
       .bControlSize = 0x01U,                                                   /* Size in bytes of an element of the bmaControls() array: n.*/
       .bmaControls = {0x03, 0x00, 0x00},                                       /* Bitmask: A bit set to 1 indicates that the mentioned Control is supported for channel (ch)
                                                                                   D0: Mute, D1: Volume, ...*/
       .iFeature = USB_STRING_NONE,                                             /* Index of a string descriptor, describing this Feature Unit.*/
    },
    .audioChat.controlRec.outputTerminal = {
        /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_RECORDER_CONTROL_OUTPUT_TERMINAL_ID,               /* Constant uniquely identifying the Terminal within
                                                                                       the audio function*/
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface */
        .bAssocTerminal = 0x00U,                      /* This Output Terminal has no association   */
        .bSourceID = USB_AUDIO_RECORDER_CONTROL_FEATURE_UNIT_ID, /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .iTerminal = USB_STRING_NONE,                            /* Index of a string descriptor, describing the Output Terminal.  */
    },
    .audioChat.controlSpkr.inputTerminal = {
        /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface .  */
        .bAssocTerminal = 0x00U,                      /* This Input Terminal has no association  */
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,         /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .wChannelConfig = {0x03U, 0x00U},             /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,             /* Index of a string descriptor, describing the name of the first logical channel.*/
        .iTerminal = USB_STRING_NONE,                 /* Index of a string descriptor, describing the Input Terminal.*/
                                           
    },
    .audioChat.controlSpkr.outputTerminal = {
        /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_CHAT_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within
                                                                                     the audio function*/
        .wTerminalType = HEADSET_TERMINAL_TYPE,                                     /* Headset/Headphones terminaltype. */
        .bAssocTerminal = USB_AUDIO_RECORDER_CONTROL_INPUT_TERMINAL_ID, /* See info in recorder audio class type for more info
                                                                           on the bAssocTerminal and why this is needed */
        .bSourceID = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,    /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .iTerminal = USB_STRING_NONE,                                   /* Index of a string descriptor, describing the Output Terminal.  */
    },
    .audioChat.controlInterruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_AC_INTERRUPT_ENDPOINT,                         /* Size of this descriptor, in bytes: 9U */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                /* ENDPOINT descriptor type */
        .bEndpointAddress = USB_AUDIO_CHAT_CONTROL_ENDPOINT |
                           (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* Endpoint address */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                         /* Transfer type */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE),
                           USB_SHORT_GET_HIGH(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE)},      /* Max Packet Size */
        .bInterval = FS_AUDIO_INTERRUPT_IN_INTERVAL,                                    /* Interval */
        .bRefresh = 0x00U,                                                              /* Unused */
        .bSynchAddress = 0x00U,                                                         /* no sychronization endpoint */
    },
    .audioChat.streamRec.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                    /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,              /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX, /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                    /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                        /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                            /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                   /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                 /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamRec.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                    /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,              /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX, /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                    /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = 0x01U,                                        /* The number of endpoints used by this interface is 1 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                            /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                   /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                 /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamRec.asInterface = {
        /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = USB_AUDIO_STREAMING_IFACE_DESC_SIZE,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_RECORDER_CONTROL_OUTPUT_TERMINAL_ID,       /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bDelay = 0x00U,                                                      /* Delay introduced by the data path. Expressed in number of frames.  */
        .wFormatTag = {0x01U, 0x00U},                                         /* PCM  */
    },
    .audioChat.streamRec.audioFormat = {
        /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = USB_AUDIO_STREAMING_TYPE_I_DESC_SIZE,                          /* bLength (11) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bNrChannels = AUDIO_IN_FORMAT_CHANNELS,                                  /* Number of Channels: one channel */
        .bSubFrameSize = AUDIO_IN_FORMAT_SIZE,                                    /* SubFrame Size: one byte per audio subframe */
        .bBitResolution = AUDIO_IN_FORMAT_BITS,                                    /* Bit Resolution: 16 bits per sample */
        .bSamFreqType = 0x01U,                                                    /* One frequency supported */
        .tSamFreq = {{TSAMFREQ2BYTES(AUDIO_OUT_SAMPLING_RATE_KHZ * 1000)}},           /* sampling frequency */
    },

#if USB_DEVICE_AUDIO_USE_SYNC_MODE
    .audioChat.streamRec.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_RECORDER_STREAM_ENDPOINT |
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),  /* IN endpoint address */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE)},
        .bInterval = FS_ISO_IN_ENDP_INTERVAL, /* The polling interval value is every 1 Frames. If Hi-Speed, every 1 uFrames*/
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = 0x00U,                                                           /* no sychronization endpoint */
    },
#else
    .audioChat.streamRec.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_RECORDER_STREAM_ENDPOINT |
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),  /* IN endpoint address */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | (0x2 << 4U),                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_FORMAT_IN_CHANNELS * AUDIO_RX_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_FORMAT_IN_CHANNELS * AUDIO_RX_FORMAT_SIZE)},
        .bInterval = FS_ISO_IN_ENDP_INTERVAL, /* The polling interval value is every 1 Frames. If Hi-Speed, every 1 uFrames*/
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = 0x00U,                                                           /* no sychronization endpoint */
    },
#endif

    .audioChat.streamRec.specificIsoEndpoint = {
        /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00U,                                         /* Bit 0: Sampling Frequency 0
                                                                          Bit 1: Pitch 0
                                                                          Bit 7: MaxPacketsOnly 0   */
        .bLockDelayUnits = 0x00U,     /* Indicates the units used for the wLockDelay field: 0: Undefined  */
        .wLockDelay = {0x00U, 0x00U}, /* Indicates the time it takes this endpoint to reliably lock its internal clock recovery circuitry */
    },
    .audioChat.streamSpkr.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_CHAT_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                         /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                             /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                        /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamSpkr.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_CHAT_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                         /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT,           /* The number of endpoints used by this interface is 2 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                        /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioChat.streamSpkr.asInterface = {
        /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = USB_AUDIO_STREAMING_IFACE_DESC_SIZE,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_CHAT_SPEAKER_CONTROL_INPUT_TERMINAL_ID,    /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bDelay = 0x01U,                                                      /* Delay introduced by the data path. Expressed in number of frames.  */
        .wFormatTag = {0x01U, 0x00U},                                         /* PCM  */
    },
    .audioChat.streamSpkr.audioFormat = {
        /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = USB_AUDIO_STREAMING_TYPE_I_DESC_SIZE,                          /* bLength (11) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,                                 /* Number of Channels*/
        .bSubFrameSize = AUDIO_OUT_FORMAT_SIZE,                              /* SubFrame Size. 2 bytes. */
        .bBitResolution = AUDIO_OUT_FORMAT_BITS,                              /* Bit Resolution. 16 bits per sample. */
        .bSamFreqType = 0x01U,                                                    /* One frequency supported */
        .tSamFreq = {{TSAMFREQ2BYTES(AUDIO_OUT_SAMPLING_RATE_KHZ * 1000)}},           /* sampling frequency */
    },

#if USB_DEVICE_AUDIO_USE_SYNC_MODE
    .audioChat.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = 0x00U,                                                           /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#else
    .audioChat.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT |
                         (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),     /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#endif

    .audioChat.streamSpkr.specificIsoEndpoint = {
        /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00U,                                         /* Bit 0: Sampling Frequency 0
                                                                          Bit 1: Pitch 0
                                                                          Bit 7: MaxPacketsOnly 0   */
        .bLockDelayUnits = 0x00U,     /* Indicates the units used for the wLockDelay field: 0: Undefined  */
        .wLockDelay = {0x00U, 0x00U}, /* Indicates the time it takes this endpoint to reliably lock its internal clock recovery circuitry */
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    .audioChat.feedbackEndpointSpkr = {
        /* Endpoint 3 Feedback ENDPOINT */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                 /* bLength */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                 /* bDescriptorType */
        .bEndpointAddress = USB_AUDIO_CHAT_SPEAKER_FEEDBACK_ENDPOINT |   /* This is an IN endpoint */
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | 0x10,          /*  Types - Transfer: ISOCHRONOUS Sync: Async Usage: Feedback EP */
        .wMaxPacketSize = {FS_ISO_FEEDBACK_ENDP_PACKET_SIZE, 0x00},      /* wMaxPacketSize */
        .bInterval = 0x01,                                               /* interval polling(2^x ms) */
        .bRefresh = 0x05,                                                /* bRefresh(32ms)  */
        .bSynchAddress = 0x00,                                           /* unused */
    },
#endif
/* -------------------------------------------------------Audio game------------------------------------------------------- */
    .audioGame.iadAudio = {
        .bLength = USB_IAD_DESC_SIZE,                                 /* Size of this descriptor in bytes*/
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* Constant name specifying INTERFACE ASSOCIATION Descriptor*/
        .bFirstInterface = USB_AUDIO_GAME_CONTROL_INTERFACE_INDEX,    /* Interface number of the first interface that is associated with this function.*/
        .bInterfaceCount = USB_AUDIO_GAME_COMPOSITE_INTERFACE_COUNT,  /* Number of contiguous interfaces that are associated with this function.*/
        .bFunctionClass = USB_AUDIO_CLASS,                            /* The function belongs to the Audio Control Device Class  */
        .bFunctionSubClass = USB_SUBCLASS_AUDIOCONTROL,               /* Subclass code (assigned by USB-IF).*/
        .bFunctionProtocol = 0x00,                                    /* Protocol code (assigned by USB-IF).*/
        .iFunction = USB_STRING_INDEX_AUDIO_GAME,                     /* The Function string descriptor index */
    },
    .audioGame.control = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_AUDIO_GAME_CONTROL_INTERFACE_INDEX,   /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting for the interface identified in the prior field */
        .bNumEndpoints = 0x01U,                                       /* Number of endpoints used by this interface (excluding endpoint zero). */
        .bInterfaceClass = USB_AUDIO_CLASS,                           /*The interface implements the Audio Interface class  */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOCONTROL,              /*The interface implements the AUDIOCONTROL Subclass  */
        .bInterfaceProtocol = 0x00U,                                  /*The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.controlSub = {
        .bLength = sizeof(usb_descriptor_class_specific_ac_interface_headphones_t),   /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,         /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_HEADER, /* HEADER descriptor subtype   */
        .bcdADC = {0x00U, 0x01U},                                          /* Audio Device compliant to the USB Audio specification version 1.00  */
        .wTotalLength = USB_HEADPHONES_AUDIO_CONTROL_INTERFACE_TOTAL_LENGTH, /* Total number of bytes returned for the class-specific AudioControl interface descriptor.
                                                                                Includes the combined length of this descriptor header and all Unit and Terminal descriptors. */
        .bInCollection = HEADPHONES_AIC_COUNT, /* The number of AudioStreaming and MIDIStreaming interfaces in the Audio Interface Collection to which this
                                                  AudioControl interface belongs   */
        .baInterfaceNr = USB_AUDIO_GAME_STREAM_INTERFACE_INDEX_LIST
    },
    .audioGame.controlSpkr.inputTerminal = {
        /* Audio Class Specific type of Input Terminal*/
        .bLength = USB_AUDIO_INPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                 /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_INPUT_TERMINAL, /* INPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within the audio
                                                                                      function. This value is used in all requests
                                                                                      to address this Terminal.  */
        .wTerminalType = USB_STREAMING_TERMINAL_TYPE, /* A Terminal dealing with a signal carried over an endpoint in an AudioStreaming interface .  */
        .bAssocTerminal = 0x00U,                      /* This Input Terminal has no association  */
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,     /* This Terminal's output audio channel cluster has 2 logical output channels.*/
        .wChannelConfig = {0x03U, 0x00U},             /* Spatial locations present in the cluster.*/
        .iChannelNames = USB_STRING_NONE,             /* Index of a string descriptor, describing the name of the first logical channel.*/
        .iTerminal = USB_STRING_NONE,                 /* Index of a string descriptor, describing the Input Terminal.*/
    },
    .audioGame.controlSpkr.outputTerminal = {
        /* Audio Class Specific type of Output Terminal */
        .bLength = USB_AUDIO_OUTPUT_TERMINAL_ONLY_DESC_SIZE,                        /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                  /* CS_INTERFACE Descriptor Type   */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_CONTROL_OUTPUT_TERMINAL, /* OUTPUT_TERMINAL descriptor subtype  */
        .bTerminalId = USB_AUDIO_GAME_SPEAKER_CONTROL_OUTPUT_TERMINAL_ID,           /* Constant uniquely identifying the Terminal within
                                                                                       the audio function*/
        .wTerminalType = HEADPHONES_TERMINAL_TYPE,                                  /* Headset/Headphones terminaltype. */
        .bAssocTerminal = 0x00U,                                                    /* This Output Terminal has no association  */
        .bSourceID = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,  /* ID of the Unit or Terminal to which this Terminal is connected.   */
        .iTerminal = USB_STRING_NONE,                                   /* Index of a string descriptor, describing the Output Terminal.  */
    },
    .audioGame.controlInterruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_AC_INTERRUPT_ENDPOINT,                         /* Size of this descriptor, in bytes: 9U */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                /* ENDPOINT descriptor type */
        .bEndpointAddress = USB_AUDIO_GAME_CONTROL_ENDPOINT |
                           (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* Endpoint address */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                         /* Transfer type */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE),
                           USB_SHORT_GET_HIGH(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE)},      /* Max Packet Size */
        .bInterval = FS_AUDIO_INTERRUPT_IN_INTERVAL,                                    /* Interval */
        .bRefresh = 0x00U,                                                              /* Unused */
        .bSynchAddress = 0x00U,                                                         /* no sychronization endpoint */
    },
    .audioGame.streamSpkr.altSet0 = {
        /* Audio Class Standard INTERFACE Descriptor, alternate setting 0  */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type   */
        .bInterfaceNumber = USB_AUDIO_GAME_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x00U,                                         /* The value used to select the alternate setting for this interface is 0   */
        .bNumEndpoints = 0x00U,                                             /* The number of endpoints used by this interface is 0 (excluding endpoint zero)   */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                        /* The interface doesn't use any class-specific protocols   */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.streamSpkr.altSet1 = {
        /* Audio Class Specific INTERFACE Descriptor, alternate setting 1 */
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                         /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,                   /* INTERFACE Descriptor Type  */
        .bInterfaceNumber = USB_AUDIO_GAME_SPEAKER_STREAM_INTERFACE_INDEX,  /* The number of this interface.  */
        .bAlternateSetting = 0x01U,                                         /* The value used to select the alternate setting for this interface is 1  */
        .bNumEndpoints = USB_AUDIO_SPEAKER_STREAM_ENDPOINT_COUNT,           /* The number of endpoints used by this interface is 2 (excluding endpoint zero)    */
        .bInterfaceClass = USB_AUDIO_CLASS,                                 /* The interface implements the Audio Interface class   */
        .bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAM,                     /* The interface implements the AUDIOSTREAMING Subclass   */
        .bInterfaceProtocol = 0x00U,                                        /* The interface doesn't use any class-specific protocols  */
        .iInterface = USB_STRING_NONE,                                      /* The device doesn't have a string descriptor describing this iInterface  */
    },
    .audioGame.streamSpkr.asInterface = {
        /* Audio Class Specific CS INTERFACE Descriptor*/
        .bLength = USB_AUDIO_STREAMING_IFACE_DESC_SIZE,                       /* Size of the descriptor, in bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,            /* CS_INTERFACE Descriptor Type  */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_AS_GENERAL, /* AS_GENERAL descriptor subtype  */
        .bTerminalLink = USB_AUDIO_GAME_SPEAKER_CONTROL_INPUT_TERMINAL_ID,    /* The Terminal ID of the Terminal to which the endpoint of this interface is connected. */
        .bDelay = 0x01U,                                                      /* Delay introduced by the data path. Expressed in number of frames.  */
        .wFormatTag = {0x01U, 0x00U},                                         /* PCM  */
    },
    .audioGame.streamSpkr.audioFormat = {
        /* Audio Class Specific type I format INTERFACE Descriptor */
        .bLength = USB_AUDIO_STREAMING_TYPE_I_DESC_SIZE,                          /* bLength (11) */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_AUDIO_CS_INTERFACE,                /* bDescriptorType (CS_INTERFACE) */
        .bDescriptorSubType = USB_DESCRIPTOR_SUBTYPE_AUDIO_STREAMING_FORMAT_TYPE, /* DescriptorSubtype: AUDIO STREAMING FORMAT TYPE */
        .bFormatType = USB_AUDIO_FORMAT_TYPE_I,                                   /* Format Type: Type I */
        .bNrChannels = AUDIO_OUT_FORMAT_CHANNELS,                                 /* Number of Channels*/
        .bSubFrameSize = AUDIO_OUT_FORMAT_SIZE,                              /* SubFrame Size. 2 bytes. */
        .bBitResolution = AUDIO_OUT_FORMAT_BITS,                              /* Bit Resolution. 16 bits per sample. */
        .bSamFreqType = 0x01U,                                                    /* One frequency supported */
        .tSamFreq = {{TSAMFREQ2BYTES(AUDIO_OUT_SAMPLING_RATE_KHZ * 1000)}},           /* sampling frequency */
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
    .audioGame.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
					        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | USB_DESCRIPTOR_ENDPOINT_ATTRIBUTE_SYNC_TYPE_SYNC,                    /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = 0x00,     /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#else
    .audioGame.streamSpkr.isoEndpoint = {
        /* ENDPOINT Descriptor */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,                                  /* Descriptor size is 9 bytes  */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                  /* Descriptor type (endpoint descriptor) */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT |
                            (USB_OUT << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT), /* OUT endpoint address 1 */
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04,                                  /* Isochronous endpoint */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES),
                           USB_SHORT_GET_HIGH(FS_ISO_OUT_ENDP_PACKET_SIZE(AUDIO_TX_FORMAT_SIZE_BYTES) + AUDIO_FORMAT_OUT_CHANNELS * AUDIO_TX_FORMAT_SIZE_BYTES)}, /* 16 bytes  */
        .bInterval = FS_ISO_OUT_ENDP_INTERVAL,                                            /* bInterval(0x01U): x ms */
        .bRefresh = 0x00U,                                                                /* Unused */
        .bSynchAddress = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT |
                         (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),     /* Synchronization Endpoint (if used) is endpoint 0x83  */
    },
#endif
    .audioGame.streamSpkr.specificIsoEndpoint = {
        /* Audio Class Specific ENDPOINT Descriptor  */
        .bLength = USB_AUDIO_STREAMING_ENDP_DESC_SIZE,                 /*  Size of the descriptor, in bytes  */
        .bDescriptorType = USB_AUDIO_STREAM_ENDPOINT_DESCRIPTOR,       /* CS_ENDPOINT Descriptor Type  */
        .bDescriptorSubType = USB_AUDIO_EP_GENERAL_DESCRIPTOR_SUBTYPE, /* AUDIO_EP_GENERAL descriptor subtype  */
        .bmAttributes = 0x00U,                                         /* Bit 0: Sampling Frequency 0
                                                                          Bit 1: Pitch 0
                                                                          Bit 7: MaxPacketsOnly 0   */
        .bLockDelayUnits = 0x00U,     /* Indicates the units used for the wLockDelay field: 0: Undefined  */
        .wLockDelay = {0x00U, 0x00U}, /* Indicates the time it takes this endpoint to reliably lock its internal clock recovery circuitry */
    },
#if USB_DEVICE_AUDIO_USE_SYNC_MODE
#else
    .audioGame.feedbackEndpointSpkr = {
        /* Endpoint 3 Feedback ENDPOINT */
        .bLength = USB_ENDPOINT_AUDIO_DESCRIPTOR_LENGTH,            /* bLength */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,            /* bDescriptorType */
        .bEndpointAddress = USB_AUDIO_GAME_SPEAKER_FEEDBACK_ENDPOINT |   /* This is an IN endpoint */
                            (USB_IN << USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT),
        .bmAttributes = USB_ENDPOINT_ISOCHRONOUS | 0x04 | 0x10,          /*  Types - Transfer: ISOCHRONOUS Sync: Async Usage: Feedback EP */
        .wMaxPacketSize = {FS_ISO_FEEDBACK_ENDP_PACKET_SIZE, 0x00},      /* wMaxPacketSize */
        .bInterval = 0x01,                                               /* interval polling(2^x ms) */
        .bRefresh = 0x05,                                                /* bRefresh(32ms)  */
        .bSynchAddress = 0x00,                                           /* unused */
    },
#endif
/* ---------------------------------------------------------HID--------------------------------------------------------- */
    .hid.consumerControlInterface = {
        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,                   /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,             /* INTERFACE Descriptor Type */
        .bInterfaceNumber = USB_HID_CONSUMER_CONTROL_INTERFACE_INDEX, /* Number of this interface. */
        .bAlternateSetting = 0x00U,                                   /* Value used to select this alternate setting
                                                                         for the interface identified in the prior field */
        .bNumEndpoints = USB_HID_CONSUMER_CONTROL_ENDPOINT_COUNT,     /* Number of endpoints used by this
                                                                         interface (excluding endpoint zero). */
        .bInterfaceClass = USB_HID_CONSUMER_CONTROL_CLASS,            /* Class code (assigned by the USB-IF). */
        .bInterfaceSubClass = USB_HID_CONSUMER_CONTROL_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
        .bInterfaceProtocol = USB_HID_CONSUMER_CONTROL_PROTOCOL,      /* Protocol code (assigned by the USB). */
        .iInterface = USB_STRING_NONE,                                /* Index of string descriptor describing this interface */
    },
    .hid.hidReport= {
        .bLength = USB_DESCRIPTOR_LENGTH_HID,                                        /* Numeric expression that is the total size of the HID descriptor. */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_HID,                                  /* Constant name specifying type of HID descriptor. */
        .bcdHID = {0x00U, 0x01U},                                                    /* Numeric expression identifying the HID Class Specification release. */
        .bCountryCode = USB_HID_COUNTRY_CODE,                                        /* Numeric expression identifying country code of the localized hardware */
        .bNumDescriptors = USB_HID_CLASS_DESCRIPTOR_COUNT,                           /* Numeric expression specifying the number of class descriptors(at least one report descriptor) */
        .bHidDescriptorType = USB_DESCRIPTOR_TYPE_HID_REPORT,                        /* Constant name identifying type of class descriptor. */
        .wDescriptorLength =
            {USB_SHORT_GET_LOW(USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT),
             USB_SHORT_GET_HIGH(USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT)}, /* Numeric expression that is the total size of the Report descriptor. */
    },
    .hid.interruptEndpoint = {
        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                                /* Size of this descriptor in bytes */
        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                                          /* ENDPOINT Descriptor Type */
        .bEndpointAddress = USB_HID_CONSUMER_CONTROL_ENDPOINT | (USB_IN << 7),                    /* The address of the endpoint on the USB device described by this descriptor. */
        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                                   /* This field describes the endpoint's attributes */
        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
                           USB_SHORT_GET_HIGH(FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
        .bInterval = FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL,                               /* Interval for polling endpoint for data transfers. */
    },
		
//		/* -------------------------------------------------------VCOM HCI-------------------------------------------------------- */
//    .vcomHci.iad = {
//        /* Interface Association Descriptor */
//        .bLength =           USB_IAD_DESC_SIZE,                         /* Size of this descriptor in bytes */
//        .bDescriptorType =   USB_DESCRIPTOR_TYPE_INTERFACE_ASSOCIATION, /* INTERFACE_ASSOCIATION Descriptor Type  */
//        .bFirstInterface =   USB_CDC_VCOM_CIC_INTERFACE_INDEX,          /* The first interface number associated with this function
//                                                                           in other words: the interface number of the VCOM class */
//        .bInterfaceCount =   USB_CDC_VCOM_INTERFACE_COUNT,              /* The number of contiguous interfaces associated with this function */
//        /* The function belongs to the Communication Device/Interface Class  */
//        .bFunctionClass =    USB_CDC_VCOM_CIC_CLASS,
//        .bFunctionSubClass = USB_CDC_VCOM_CIC_SUBCLASS,
//        .bFunctionProtocol = 0x00,                      /* The function uses the No class specific protocol required Protocol  */
//        .iFunction =         USB_STRING_INDEX_VCOM,     /* The Function string descriptor index */
//
//    },
//    .vcomHci.cic = {
//        /* Interface Descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,           /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,     /* INTERFACE Descriptor Type */
//        .bInterfaceNumber = USB_CDC_VCOM_CIC_INTERFACE_INDEX, /* Number of this interface. Zero-based value identifying the index in the array of
//                                                                 concurrent interfaces supported by this configuration. */
//        .bAlternateSetting =  0x00,                           /* Value used to select this alternate setting
//                                                                 for the interface identified in the prior field */
//        .bNumEndpoints = USB_CDC_VCOM_CIC_ENDPOINT_COUNT,     /* Number of endpoints used by this interface (excluding endpoint zero). */
//        .bInterfaceClass = USB_CDC_VCOM_CIC_CLASS,            /* Class code (assigned by the USB-IF). */
//        .bInterfaceSubClass = USB_CDC_VCOM_CIC_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
//        .bInterfaceProtocol = USB_CDC_VCOM_CIC_PROTOCOL,      /* Protocol code (assigned by the USB). */
//        .iInterface = USB_STRING_NONE,                        /* Index of string descriptor describing this interface */
//    },
//    .vcomHci.functionalHeader = {
//        /* CDC Class-Specific descriptor */
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_HEADER_FUNC, /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE,  /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_HEADER_FUNC_DESC,
//        .bcdUSB = {0x10, 0x01}, /* USB Class Definitions for Communications the Communication specification version 1.10 */
//    },
//    .vcomHci.callMgmt = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_CALL_MANAG, /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE, /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_CALL_MANAGEMENT_FUNC_DESC,
//        .bmCapabilities = 0x01, /*Bit 0: Whether device handle call management itself 1,
//                                  Bit 1: Whether device can send/receive call management information over a Data Class Interface 0 */
//        .bDataInterface = 0x01, /* Indicates multiplexed commands are handled via data interface */
//    },
//    .vcomHci.abstractControl = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_ABSTRACT,   /* Size of this descriptor in bytes */
//        .bdescriptortype = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE, /* CS_INTERFACE Descriptor Type */
//        .bdescriptorsubtype = USB_CDC_ABSTRACT_CONTROL_FUNC_DESC,
//        .bmCapabilities = 0x06, /* Bit 0: Whether device supports the request combination of Set_Comm_Feature, Clear_Comm_Feature, and
//                 Get_Comm_Feature 0, Bit 1: Whether device supports the request combination of Set_Line_Coding,
//                 Set_Control_Line_State, Get_Line_Coding, and the notification Serial_State 1, Bit ...  */
//    },
//    .vcomHci.unionInterface = {
//        .bFunctionLength = USB_DESCRIPTOR_LENGTH_CDC_UNION_FUNC,   /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_CDC_CS_INTERFACE,   /* CS_INTERFACE Descriptor Type */
//        .bDescriptorSubtype = USB_CDC_UNION_FUNC_DESC,
//        .bControlInterface = USB_CDC_VCOM_CIC_INTERFACE_INDEX,     /* The interface number of the Communications or Data Class interface  */
//        .bSubordinateInterface = USB_CDC_VCOM_DIC_INTERFACE_INDEX, /* Interface number of subordinate interface in the Union  */
//    },
////    .vcomHci.interruptEndpoint = {
////        /*Notification Endpoint descriptor */
////        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                    /* Size of this descriptor in bytes */
////        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                              /* ENDPOINT Descriptor Type */
////        .bEndpointAddress = USB_CDC_VCOM_CIC_INTERRUPT_IN_ENDPOINT | (USB_IN << 7U),  /* The address of the endpoint on the USB device described by this descriptor. */
////        .bmAttributes = USB_ENDPOINT_INTERRUPT,                                       /* This field describes the endpoint's attributes */
////        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_INTERRUPT_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
////                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_INTERRUPT_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
////        .bInterval = FS_CDC_VCOM_INTERRUPT_IN_INTERVAL,                               /* Interval for polling endpoint for data transfers. */
////    },
//    .vcomHci.dic = {
//        /* Data Interface Descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_INTERFACE,           /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,     /* INTERFACE Descriptor Type */
//        .bInterfaceNumber = USB_CDC_VCOM_DIC_INTERFACE_INDEX, /* Number of this interface. Zero-based value identifying the index in the array of
//                                                                 concurrent interfaces supported by this configuration. */
//        .bAlternateSetting = 0x00,                            /* Value used to select this alternate setting
//                                                                 for the interface identified in the prior field */
//        .bNumEndpoints = USB_CDC_VCOM_DIC_ENDPOINT_COUNT,     /* Number of endpoints used by this interface (excluding endpoint zero). */
//        .bInterfaceClass = USB_CDC_VCOM_DIC_CLASS,            /* Class code (assigned by the USB-IF). */
//        .bInterfaceSubClass = USB_CDC_VCOM_DIC_SUBCLASS,      /* Subclass code (assigned by the USB-IF). */
//        .bInterfaceProtocol = USB_CDC_VCOM_DIC_PROTOCOL,      /* Protocol code (assigned by the USB). */
//        .iInterface = USB_STRING_NONE,                        /* Index of string descriptor describing this interface */
//    },
//    .vcomHci.bulkIn = {
//        /*Bulk IN Endpoint descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                               /* Size of this descriptor in bytes */
//        .bDescriptorType =  USB_DESCRIPTOR_TYPE_ENDPOINT,                        /* ENDPOINT Descriptor Type */
//        .bEndpointAddress = USB_CDC_VCOM_DIC_BULK_IN_ENDPOINT | (USB_IN << 7U),  /* The address of the endpoint on the USB device described by this descriptor. */
//        .bmAttributes = USB_ENDPOINT_BULK,                                       /* This field describes the endpoint's attributes */
//        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_BULK_IN_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
//                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_BULK_IN_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
//        .bInterval = 0x00,                                                       /* The polling interval value is every 0 Frames */
//    },
//    .vcomHci.bulkOut = {
//         /*Bulk OUT Endpoint descriptor */
//        .bLength = USB_DESCRIPTOR_LENGTH_ENDPOINT,                                /* Size of this descriptor in bytes */
//        .bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,                          /* ENDPOINT Descriptor Type */
//        .bEndpointAddress = USB_CDC_VCOM_DIC_BULK_OUT_ENDPOINT | (USB_OUT << 7U), /* The address of the endpoint on the USB device described by this descriptor. */
//        .bmAttributes = USB_ENDPOINT_BULK,                                        /* This field describes the endpoint's attributes */
//        .wMaxPacketSize = {USB_SHORT_GET_LOW(FS_CDC_VCOM_BULK_OUT_PACKET_SIZE),   /* Maximum packet size this endpoint is capable of */
//                           USB_SHORT_GET_HIGH(FS_CDC_VCOM_BULK_OUT_PACKET_SIZE)}, /* sending or receiving when this configuration is selected. */
//        .bInterval = 0x00,                                                        /* The polling interval value is every 0 Frames */
//    },
#endif
};

#if (defined(USB_DEVICE_CONFIG_CV_TEST) && (USB_DEVICE_CONFIG_CV_TEST > 0U))
USB_DMA_INIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE)
uint8_t g_UsbDeviceQualifierDescriptor[] = {
    USB_DESCRIPTOR_LENGTH_DEVICE_QUALITIER, /* Size of this descriptor in bytes */
    USB_DESCRIPTOR_TYPE_DEVICE_QUALITIER,   /* DEVICE Descriptor Type */
    USB_SHORT_GET_LOW(USB_DEVICE_SPECIFIC_BCD_VERSION),
    USB_SHORT_GET_HIGH(USB_DEVICE_SPECIFIC_BCD_VERSION), /* USB Specification Release Number in
                                                            Binary-Coded Decimal (i.e., 2.10 is 210H). */
    USB_DEVICE_CLASS,                                    /* Class code (assigned by the USB-IF). */
    USB_DEVICE_SUBCLASS,                                 /* Subclass code (assigned by the USB-IF). */
    USB_DEVICE_PROTOCOL,                                 /* Protocol code (assigned by the USB-IF). */
    USB_CONTROL_MAX_PACKET_SIZE,                         /* Maximum packet size for endpoint zero
                                                            (only 8, 16, 32, or 64 are valid) */
    0x00U,                                               /* Number of Other-speed Configurations */
    0x00U,                                               /* Reserved for future use, must be zero */
};

#endif


/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief USB device get device descriptor function.
 *
 * This function gets the device descriptor of the USB device.
 *
 * @param handle The USB device handle.
 * @param deviceDescriptor The pointer to the device descriptor structure.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceGetDeviceDescriptor(usb_device_handle handle,
                                           usb_device_get_device_descriptor_struct_t *deviceDescriptor)
{
    deviceDescriptor->buffer = g_UsbDeviceDescriptor;
    deviceDescriptor->length = USB_DESCRIPTOR_LENGTH_DEVICE;
    return kStatus_USB_Success;
}
#if (defined(USB_DEVICE_CONFIG_CV_TEST) && (USB_DEVICE_CONFIG_CV_TEST > 0U))
/* Get device qualifier descriptor request */
usb_status_t USB_DeviceGetDeviceQualifierDescriptor(
    usb_device_handle handle, usb_device_get_device_qualifier_descriptor_struct_t *deviceQualifierDescriptor)
{
    deviceQualifierDescriptor->buffer = g_UsbDeviceQualifierDescriptor;
    deviceQualifierDescriptor->length = USB_DESCRIPTOR_LENGTH_DEVICE_QUALITIER;
    return kStatus_USB_Success;
}
#endif
/*!
 * @brief USB device get configuration descriptor function.
 *
 * This function gets the configuration descriptor of the USB device.
 *
 * @param handle The USB device handle.
 * @param configurationDescriptor The pointer to the configuration descriptor structure.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceGetConfigurationDescriptor(
    usb_device_handle handle, usb_device_get_configuration_descriptor_struct_t *configurationDescriptor)
{
    if (USB_COMPOSITE_CONFIGURE_INDEX > configurationDescriptor->configuration)
    {
        configurationDescriptor->buffer = (uint8_t *)&g_UsbDeviceConfigurationDescriptor;
        configurationDescriptor->length = sizeof(g_UsbDeviceConfigurationDescriptor);
        return kStatus_USB_Success;
    }
    return kStatus_USB_InvalidRequest;
}

/*!
 * @brief USB device get string descriptor function.
 *
 * This function gets the string descriptor of the USB device.
 *
 * @param handle The USB device handle.
 * @param stringDescriptor The pointer to the string descriptor structure.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceGetStringDescriptor(usb_device_handle handle,
                                           usb_device_get_string_descriptor_struct_t *stringDescriptor)
{
    if (stringDescriptor->stringIndex == 0U)
    {
        stringDescriptor->buffer = (uint8_t *)g_UsbDeviceLanguageList.languageString;
        stringDescriptor->length = g_UsbDeviceLanguageList.stringLength;
    }
    else
    {
        uint8_t languageId    = 0U;
        uint8_t languageIndex = USB_DEVICE_STRING_COUNT;

        for (; languageId < USB_DEVICE_LANGUAGE_COUNT; languageId++)
        {
            if (stringDescriptor->languageId == g_UsbDeviceLanguageList.languageList[languageId].languageId)
            {
                if (stringDescriptor->stringIndex < USB_DEVICE_STRING_COUNT)
                {
                    languageIndex = stringDescriptor->stringIndex;
                }
                break;
            }
        }

        if (USB_DEVICE_STRING_COUNT == languageIndex)
        {
            return kStatus_USB_InvalidRequest;
        }
        stringDescriptor->buffer = (uint8_t *)g_UsbDeviceLanguageList.languageList[languageId].string[languageIndex];
        stringDescriptor->length = g_UsbDeviceLanguageList.languageList[languageId].length[languageIndex];
    }
    return kStatus_USB_Success;
}

/* Get hid report descriptor request */
usb_status_t USB_DeviceGetHidReportDescriptor(usb_device_handle handle,
                                              usb_device_get_hid_report_descriptor_struct_t *hidReportDescriptor)
{
    if (USB_HID_CONSUMER_CONTROL_INTERFACE_INDEX == hidReportDescriptor->interfaceNumber)
    {
        hidReportDescriptor->buffer = g_UsbDeviceHidConsumerControlReportDescriptor;
        hidReportDescriptor->length = USB_DESCRIPTOR_LENGTH_HID_CONSUMER_CONTROL_REPORT;
    }
    else
    {
        return kStatus_USB_InvalidRequest;
    }
    return kStatus_USB_Success;
}

/* Due to the difference of HS and FS descriptors, the device descriptors and configurations need to be updated to match
 * current speed.
 * As the default, the device descriptors and configurations are configured by using FS parameters for both EHCI and
 * KHCI.
 * When the EHCI is enabled, the application needs to call this function to update device by using current speed.
 * The updated information includes endpoint max packet size, endpoint interval, etc. */
usb_status_t USB_DeviceSetSpeed(usb_device_handle handle, uint8_t speed)
{
    usb_descriptor_union_t *descriptorHead;
    usb_descriptor_union_t *descriptorTail;

    descriptorHead = (usb_descriptor_union_t *)&g_UsbDeviceConfigurationDescriptor;
    descriptorTail =  (usb_descriptor_union_t *)((uint8_t *)descriptorHead + sizeof(g_UsbDeviceConfigurationDescriptor) - 1U);
     // descriptorTail =  (usb_descriptor_union_t *)((uint8_t *) & g_UsbDeviceConfigurationDescriptor[USB_DESCRIPTOR_LENGTH_CONFIGURATION_ALL - 1U]);

    while (descriptorHead < descriptorTail)
    {
        if (descriptorHead->common.bDescriptorType == USB_DESCRIPTOR_TYPE_ENDPOINT)
        {
            if (USB_SPEED_HIGH == speed)
            {
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
                if ((USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((HS_ISO_OUT_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((HS_ISO_OUT_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_RECORDER_STREAM_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_IN_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((HS_ISO_IN_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
#else
                if ((USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(
                        (HS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                        descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_FEEDBACK_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(HS_ISO_FEEDBACK_ENDP_PACKET_SIZE,
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_RECORDER_STREAM_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = HS_ISO_IN_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(
                        (HS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE),
                        descriptorHead->endpoint.wMaxPacketSize);
                }
#endif
                else if ((USB_HID_CONSUMER_CONTROL_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = HS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(HS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE,
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else
                {
                }
            }
            else
            {
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
                if ((USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((FS_ISO_OUT_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_GAME_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((FS_ISO_OUT_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_RECORDER_STREAM_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_IN_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS((FS_ISO_IN_ENDP_PACKET_SIZE),
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
#else
                if ((USB_AUDIO_CHAT_SPEAKER_STREAM_ENDPOINT ==
                     (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                    ((descriptorHead->endpoint.bEndpointAddress >> USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) ==
                     USB_OUT))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_OUT_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(
                        (FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE),
                        descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_SPEAKER_FEEDBACK_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_FEEDBACK_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(FS_ISO_FEEDBACK_ENDP_PACKET_SIZE,
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else if ((USB_AUDIO_RECORDER_STREAM_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = FS_ISO_IN_ENDP_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(
                        (FS_ISO_IN_ENDP_PACKET_SIZE + AUDIO_IN_FORMAT_CHANNELS * AUDIO_IN_FORMAT_SIZE),
                        descriptorHead->endpoint.wMaxPacketSize);
                }
#endif
                else if ((USB_HID_CONSUMER_CONTROL_ENDPOINT ==
                          (descriptorHead->endpoint.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK)) &&
                         ((descriptorHead->endpoint.bEndpointAddress >>
                           USB_DESCRIPTOR_ENDPOINT_ADDRESS_DIRECTION_SHIFT) == USB_IN))
                {
                    descriptorHead->endpoint.bInterval = FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL;
                    USB_SHORT_TO_LITTLE_ENDIAN_ADDRESS(FS_AUDIO_INTERRUPT_IN_PACKET_SIZE,
                                                       descriptorHead->endpoint.wMaxPacketSize);
                }
                else
                {
                }
            }
        }
        descriptorHead = (usb_descriptor_union_t *)((uint8_t *)descriptorHead + descriptorHead->common.bLength);
    }

    for (int i = 0U; i < USB_AUDIO_ENDPOINT_COUNT; i++)
    {
        if (USB_SPEED_HIGH == speed)
        {
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
            g_UsbDeviceAudioChatSpeakerEndpoints[0].maxPacketSize = (HS_ISO_OUT_ENDP_PACKET_SIZE);
            g_UsbDeviceAudioChatSpeakerEndpoints[0].interval      = (HS_ISO_OUT_ENDP_INTERVAL);
#else
            g_UsbDeviceAudioRecorderEndpoints[0].maxPacketSize = HS_ISO_IN_ENDP_PACKET_SIZE;
            g_UsbDeviceAudioRecorderEndpoints[0].interval      = HS_ISO_IN_ENDP_INTERVAL;

            g_UsbDeviceAudioChatSpeakerEndpoints[0].maxPacketSize =
                (HS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE);
            g_UsbDeviceAudioChatSpeakerEndpoints[0].interval      = (HS_ISO_OUT_ENDP_INTERVAL);
            g_UsbDeviceAudioChatSpeakerEndpoints[1].maxPacketSize = HS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
            g_UsbDeviceAudioChatSpeakerEndpoints[1].interval      = HS_ISO_FEEDBACK_ENDP_INTERVAL;
#endif
        }
        else
        {
#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
            g_UsbDeviceAudioChatSpeakerEndpoints[0].maxPacketSize = (FS_ISO_OUT_ENDP_PACKET_SIZE);
            g_UsbDeviceAudioChatSpeakerEndpoints[0].interval      = (FS_ISO_OUT_ENDP_INTERVAL);
#else
            g_UsbDeviceAudioChatSpeakerEndpoints[0].maxPacketSize =
                (FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE);
            g_UsbDeviceAudioChatSpeakerEndpoints[0].interval      = (FS_ISO_OUT_ENDP_INTERVAL);
            g_UsbDeviceAudioChatSpeakerEndpoints[1].maxPacketSize = FS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
            g_UsbDeviceAudioChatSpeakerEndpoints[1].interval      = FS_ISO_FEEDBACK_ENDP_INTERVAL;
#endif
        }
    }

    for (int i = 0U; i < USB_HID_CONSUMER_CONTROL_ENDPOINT_COUNT; i++)
    {
        if (USB_SPEED_HIGH == speed)
        {
            g_UsbDeviceHidEndpoints[i].maxPacketSize = HS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE;
            g_UsbDeviceHidEndpoints[i].interval      = HS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL;
        }
        else
        {
            g_UsbDeviceHidEndpoints[i].maxPacketSize = FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_PACKET_SIZE;
            g_UsbDeviceHidEndpoints[i].interval      = FS_HID_CONSUMER_CONTROL_INTERRUPT_IN_INTERVAL;
        }
    }

    return kStatus_USB_Success;
}
