/* ---------------------------------------------------------------------------- */
/* Copyright 2020, 2024 NXP.                                                    */
/* SPDX-License-Identifier: BSD-3-Clause                                        */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */

#ifndef AUDIO_TX_H_
#define AUDIO_TX_H_

/* ---------------------------------------------------------------------------- */
/* Include files                                                                */
/* ---------------------------------------------------------------------------- */

#include <stdint.h>
#include "fsl_i2s_dma.h"

/* ---------------------------------------------------------------------------- */
/* Defines                                                                      */
/* ---------------------------------------------------------------------------- */

/**
 * Callback type used by the Audio Transmitter to get new samples to be transmitted.
 *
 * @param[out] pDest Pointer to the (memory) destination address, where the requested samples have to be written to.
 * @param[in]  nbSamples Number of audio samples to be written.
 */
typedef void (*audio_get_samples_cb)(void *pDest, uint16_t nbSamples);

/* -------------------------------------------------------------------------
 * Public functions
 * ------------------------------------------------------------------------- */

/**
 * Initialization of Transmitter part of audio
 *  Configures I2S, Inits I2S, Inits Audio Formats, Inits SAI,
 *  Inits DMA
 */
void audio_InitTx(void);

/**
 * Start sending audio samples to I2S
 *  Starts DMA, Starts SAI, Enables I2S interrupts
 *
 * @param cb The transmitter will repeatedly call this function to request new samples to be transmitted.
 */
void audio_StartTx(audio_get_samples_cb getEmptySamplesCb, audio_get_samples_cb getMixedSamplesCb);

/**
 * Stops the data transmission.
 */
void audio_StopTx(void);

/**
 * Resets the transmitter
 */
void audio_ResetTx(void);

void audio_WriteBuffer(uint8_t *pBufferInput, uint16_t NbSamplesReceived);

/**
 * Enable bit clock and WS signal.
 */
void audio_EnableBitClock(void);

/**
 * Determine if bit clock is enabled.
 */
bool audio_IsBitClockEnabled(void);

void audio_RingbufferTxInit(void);

void dma_AbortI2sDmaTransfer(I2S_Type *base, i2s_dma_handle_t *handle);

/** @} */

#endif /* AUDIO_TX_H_ */
