/* ---------------------------------------------------------------------------- */
/* Copyright 2018, 2024 NXP.                                                    */
/* SPDX-License-Identifier: BSD-3-Clause                                        */
/*                                                                              */
/* This software is owned or controlled by NXP and may only be used strictly    */
/* in accordance with the applicable license terms. By expressly accepting such */
/* terms or by downloading, installing, activating and/or otherwise using the   */
/* software, you are agreeing that you have read, and that you agree to comply  */
/* with and are bound by, such license terms. If you do not agree to be bound   */
/* by the applicable license terms, then you may not retain, install, activate  */
/* or otherwise use the software.                                               */
/* ---------------------------------------------------------------------------- */

/* ---------------------------------------------------------------------------- */
/* Include files                                                                */
/* ---------------------------------------------------------------------------- */
#include <stdbool.h>
#include <stdint.h>

#include "audio_tx.h"
#include "audio_ringbuffer.h"
#include "audio_mixer.h"
#include "fsl_adapter_audio.h"
#include "usb_device_config.h"
 
#include "fsl_debug_console.h"
#include "audio_latency_debug.h"
#include "critical.h"
#include "audio.h"


/* need to include "usb.h, usb_device.h, usb_device_class.h" before include "usb_device_descriptor.h" */
#include "usb.h"
#include "usb_device.h"
#include "usb_device_class.h"
#include "usb_device_descriptor.h"
#include "composite.h"
                                                    
/* ---------------------------------------------------------------------------- */
/* Defines                                                                      */
/* ---------------------------------------------------------------------------- */
/** Audio out buffer half marker */
#define AUDIO_OUT_RING_BUFFER_HALF_FILL  (AUDIO_OUTPUT_RING_BUFFER_SIZE / 2)
                               
/** Current audio Service status */
static uint32_t s_audioService_TxStatus = 0;

/** Set a status flag of the audio Service */
#define AUDIO_SERVICE_SET_STATUS(x)  (s_audioService_TxStatus |= ((x)))

/** Clears a status flag of the audio Service */
#define AUDIO_SERVICE_CLEAR_STATUS(x)  (s_audioService_TxStatus &= ~((x)))

/** Check the current status of the audio Service */
#define AUDIO_SERVICE_CHECK_STATUS(x)  (s_audioService_TxStatus & ((x)))

#define AUDIO_OUTPUT_RING_BUFFER_SIZE            MIXER_INPUT_RING_BUFFER_SIZE
/* -------------------------------------------------------------------------
 * Local function prototypes
 * ------------------------------------------------------------------------- */
                         

/* -------------------------------------------------------------------------
 * Global function prototypes
 * ------------------------------------------------------------------------- */
extern void SDKTxCallback(void);
extern void AUDIO_DMA_EDMA_Start();
extern uint32_t AUDIO_GetPlayTransferSize(void);

void audio_StopTx(void);
void TxCallback(hal_audio_handle_t handle, hal_audio_status_t completionStatus, void *callbackParam);
uint32_t AUDIO_GetTxDMATransferSize(void);

/* -------------------------------------------------------------------------
 * variable
 * ------------------------------------------------------------------------- */
extern uint8_t g_RxDmaStartFlag;
extern hal_audio_transfer_t s_TxTransfer; 
extern hal_audio_config_t audioTxConfig;
extern usb_device_composite_struct_t g_composite;
extern HAL_AUDIO_HANDLE_DEFINE(audioTxHandle);

extern uint32_t g_txRestartTimes;
extern uint32_t g_mixerRingBufferFilling[2];

uint8_t g_I2sTxStarted;

/* ---------------------------------------------------------------------------- */
/* Local variables                                                              */
/* ---------------------------------------------------------------------------- */
static i2s_dma_handle_t s_TxHandle; 
static uint32_t s_audioPlayTransferSize;
static audio_get_samples_cb s_audio_GetSamplesCb;

/** ring buffer required to allocate the output samples of the application */
phRingBuffer_t s_audioService_RingBufferOut;

/** defined buffer to store the audio output samples using the DMA */
uint8_t __attribute__((__aligned__(4))) s_audioService_BufferOut[AUDIO_OUTPUT_RING_BUFFER_SIZE];

/**
 * Callback passed to @ref audio_StartTx to get the samples to be transmitted.
 */     
static void audio_GetAndTransmitSamples(uint16_t nbSamples)
{
    /* we're not using @ref audio_ringbuffer_Write for optimization reasons;
       instead we let the callback write to the output ring buffer directly */
    uint8_t audioTxSampleSizeBytes = audio_GetTxSampleSizeBytes();
    uint16_t nbBytes = nbSamples * audioTxSampleSizeBytes;


    /* write until the end of the buffer */
    if ((s_audioService_RingBufferOut.pWritePointer + nbBytes) > (uint8_t *)s_audioService_RingBufferOut.EndAddress) {
        uint16_t nbBytesEnd = (uint8_t *)s_audioService_RingBufferOut.EndAddress - s_audioService_RingBufferOut.pWritePointer + 1;
        uint16_t nbSamplesEnd = nbBytesEnd / audioTxSampleSizeBytes;
        s_audio_GetSamplesCb((void *)s_audioService_RingBufferOut.pWritePointer, nbSamplesEnd);
        s_audioService_RingBufferOut.pWritePointer = (uint8_t *)s_audioService_RingBufferOut.StartAddress;
        nbSamples -= nbSamplesEnd;
        nbBytes -= nbBytesEnd;
    }

    /* write the remaining samples */
    s_audio_GetSamplesCb((void *)s_audioService_RingBufferOut.pWritePointer, nbSamples);
    s_audioService_RingBufferOut.pWritePointer += nbBytes;
}
                                                                      

/**
 * @brief Call back function of DMA Tx
 *
 * @param handle pointer to DMA handle
 * @param userData pointer to Callback userdata
 * @return void.
 */
void TxCallback(hal_audio_handle_t handle, hal_audio_status_t completionStatus, void *callbackParam)
{
    AUDIO_LatencyDebug_SendingAudioToI2s(true);
    
    uint8_t audioTxSampleSizeBytes = audio_GetTxSampleSizeBytes();
    
    s_audioPlayTransferSize = AUDIO_GetTxDMATransferSize();
    
    uint16_t nbSamples =  s_audioPlayTransferSize / audioTxSampleSizeBytes;
    
    s_TxTransfer.data     = (uint8_t *)s_audioService_RingBufferOut.pWritePointer;
    s_TxTransfer.dataSize = s_audioPlayTransferSize;
    audio_GetAndTransmitSamples(nbSamples); /* write mixed samples into the transmitter ring buffer */
    HAL_AudioTransferSendNonBlocking((hal_audio_handle_t)&audioTxHandle[0], &s_TxTransfer);
    
#if ((defined(USB_DEVICE_CONFIG_AUDIO_SPEAKER)) && (USB_DEVICE_CONFIG_AUDIO_SPEAKER > 0U))

    uint8_t biggestIndex = 0;
    uint8_t minimalIndex = 0;

    uint32_t biggestFilling = AUDIO_MIXER_GetBiggestFilling(&biggestIndex);
    uint32_t minimalFilling = AUDIO_MIXER_GetMininalFillingLevel(&minimalIndex);
    

    if (biggestFilling > ((uint32_t)(MIXER_INPUT_RING_BUFFER_SIZE - 2 * s_audioPlayTransferSize)))
    {
        AUDIO_MIXER_StopInterface(biggestIndex);
        PRINTF("samples overflow, %d\n", biggestFilling);
    }
    if(minimalFilling < s_audioPlayTransferSize)
    {
        AUDIO_MIXER_StopInterface(minimalIndex);
        /* The execution time of the PRIMTF function should be minimized as much as possible, and it must be less than 1ms, otherwise noise will occur */
        PRINTF("no sample, %d\n", minimalFilling);
    }

#endif

    AUDIO_LatencyDebug_SendingAudioToI2s(false);
        
    /** trigger proper events */
    AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_TX_COMPLETE_MASK);
   
}

/* ---------------------------------------------------------------------------- */
/* Local functions-TX                                                           */
/* ---------------------------------------------------------------------------- */
static void audio_TxSpaceWarningCb(uint32_t Filling, void *pCtxt)
{
    (void)Filling;    /* make release build happy */
    (void)pCtxt;      /* make release build happy */

    PRINTF("TxSpaceWarning, F=%d", Filling);
}

/* ---------------------------------------------------------------------------- */
/* Public Functions-TX                                                          */
/* ---------------------------------------------------------------------------- */
///* Enable bit clock and WS signal */
//void audio_EnableBitClock(void)
//{
//    DEMO_I2S_TX->CFG1 |= I2S_CFG1_MAINENABLE_MASK;
//}
//
//bool audio_IsBitClockEnabled(void)
//{
//    return DEMO_I2S_TX->CFG1 & I2S_CFG1_MAINENABLE_MASK;
//}
/**
 * Initializes and starts the I2S-Tx as part of audio-SAI
 */
void audio_InitTx(void)
{
    
}


void audio_StartTx(audio_get_samples_cb getEmptySamplesCb, audio_get_samples_cb getMixedSamplesCb)
{
    assert(getEmptySamplesCb);
    assert(getMixedSamplesCb);
    uint8_t audioTxSampleSizeBytes = audio_GetTxSampleSizeBytes();
    s_audioPlayTransferSize = AUDIO_GetTxDMATransferSize();

    if (!AUDIO_SERVICE_CHECK_STATUS(AUDIO_SERVICE_TX_STARTED_MASK)) {
        AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_TX_COMPLETE_MASK);
      
        g_I2sTxStarted = 1;
        //s_audio_GetSamplesCb = getMixedSamplesCb;  /* set callback for getting mixed samples */
              
        s_TxTransfer.dataSize = s_audioPlayTransferSize;
        s_TxTransfer.data     = (uint8_t *)s_audioService_RingBufferOut.pWritePointer;

        /* prepare the first transmission */
        uint16_t nbSamples = s_audioPlayTransferSize / audioTxSampleSizeBytes;

        s_audio_GetSamplesCb = getMixedSamplesCb;
        /* get and write samples into the output ring buffer */
        audio_GetAndTransmitSamples(nbSamples);   
        
        /* The following code is to void I2S L/R shift issue, this issue may occur when I2S RX is enabled first */
        /* If MAINENABLE bit is already been enabled. This means that bit clock and WS have appeared. */
        if(DEMO_I2S_TX->CFG1 & I2S_CFG1_MAINENABLE_MASK)
        {
            /* Wait until the WS signal is low */
            while((DEMO_I2S_TX->STAT & I2S_STAT_LR_MASK))
            {
            }

            /* Wait until the rising edge of WS occurs. The execution of HAL_AudioTransferSendNonBlocking function takes 30.2us when core clock is 96MHz
             * and the enabling of I2S TX DMA request occurs at 20-30us. .
             * The time when WS is low/high is 10.4us, so the TX DMA request is enabled when WS is high.
            * The first data in the FIFO will be sent on the next falling edge of WS and there is no sample shift.*/
            while(!(DEMO_I2S_TX->STAT & I2S_STAT_LR_MASK))
            {

            }
        }

        HAL_AudioTransferSendNonBlocking((hal_audio_handle_t)&audioTxHandle[0], &s_TxTransfer);
        AUDIO_LatencyDebug_SendingAudioToI2s(true);
        AUDIO_LatencyDebug_SendingAudioToI2s(false);
        
        /* prepare the second transmission */
        audio_GetAndTransmitSamples(nbSamples);
        HAL_AudioTransferSendNonBlocking((hal_audio_handle_t)&audioTxHandle[0], &s_TxTransfer);
        

        /** set the internal flags */
        AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_TX_STARTED_MASK);
        AUDIO_SERVICE_CLEAR_STATUS(AUDIO_SERVICE_TX_COMPLETE_MASK);
    }
}


void audio_StopTx(void)
{
    critical_Enter();
    g_I2sTxStarted = 0;
    g_txRestartTimes ++;
    
    /* Disable the TX DMA */
    dma_AbortI2sDmaTransfer(DEMO_I2S_TX, (i2s_dma_handle_t *)((uint8_t *)&audioTxHandle[0] + 8));

    /** resets the audio buffer address to start*/
    audio_ringbuffer_Reset(&s_audioService_RingBufferOut);

    /** Resets the AudioService status */
    AUDIO_SERVICE_CLEAR_STATUS(AUDIO_SERVICE_TX_STARTED_MASK);
    AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_TX_COMPLETE_MASK);

    critical_Exit();
}


void audio_RingbufferTxInit(void)
{
    uint8_t audioTxSampleSizeBytes = audio_GetTxSampleSizeBytes();
    s_audioPlayTransferSize = AUDIO_GetTxDMATransferSize();
    /** Initialize the ring buffer for tx */
    audio_ringbuffer_Init(
        &s_audioService_RingBufferOut,
        (uint8_t *)&s_audioService_BufferOut[0],
        sizeof(s_audioService_BufferOut));
    /** Configure a warning callback when exceeding the ring buffer filling margins */
    audio_ringbuffer_ConfigureFillingWarning(
        &s_audioService_RingBufferOut,
        s_audioPlayTransferSize,
        audio_TxSpaceWarningCb,
        NULL);
                
    AUDIO_SERVICE_CLEAR_STATUS(AUDIO_SERVICE_TX_STARTED_MASK);
    AUDIO_SERVICE_SET_STATUS(AUDIO_SERVICE_TX_COMPLETE_MASK);

}

void audio_ResetTx(void)
{
}

void I2S0_IRQHandler(void)
{
}



uint8_t audio_GetTxSampleSizeBytes(void)
{
    return I2S_TX_WORD_LENGTH_BYTES;
}

void dma_AbortI2sDmaTransfer(I2S_Type *base, i2s_dma_handle_t *handle)
{

    assert(handle != NULL);
    assert(handle->dmaHandle != NULL);
    DMA_Type *dmaBase;
    uint32_t channel;

    dmaBase = handle->dmaHandle->base;
    channel = handle->dmaHandle->channel;
    assert(handle->dmaHandle->channel < (uint32_t)FSL_FEATURE_DMA_NUMBER_OF_CHANNELSn(dmaBase));
    /* Disable DMA interrupt */
    DMA_COMMON_REG_GET(dmaBase, channel, INTENCLR) |= 1UL << DMA_CHANNEL_INDEX(base,channel);

    /* Abort operation */
    DMA_AbortTransfer(handle->dmaHandle);


    /* kI2S_DmaStateTx */
    if (handle->state == (uint32_t)1)
    {
        /* Wait until all transmitted data get out of FIFO */
        while ((base->FIFOSTAT & I2S_FIFOSTAT_TXEMPTY_MASK) == 0U)
        {
        }
        /* The last piece of valid data can be still being transmitted from I2S at this moment */

        /* Write additional data to FIFO */
        base->FIFOWR = 0U;
        while ((base->FIFOSTAT & I2S_FIFOSTAT_TXEMPTY_MASK) == 0U)
        {
        }
        /* At this moment the additional data are out of FIFO, starting being transmitted.
         * This means the preceding valid data has been just transmitted and we can stop I2S. */
       //I2S_TxEnableDMA(base, false);
        base->FIFOCFG &= (~I2S_FIFOCFG_DMATX_MASK);
        base->FIFOCFG |= I2S_FIFOCFG_EMPTYTX_MASK;
    }

    /* bit clock and WS keep running */
    //I2S_Disable(base);

    /* Reset state, kI2S_DmaStateIdle */
    handle->state = (uint32_t)0;

    /* Clear transfer queue */
    (void)memset((void *)&(handle->i2sQueue), 0, sizeof(handle->i2sQueue));
    handle->queueDriver = 0U;
    handle->queueUser   = 0U;

}

uint32_t AUDIO_GetTxDMATransferSize(void)
{
    /* I2S TX word length is 32 bits. */
    return (g_composite.audioUnified.audioPlayTransferSize / AUDIO_TX_FORMAT_BYTES) * I2S_TX_WORD_LENGTH_BYTES;
}

uint8_t DMA_GetValidTransfer(i2s_dma_handle_t *handle)
{
    uint8_t validTransferNum = 0;
    assert(handle != NULL);
    assert(handle->dmaHandle != NULL);

    if(handle->queueUser >= handle->queueDriver)
    {
        validTransferNum = handle->queueUser - handle->queueDriver;
    }
    else
    {
        validTransferNum = handle->queueUser + 4 - handle->queueDriver;
    }


    return validTransferNum;

}
/*************************************************************************************/

/** @} */
