/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017, 2022, 2024 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_irtc.h"
#include "fsl_power.h"
#include "fsl_itrc.h"
#include "fsl_puf_v3.h"
#include "fsl_common.h"
#include "fsl_iocon.h"
#include "pin_mux.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define ITRC ITRC0
#define SPEW 1
#define UINT_MAX 4294967295

/* Core Mode Controller register  */

char *ELS_AS_FLAG0_REG[]= {"FLAG_AP_ENABLE_CPU0","Reserved1","Reserved2","Reserved3",
			"FLAG_BOD_VBAT_RESET_OCCURED","FLAG_BOD_CORE_RESET_OCCURED","FLAG_BOD_VBAT_IRQ_OCCURED","FLAG_BOD_CORE_IRQ_OCCURED",
			"FLAG_WDT_RESET_OCCURED","FLAG_CWDT_RESET_OCCURED","FLAG_WDT_IRQ_OCCURED","FLAG_CWDT_IRQ_OCCURED",
			"FLAG_QK_ERROR","FLAG_ELS_GLITCH_DETECTED","Reserved14","FLAG_TAMPER_EVENT_DETECTED",
			"FLAG_FLASH_ECC_INVALID","FLAG_SEC_VIOL_IRQ_OCURRED","FLAG_CPU0_NS_C_ACC_OCCURED","FLAG_CPU0_NS_D_ACC_OCCURED",
			"Reserved20","Reserved21","Reserved22","Reserved23",
			"Reserved24","Reserved25","Reserved26","Reserved27",
			"Reserved28","Reserved29","Reserved30","Reserved31"

};
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void BOD_levelsetup(void);
void RTC_setup(void);
/*******************************************************************************
 * Variables
 ******************************************************************************/
unsigned int g_rtc_status = 0;
/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief ISR for Tamper interrupt
 *
 * This function change state of busyWait.
 */

//ITRC IRQ Handler
void ITRC0_DriverIRQHandler(void)
{
	NVIC_DisableIRQ(ITRC0_IRQn);
	PRINTF("\r\nITRC IRQ Reached!\r\n");


	PRINTF("Clear ITRC IRQ STATUS\r\n");
    if (ITRC_ClearAllStatus(ITRC) != kStatus_Success)
	{
		PRINTF("Error while ITRC STATUS Clear.\r\n");
	}
	NVIC_ClearPendingIRQ(ITRC0_IRQn);
}


void WDT_BOD_IRQHandler(void)
{
	NVIC_DisableIRQ(WDT_BOD_IRQn);
	NVIC_ClearPendingIRQ(WDT_BOD_IRQn);
	PRINTF("\n\rIRQ WWDT\n\r");
    SDK_ISR_EXIT_BARRIER;
}

/*Configure BOD Reset for Time stamping retention in VBAT mode*/
void BOD_levelsetup(void)
{
    PMC->RESETCTRL &= ~(PMC_RESETCTRL_BODVDDMAINRESETENA_SECURE_DP_MASK | PMC_RESETCTRL_BODVDDMAINRESETENA_SECURE_MASK |
                        PMC_RESETCTRL_BODCORERESETENA_SECURE_MASK | PMC_RESETCTRL_BODCORERESETENA_SECURE_DP_MASK);
    /* disable BOD reset, BOD reset will cause RTC to not retain counting value */
    PMC->RESETCTRL |= PMC_RESETCTRL_BODVDDMAINRESETENA_SECURE_DP(2) | PMC_RESETCTRL_BODVDDMAINRESETENA_SECURE(2) |
                      PMC_RESETCTRL_BODCORERESETENA_SECURE(2) | PMC_RESETCTRL_BODCORERESETENA_SECURE_DP(2);

    ANACTRL->BOD_DCDC_INT_CTRL |= (ANACTRL_BOD_DCDC_INT_CTRL_BODCORE_INT_ENABLE_MASK | ANACTRL_BOD_DCDC_INT_CTRL_BODVBAT_INT_ENABLE_MASK);
    PMC->BODVDDMAIN = PMC_BODVDDMAIN_HYST(0) | PMC_BODVDDMAIN_TRIGLVL(0x14);     /* 2.8V */
    PMC->BODCORE = PMC_BODCORE_HYST(0) | PMC_BODCORE_TRIGLVL(0x6);               /* 0.9V - loweest */

    /*
     * Enable the BOD ISR, once triggers can take actions such as execute deep power down
     */
}

void RTC_setup(void)
{
    /* If Always on register is not program, init the RTC */
    if ((RTC->GP_DATA_REG & 2) == 0)
    {
        irtc_datetime_t date;
        irtc_config_t irtcConfig;

        /* Set a start date time and start RTC */
        date.year    = 2024U;
        date.month   = 8U;
        date.day     = 31U;
        date.weekDay = 4U;
        date.hour    = 18U;
        date.minute  = 01;
        date.second  = 0;

        /* Enable 32Khz crystal */
        CLOCK_AttachClk(kXTAL32K_to_OSC32K);


        IRTC_GetDefaultConfig(&irtcConfig);

        /* Init RTC */
        IRTC_Init(RTC, &irtcConfig);
        PRINTF("Init RTC\r\n");

        /* Enable the RTC 32KHz oscillator at CFG0 by writing a 0 */
        IRTC_Enable32kClkDuringRegisterWrite(RTC, true);

        if(RTC->STATUS & 0x8 == 0){
			/* Clear all Tamper events by writing a 1 to the bits */
			IRTC_ClearTamperStatusFlag(RTC);
        }
        IRTC_SetDatetime(RTC, &date);
        RTC->GP_DATA_REG |= 2;

        __NVIC_ClearPendingIRQ(WDT_BOD_IRQn);
        BOD_levelsetup();
    }
}


/*!
 * @brief Main function
 */
int main(void)
{
	irtc_datetime_t Datetime;
	RTC_Type top;
	irtc_tamper_config_t TamperConfig;
    irtc_tamper_config_t *irtcTamperConfig = &TamperConfig;

    /* power on XTAL32K and select XTAL32K as clock source of RTC */
    POWER_DisablePD(kPDRUNCFG_PD_XTAL32K);
    CLOCK_AttachClk(kXTAL32K_to_OSC32K);

    SYSCON->AHBCLKCTRL0 |= SYSCON_AHBCLKCTRL0_RTC_MASK;
    // Check whether its a POR exit or return from VBAT mode and accordingly disable BOD
    if (RTC->GP_DATA_REG & 2)
    {
        BOD_levelsetup();
    }

    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

//---------------------------------------------------------------------------
	RTC_setup();

    if (IRTC_GetStatusFlags(RTC)) {
    	/* Unlock to allow register write operation */
		IRTC_SetWriteProtection(RTC, false);
    }

//---------------------------------------------------------------------------

    /* Board pin, clock, debug console init */
    BOARD_InitPins();
    //BOARD_BootClockPLL100M();
    BOARD_InitDebugConsole();
    PRINTF("\r\nPress any key to Start.\r\n");
    GETCHAR();
	/* Enable ITRC IRQ */
	if (ITRC_Init(ITRC) != kStatus_Success)
	{
		PRINTF("Error while ITRC Init.\r\n");
		return 1;
	}

    PRINTF("\n\r.................................................................\r\n");
    PRINTF("             RTC Time stamping Example start:              ");
    PRINTF("\n\r.................................................................\r\n");

    if (ITRC_ClearAllStatus(ITRC) != kStatus_Success)
	{
		PRINTF("Error while ITRC STATUS Clear.\r\n");
	}

	//Check whether Its a POR exit or a return from VBAT mode
	if (RTC->STATUS & 0x8) {
		PRINTF("\n\r***POR Exit***\n\r");
	}
	else {
		PRINTF("\n\r***STANDBY Mode Exit***\n\r");
	}

    for (;;)
    {
    	IRTC_GetDatetime(RTC, &Datetime);
    	PRINTF("\r\nPress any key to check time.\r\n");
    	GETCHAR();
        PRINTF("\r\nCurrent TimeStamp is %d/%d/%d %d:%d:%2d\r\n", 2024, Datetime.month, Datetime.day,
        Datetime.hour, Datetime.minute, Datetime.second);
    }
}
