/*
 * Copyright 2022-2023 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "fsl_flash.h"
#include "fsl_flash_ffr.h"
#include "fsl_common.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"

#include "fsl_clock.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define REG_SET1  1
#define REG_SET2  2
#define REG_UNSET 0

//define flash command values
#define RD1ALL 		0x00 //read 1s all
#define RD1BLK 		0x01 //read 1s block
#define RD1SCR 		0x02 //read 1s sector
#define RD1PG 		0x03 //read 1s page
#define RD1PHR 		0x04 //read 1s phrase
#define RDMISR		0x05 //read into MISR
#define RD1ISCR		0x12 //read 1s IFR sector
#define RD1IPG		0x13 //read 1s IFR page
#define RD1IPHR		0x14 //read 1s IFR phrase
#define RDIMISR		0x15 //read IFR into MISR
#define PGMPG		0x23 //program page
#define PGMPHR		0x24 //program phrase
#define ERSALL		0x40 //erase all
#define ERSSCR		0x42 //ease sector

//define FSTAT values
#define FSTAT_CLEARERR 		0x00000034
#define FSTAT_CLEARCCIF 	0x00000080
#define FSTAT_CLEARPERDY 	0x80000000

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void error_trap();
void wait_FSTAT(unsigned int mask, unsigned int value);
void error_handle();
void app_finalize(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
/*
 * @brief Gets called when an error occurs.
 *
 * @details Print error message and trap forever.
 */
void error_trap(void)
{
    PRINTF("\r\n\r\n\r\n\t---- HALTED DUE TO FLASH ERROR! ----");
    while (1)
    {
    }
}

/*
 * @brief gets called when waiting for command controller
 *
 * @details wait for specified register to be set to expected value
 */
void wait_FSTAT(unsigned int mask, unsigned int value)
{
	int count = 0;
	int timeout = 100000;
	while(1){
		count++;
		if (count == timeout)
		{
			PRINTF("\r\n Timout while waiting on command controller \r\n");
			error_trap();
		}
		switch (mask)
		{
			//CCIF set on command complete
			case FMU_FSTAT_CCIF_MASK:
				if (((FMU0->FSTAT & FMU_FSTAT_CCIF(1)) >> FMU_FSTAT_CCIF_SHIFT) == value)
				{
					return;
				}
				break;
			//PEWEN Set by controller to enable writes and cleared to disable
			case FMU_FSTAT_PEWEN_MASK:
				if (((FMU0->FSTAT & FMU_FSTAT_PEWEN(value)) >> FMU_FSTAT_PEWEN_SHIFT) == value)
				{
					return;
				}
				//if PEWN expected to be 1, but is not and CCIF == 1 then there is an access err
				//because CCIF 1 indicates that the command is completed
				else if (value == 1 &&((FMU0->FSTAT & FMU_FSTAT_CCIF(1)) >> FMU_FSTAT_CCIF_SHIFT) == 1)
				{
					PRINTF("\r\n Access Error \r\n");
					error_trap();
				}
				break;
			//PERDY set when by controller when ready to execute and cleared by user
			case FMU_FSTAT_PERDY_MASK:
				//PRINTF("\r\n Waiting for PERDY == %d \r\n", value);
				if (((FMU0->FSTAT & FMU_FSTAT_PERDY(1)) >> FMU_FSTAT_PERDY_SHIFT) == value)
				{
					return;
				}
				//if PERDY expected to be 1, but is not and CCIF == 1 then there is an access err
				//because CCIF 1 indicates that the command is completed
				else if (value == 1 && ((FMU0->FSTAT & FMU_FSTAT_CCIF(1)) >> FMU_FSTAT_CCIF_SHIFT) == 1)
				{
					PRINTF("\r\n Access Error \r\n");
					error_trap();
				}
				break;
		}

	}
}

/*
 * @brief Gets called after a flash command.
 *
 * @details Print error message and trap forever IF there is an error
 */
void error_handle()
{

	if (((FMU0->FSTAT & FMU_FSTAT_ACCERR(1)) >> FMU_FSTAT_ACCERR_SHIFT) == 1)
	{
		PRINTF("\r\n Access Error \r\n");
		error_trap();
	}
	else if (((FMU0->FSTAT & FMU_FSTAT_PVIOL(1)) >> FMU_FSTAT_PVIOL_SHIFT) == 1)
	{
		PRINTF("\r\n Protection Violation \r\n");
		error_trap();
	}
	else if (((FMU0->FSTAT & FMU_FSTAT_CMDABT(1)) >> FMU_FSTAT_CMDABT_SHIFT) == 1)
	{
		PRINTF("\r\n Operation Is Aborted \r\n");
	}
	else if(((FMU0->FSTAT & FMU_FSTAT_FAIL(1)) >> FMU_FSTAT_FAIL_SHIFT) == 1)
	{
	    PRINTF("\r\n Command Failed \r\n");
	    error_trap();
	}
}
/*
 * @brief Gets called when the app is complete.
 *
 * @details Print finished message and trap forever.
 */
void app_finalize(void)
{
    /* Print finished message. */
    PRINTF("\r\n End of Flash Programming Example! \r\n");
    while (1)
    {
    }
}

int main()
{
    uint32_t destAdrss; /* Address of the target location */
    //device specific flash properties
    uint32_t pflashBlockBase  = 0U;
    uint32_t pflashTotalSize  = 2097152U; //2MB
    uint32_t pflashSectorSize = 8192U;    //8KB
    uint32_t PflashPageSize   = 128U;     //128B
    uint32_t sector;

    /* Init board hardware. */
    /* attach FRO 12M to FLEXCOMM4 (debug console) */
    CLOCK_SetClkDiv(kCLOCK_DivFlexcom4Clk, 1u);
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    BOARD_InitPins();
    BOARD_InitBootClocks();
    BOARD_InitDebugConsole();

    //Enable 12MHz FRO for flash
    SYSCON0->CLOCK_CTRL |= SYSCON_CLOCK_CTRL_FRO12MHZ_ENA_MASK;

    //start in the second half of flash
    uint32_t startAdrss = pflashBlockBase + 0x100000;
    uint32_t endAdrss = 0x1FFFFF;
    //need to write 2048 512 byte sections
    PRINTF("Flash Command Erase / Programming example:\n\r");
    PRINTF("This application erases the flash area from 0x0010_0000 -> 0x001F_FFFF and then programs with 0x1234_5678.\n\r");
    PRINTF("Begin erase:  ");

    for (destAdrss = startAdrss; destAdrss < endAdrss; destAdrss += pflashSectorSize)
    {
    	//erase region
    	//FCCOB0 [7:0] contains flash command code
    	//wait for previous command complete
    	wait_FSTAT(FMU_FSTAT_CCIF_MASK, REG_SET1);
    	//clear previous errors
    	FMU0->FSTAT = 0x34;
    	//42h is erase sector command ERSSCR
    	//specify command
    	FMU0->FCCOB[0] = 0x42;
    	//clear ccif to launch
    	FMU0->FSTAT = 0x80;
    	//wait for pewen to be set
    	wait_FSTAT(FMU_FSTAT_PEWEN_MASK, REG_SET1);
    	//the first write must be sector aligned but contents do not matter
    	*(volatile uint32_t *)(destAdrss) = 0x0;
    	*(volatile uint32_t *)(destAdrss + 4) = 0x0;
    	*(volatile uint32_t *)(destAdrss + 8) = 0x0;
    	*(volatile uint32_t *)(destAdrss + 12) = 0x0;
    	//wait for PERDY to be set
    	wait_FSTAT(FMU_FSTAT_PERDY_MASK, REG_SET1);
    	//controller should erase AND verify after we clear PERDY
    	FMU0->FSTAT = 0x80000000;
    	//wait for previous command complete
    	wait_FSTAT(FMU_FSTAT_CCIF_MASK, REG_SET1);
    	//see if any error flags set
    	//check FSTAT registers
    	error_handle();

      	//Verify if the given flash range is successfully erased.
    	//clear previous errors
    	FMU0->FSTAT = FSTAT_CLEARERR;
    	//specify command and destination address at call time
    	FMU0->FCCOB[0] = RD1SCR;
    	FMU0->FCCOB[2] = destAdrss;
    	//clear ccif to launch
    	FMU0->FSTAT = FSTAT_CLEARCCIF;
    	wait_FSTAT(FMU_FSTAT_CCIF_MASK, REG_SET1);
    	//error handling for read 1s sector command
    	error_handle();
    }
    PRINTF(" Success!\n\r");

    PRINTF("Begin Program:");
    /* Now program aread */
    for (destAdrss = startAdrss; destAdrss < endAdrss; destAdrss += pflashSectorSize)
    {
    	//can fit  64 128 byte pages into one sector of 8192 bytes 8192/128 = 64
        for (int index = 0; index < pflashSectorSize; index += PflashPageSize)
        {
        	//two programming options - program page or program phrase
        	//program page writes 32 words of data
        	//each "word" is 32 bits or 4 bytes
        	// 4 bytes * 32 = 128 bytes or one "page" of size 0x80
        	//program phrase writes a much smaller section of 4 words
      
        	//wait for previous command complete
        	wait_FSTAT(FMU_FSTAT_CCIF_MASK, REG_SET1);
        	//clear previous errors
        	FMU0->FSTAT = FSTAT_CLEARERR;
        	//only need to specify command at call time
        	FMU0->FCCOB[0] = PGMPG;
        	//clear CCIF to launch
        	FMU0->FSTAT = FSTAT_CLEARCCIF;
        	//wait for PEWEN == 2
        	wait_FSTAT(FMU_FSTAT_PEWEN_MASK, REG_SET2);
        	//write 32 consecutive words to flash space
        	//one word = 4 bytes
        	for (int i = 0; i < 32; i++)
        	{
        		*(volatile uint32_t *)(destAdrss + index + (i*4)) = 0x12345678;
        	}
        	//wait for PERDY set
        	wait_FSTAT(FMU_FSTAT_PERDY_MASK, REG_SET1);
        	//clear PERDY
        	FMU0->FSTAT = FSTAT_CLEARPERDY;
        	//controller should program and verify
        	error_handle();

        	//check write
        	/* Verify if the given flash region is successfully programmed with given data */
        	//wait for previous command complete
        	wait_FSTAT(FMU_FSTAT_CCIF_MASK, REG_SET1);

        }
        /* Verify programming by reading back from flash directly */
        //read each word in sector
        for (uint32_t i = 0; i < (pflashSectorSize/ 4); i++)
        {
        		uint32_t buf1 = 0x12345678;
        		uint32_t buf2 = *(volatile uint32_t *)(destAdrss + (i * 4));
        		if (buf1 != buf2)
        	{
        		PRINTF("\r\n Word Failed Verification \r\n");
        		error_trap();
        	}
        }
    }
    PRINTF(" Success!\n\r");

    app_finalize();

    return 0;
}
