/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017, 2023-2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "fsl_flexio_qspi_smartdma.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_inputmux.h"
#include "fsl_gpio.h"
#include "app.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CMD_WRITE_ENABLE            0x06
#define CMD_READ_VOLATILE_REGISTER  0x85
#define CMD_WRITE_VOLATILE_REGISTER 0x81
#define CMD_READ                    0x03
#define CMD_QUAD_FAST_READ          0x6B
#define VAL_DUMMY_CYCLE_REGISTER    0x8B

#define READ_ADR                    0x000000

#define COMMAND_CYCLE 8
#define ADDRESS_CYCLE 24
#define DUMMY_CYCLE   8
#define READ_CYCLE    128
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* SPI user callback */
void FLEXIO_QSPI_MasterUserCallback(FLEXIO_QSPI_Type *base,
                                    flexio_qspi_smartdma_handle_t *handle,
                                    status_t status,
                                    void *userData);
/*******************************************************************************
 * Variables
 ******************************************************************************/
AT_NONCACHEABLE_SECTION_ALIGN(uint8_t header[(COMMAND_CYCLE+ADDRESS_CYCLE+DUMMY_CYCLE)/2], 4);
AT_NONCACHEABLE_SECTION_ALIGN(uint8_t body[READ_CYCLE/2], 4);

#define TRANSFER_SIZE ((sizeof(header) + sizeof(body)) / 4)    /*! Transfer dataSize */

flexio_qspi_smartdma_handle_t g_m_handle;

volatile bool isTransferCompleted = false;

/*******************************************************************************
 * Code
 ******************************************************************************/
static inline void setNibbleByteSwappedBit(uint8_t* buf, uint8_t data, uint8_t data_line, uint8_t nbit)
{
    #define BITSET(byte, bit, nbit)   ((byte) = ((byte) & ~(1<<(nbit))) | ((bit)<<(nbit)))
    BITSET(buf[3 - (nbit/2)], (data>>nbit) & 0x1 , data_line + 4*(nbit%2));
}

static void setData(uint8_t* buf, uint8_t data, uint8_t data_line)
{
    setNibbleByteSwappedBit(buf, data, data_line, 0);
    setNibbleByteSwappedBit(buf, data, data_line, 1);
    setNibbleByteSwappedBit(buf, data, data_line, 2);
    setNibbleByteSwappedBit(buf, data, data_line, 3);
    setNibbleByteSwappedBit(buf, data, data_line, 4);
    setNibbleByteSwappedBit(buf, data, data_line, 5);
    setNibbleByteSwappedBit(buf, data, data_line, 6);
    setNibbleByteSwappedBit(buf, data, data_line, 7);
}

static void setCmd(uint8_t *header, uint8_t cmd)
{
    setData(&header[0], cmd, 0);
    setData(&header[0], 0xFF, 3);
}

static void setAddress(uint8_t *header, uint32_t address)
{
    setData(&header[4] , address >> 16 & 0xff, 0);
    setData(&header[8] , address >> 8  & 0xff, 0);
    setData(&header[12], address       & 0xff, 0);
}

void FLEXIO_QSPI_MasterUserCallback(FLEXIO_QSPI_Type *base,
                                    flexio_qspi_smartdma_handle_t *handle,
                                    status_t status,
                                    void *userData)
{
    if (status == kStatus_Success)
    {
        __NOP();
    }
    PRINTF("This is QSPI Master call back.\r\n");
    isTransferCompleted = true;
}

uint32_t show_buf(uint8_t *buf, uint32_t len)
{
    for(uint32_t i = 0; i < len; i++)
    {
        if((i) % 16 == 0)
        {
            PRINTF("0x%04x: ", i);
        }
        PRINTF("%02x ", buf[i]);
        if((i+1) % 16 == 0)
        {
            PRINTF("\r\n");
        }
    }
    PRINTF("\r\n");
}

int main(void)
{
    /* Init board hardware. */
    BOARD_InitHardware();

    PRINTF("FlexIO emulating half-duplex QSPI with SmartDMA demo starts.\r\n");

    flexio_spi_master_config_t masterConfig;
    flexio_qspi_transfer_t masterXferQspi;

    FLEXIO_QSPI_Type qspiDev = {
        .flexioBase      = BOARD_FLEXIO_BASE,
        .SDOPinIndex     = FLEXIO_QSPI_MOSI_PIN,
        .SCKPinIndex     = FLEXIO_QSPI_SCK_PIN,
        .gpioBase        = CS_GPIO_BASE,
        .csGpioPort      = CS_GPIO_PORT,
        .csGpioPin       = CS_GPIO_PIN,
    };
    
    /* Init INPUTMUX */
    RESET_PeripheralReset(kINPUTMUX_RST_SHIFT_RSTn);
    INPUTMUX_Init(INPUTMUX);
    INPUTMUX_AttachSignal(INPUTMUX, 0, kINPUTMUX_FlexioIrqToSmartDmaInput);
    /* Turnoff clock to inputmux to save power. Clock is only needed to make changes */
    INPUTMUX_Deinit(INPUTMUX);

    /*Start master transfer*/
    masterXferQspi.bufs[0].txData   = header;
    masterXferQspi.bufs[0].rxData   = NULL;
    masterXferQspi.bufs[0].dataSize = sizeof(header)/4;
    masterXferQspi.bufs[1].txData   = NULL;
    masterXferQspi.bufs[1].rxData   = body;
    masterXferQspi.bufs[1].dataSize = sizeof(body)/4;

    /* Set up master transfer */
    FLEXIO_QSPI_MasterGetDefaultConfig(&masterConfig);
    masterConfig.baudRate_Bps = TRANSFER_BAUDRATE;
    FLEXIO_QSPI_MasterInit(&qspiDev, &masterConfig, FLEXIO_CLOCK_FREQUENCY);
    FLEXIO_QSPI_TransferCreateHandleSMARTDMA(&qspiDev, &g_m_handle, FLEXIO_QSPI_MasterUserCallback, NULL);

    while (1)
    {
        isTransferCompleted = false;
        memset(header, 0, sizeof(header));
        setCmd(header, CMD_WRITE_ENABLE);
        masterXferQspi.bufs[0].dataSize = 1;
        masterXferQspi.bufs[1].dataSize = 0;
        FLEXIO_QSPI_TransferSMARTDMA(&qspiDev, &g_m_handle, &masterXferQspi);
        while (!isTransferCompleted);

        isTransferCompleted = false;
        memset(header, 0, sizeof(header));
        setCmd(header, CMD_WRITE_VOLATILE_REGISTER);
        setCmd(&header[4], VAL_DUMMY_CYCLE_REGISTER);
        masterXferQspi.bufs[0].dataSize = 2;
        masterXferQspi.bufs[1].dataSize = 0;
        FLEXIO_QSPI_TransferSMARTDMA(&qspiDev, &g_m_handle, &masterXferQspi);
        while (!isTransferCompleted);

        isTransferCompleted = false;
        memset(header, 0, sizeof(header));
        setCmd(header, CMD_READ_VOLATILE_REGISTER);
        masterXferQspi.bufs[0].dataSize = 1;
        masterXferQspi.bufs[1].dataSize = 1;
        FLEXIO_QSPI_TransferSMARTDMA(&qspiDev, &g_m_handle, &masterXferQspi);
        while (!isTransferCompleted);

        isTransferCompleted = false;
        memset(header, 0, sizeof(header));
        memset(body, 0, sizeof(body));
        setCmd(header, CMD_QUAD_FAST_READ);
        setAddress(header, READ_ADR);
        masterXferQspi.bufs[0].dataSize = sizeof(header)/4;
        masterXferQspi.bufs[1].dataSize = sizeof(body)/4;
        FLEXIO_QSPI_TransferSMARTDMA(&qspiDev, &g_m_handle, &masterXferQspi);
        while (!isTransferCompleted);

        show_buf(body, sizeof(body));
        /* Wait for press any key */
        PRINTF("\r\n Press any key to run again\r\n\r\n");
        GETCHAR();
    }

}
