/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2020 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/

#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_ctimer.h"

#include "fsl_plu.h"

#include <stdbool.h>
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CTIMER          CTIMER0         /* Timer 0 */
#define CTIMER_MAT_OUT1  kCTIMER_Match_1 /* Match output 1 */

#define CTIMER_MAT_OUT2  kCTIMER_Match_2 /* Match output 2 */

#define CTIMER_CLK_FREQ CLOCK_GetFreq(kCLOCK_CoreSysClk)
#ifndef CTIMER_MAT_PWM_PERIOD_CHANNEL
#define CTIMER_MAT_PWM_PERIOD_CHANNEL kCTIMER_Match_3
#endif


/* PLU module */
#define DEMO_PLU_BASE         PLU

//#define DEMO_PLU_LUT_IN_SRC_0 kPLU_LUT_IN_SRC_PLU_IN_2   /* =2 */
//#define DEMO_PLU_LUT_IN_SRC_1 kPLU_LUT_IN_SRC_PLU_IN_3   /* =3 */

#define DEMO_PLU_LUT_IN_SRC_0 			kPLU_LUT_IN_SRC_PLU_IN_4   /* Using PLU IN4(P0_12) as LUT IN0  */
#define DEMO_PLU_LUT_IN_SRC_1 			kPLU_LUT_IN_SRC_PLU_IN_5   /* Using PLU IN5(P0_13) as LUT IN1 */

#define DEMO_PLU_LUT_OUT_SRC_0     	kPLU_OUT_SRC_LUT_0    		 /* Using LUT0 OUT as PLU OUT0(P0_14) */
#define DEMO_PLU_LUT_0_TRUTH_TABLE 	0x0000004             		 /* 0b00000110 */



/*******************************************************************************
 * Prototypes
 ******************************************************************************/

static void PLU_Configuration(void);


/*******************************************************************************
 * Variables
 ******************************************************************************/
volatile uint32_t g_pwmPeriod1   = 0U;
volatile uint32_t g_pulsePeriod1 = 0U;

volatile uint32_t g_pwmPeriod2   = 0U;
volatile uint32_t g_pulsePeriod2 = 0U;



/*******************************************************************************
 * Code
 ******************************************************************************/
status_t CTIMER_GetPwmPeriodValue1(uint32_t pwmFreqHz, uint8_t dutyCyclePercent, uint32_t timerClock_Hz)
{
    /* Calculate PWM period match value */
    g_pwmPeriod1 = (timerClock_Hz / pwmFreqHz) - 1U;

    /* Calculate pulse width match value */
    g_pulsePeriod1 = (g_pwmPeriod1 + 1U) * (100 - dutyCyclePercent) / 100;

    return kStatus_Success;
}


status_t CTIMER_GetPwmPeriodValue2(uint32_t pwmFreqHz, uint8_t dutyCyclePercent, uint32_t timerClock_Hz)
{
    /* Calculate PWM period match value */
    g_pwmPeriod2 = (timerClock_Hz / pwmFreqHz) - 1U;

    /* Calculate pulse width match value */
    g_pulsePeriod2 = (g_pwmPeriod2 + 1U) * (100 - dutyCyclePercent) / 100;

    return kStatus_Success;
}


/*!
 * @brief Configure the PLU combinational logic network
 */
static void PLU_Configuration(void)
{
    /* Set two input sources to LUT0. */
    PLU_SetLutInputSource(DEMO_PLU_BASE, kPLU_LUT_0, kPLU_LUT_IN_0, DEMO_PLU_LUT_IN_SRC_0);
    PLU_SetLutInputSource(DEMO_PLU_BASE, kPLU_LUT_0, kPLU_LUT_IN_1, DEMO_PLU_LUT_IN_SRC_1);

    /* Set truthtable for LUTO. */
    PLU_SetLutTruthTable(DEMO_PLU_BASE, kPLU_LUT_0, DEMO_PLU_LUT_0_TRUTH_TABLE);
    /* Set LUT0 output to output source 0. */
    PLU_SetOutputSource(DEMO_PLU_BASE, kPLU_OUTPUT_0, DEMO_PLU_LUT_OUT_SRC_0);

}



/*!
 * @brief Main function
 */
int main(void)
{
    ctimer_config_t config;
    uint32_t srcClock_Hz;
    uint32_t timerClock;

    /* Init hardware*/
    /* Attach 12 MHz clock to USART0 (debug console) */
    CLOCK_Select(BOARD_DEBUG_USART_CLK_ATTACH);

    BOARD_InitBootPins();
    BOARD_InitBootClocks();
    BOARD_InitDebugConsole();
	
	
    /* Init plu module */
    PLU_Init(DEMO_PLU_BASE);

	  /* Configure input, output, truthtable one time through the API */
    PLU_Configuration();

    /* Once the PLU module is configured, the PLU bus clock can be shut-off to conserve power */
    PLU_Deinit(DEMO_PLU_BASE);
	
	

    /* CTimer0 counter uses the AHB clock, some CTimer1 modules use the Aysnc clock */
    srcClock_Hz = CTIMER_CLK_FREQ;

    PRINTF("CTimer example to generate a PWM signal\r\n");

    CTIMER_GetDefaultConfig(&config);
    timerClock = srcClock_Hz / (config.prescale + 1);

    CTIMER_Init(CTIMER, &config);

    /* Get the PWM period match value and pulse width match value of 1hz PWM signal with needed dutycycle */
    CTIMER_GetPwmPeriodValue1(1, 40, timerClock);
    CTIMER_SetupPwmPeriod(CTIMER, CTIMER_MAT_PWM_PERIOD_CHANNEL, CTIMER_MAT_OUT1, g_pwmPeriod1, g_pulsePeriod1, false);
		
    CTIMER_GetPwmPeriodValue2(1, 60, timerClock);
    CTIMER_SetupPwmPeriod(CTIMER, CTIMER_MAT_PWM_PERIOD_CHANNEL, CTIMER_MAT_OUT2, g_pwmPeriod2, g_pulsePeriod2, false);		
		
    CTIMER_StartTimer(CTIMER);

    while (1)
    {
    }
}
