/* -------------------------------------------------------------------------- */
/*                           Copyright 2021-2022 NXP                          */
/*                            All rights reserved.                            */
/*                    SPDX-License-Identifier: BSD-3-Clause                   */
/* -------------------------------------------------------------------------- */

#ifndef _FWK_PLATFORM_FLASH_H_
#define _FWK_PLATFORM_FLASH_H_

/*!
 * @addtogroup FWK_Platform_module
 * @{
 */
/*!
 * @addtogroup FWK_Platform_Flash
 * The FWK_Platform_Flash module
 *
 * FWK_Platform_Flash module provides APIs to handle internal/external flash
 * operations specific to the platform
 * @{
 */

/* -------------------------------------------------------------------------- */
/*                                  Includes                                  */
/* -------------------------------------------------------------------------- */

#include <stdint.h>

#include "fsl_common.h"

/* -------------------------------------------------------------------------- */
/*                                Public macros                               */
/* -------------------------------------------------------------------------- */

#define PLATFORM_EXTFLASH_SECTOR_SIZE 4096U
#define PLATFORM_EXTFLASH_PAGE_SIZE   256U

/* -------------------------------------------------------------------------- */
/*                              Public prototypes                             */
/* -------------------------------------------------------------------------- */

/*!
 * \brief Initialize platform specific ressources for external flash
 *
 * \param[in] baudrate baudrate to be used with external flash
 * \return status_t kStatus_Success on success, other on failure
 */
status_t PLATFORM_InitExternalFlash(uint32_t baudrate);

/*!
 * \brief Erase a range of external flash
 *
 * \param[in] address address to start erasing from
 * \param[in] size size to erase
 * \return status_t kStatus_Success on success, other on failure
 */
status_t PLATFORM_EraseExternalFlash(uint32_t address, uint32_t size);

/*!
 * \brief Read data from the external flash
 *
 * \param[out] dest output buffer where the data is stored, provided by the caller
 * \param[in] length size of the data to read
 * \param[in] address address to start reading from
 * \param[in] requestFastRead request fast read or not (if the feature is supported)
 * \return status_t kStatus_Success on success, other on failure
 */
status_t PLATFORM_ReadExternalFlash(uint8_t *dest, uint32_t length, uint32_t address, bool requestFastRead);

/*!
 * \brief Write data to the external flash
 *
 * \param[in] data input buffer containing the data to write to the external flash
 * \param[in] length size of the data to be written
 * \param[in] address address to start writing from
 * \return status_t kStatus_Success on success, other on failure
 */
status_t PLATFORM_WriteExternalFlash(uint8_t *data, uint32_t length, uint32_t address);

/*!
 * \brief Check busy status of the external flash
 *
 * \param[out] isBusy true for busy, false for idle
 * \return status_t kStatus_Success on success, other on failure
 */
status_t PLATFORM_IsExternalFlashBusy(bool *isBusy);

#endif /* _FWK_PLATFORM_FLASH_H_ */
