/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _DUALCORE_RINGBUFFER_OF_H_
#define _DUALCORE_RINGBUFFER_OF_H_

#include <stdint.h>

/*******************************************************************************
 * Definitions
 ******************************************************************************/
struct dcringbuf
{
    uint8_t *buf;
    uint32_t head;
    uint32_t tail;
    uint32_t size;
    uint32_t occ;
    uint32_t gate;
	uint32_t mode;
};

#define RINGBUF_MODE_HALF         1
#define RINGBUF_MODE_NORMAL       0

#define ARM_ADDRESS_TO_DSP_ADDRESS(addr) ((addr - 0x20000000) + 0x800000)
#define DSP_ADDRESS_TO_ARM_ADDRESS(addr) ((addr - 0x800000) + 0x20000000)

typedef struct dcringbuf dcringbuf_t;

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @brief    Create and initialize a ring buffer structure
 *
 * @param    size Buffer size to be malloc'd
 * @returns  Initialized and allocated pointer to ring buffer object
 */
dcringbuf_t *dc_ringbuf_create(uint8_t* mem, uint32_t size);

/*!
 * @brief    Destroy and free ring buffer structure
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 */
void dc_ringbuf_destroy(dcringbuf_t *rb);

/*!
 * @brief    Empty ring buffer structure
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 */
void dc_ringbuf_clear(dcringbuf_t *rb);

/*!
 * @brief    Return the number of bytes stored in the ringbuffer
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 * @returns  Number of bytes of data currently occupying space in the buffer
 */
uint32_t dc_ringbuf_get_occupancy(dcringbuf_t *rb);

/*!
 * @brief    Write data to ring buffer
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 * @param    data Input buffer of data to write to ring buffer
 * @param    size Number of bytes to write to ring buffer, from 'data'
 * @returns  Number of bytes successfully written to buffer
 */
uint32_t dc_ringbuf_write(dcringbuf_t *rb, uint8_t *data, uint32_t size);

/*!
 * @brief    Read data from ring buffer
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 * @param    data Output buffer of data to write from ring buffer.
 *                Can be NULL, in which case data is discarded.
 * @param    size Number of bytes to read from ring buffer, to store in 'data'
 * @returns  Number of bytes successfully read to buffer or discarded
 */
uint32_t dc_ringbuf_read(dcringbuf_t *rb, uint8_t *data, uint32_t size);

/*!
 * @brief    Peak data from ring buffer without removing it
 *
 * @param    rb Pointer to ring buffer object created with ringbuf_create
 * @param    data Output buffer of data to write from ring buffer.
 * @param    size Number of bytes to read from ring buffer, to store in 'data'
 * @returns  Number of bytes successfully read to buffer
 */
uint32_t dc_ringbuf_peek(dcringbuf_t *rb, uint8_t *data, uint32_t size);


void dc_ringbuf_setmode(dcringbuf_t *rb, uint8_t mode);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* _DUALCORE_RINGBUFFER_H_ */
