/*
 * Copyright (c) 2018-2021 Arm Limited. All rights reserved.
 * Copyright 2019-2021 NXP. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __FLASH_LAYOUT_H__
#define __FLASH_LAYOUT_H__

/* Flash layout on LPC55S36 (without BL2):
 *
 * 0x0000_0000 Primary image area(224 kB):
 *    0x0000_0000 Secure     image (96 kB) 
 *    0x0001_8000 Non-secure image (128 kB)
 * Reserved area:
 * 0xXXXX_XXXX Secure Binary tail Area (4 KB), if SB is used.
 * 0xXXXX_XXXX Internal Trusted Storage Area (3 KB)
 * 0xXXXX_XXXX Protected Storage Area (3 KB)
 * 0xXXXX_XXXX NV counters area (512 B)
 * 0xXXXX_XXXX Unused
 */

/* This header file is included from linker scatter file as well, where only a
 * limited C constructs are allowed. Therefore it is not possible to include
 * here the platform_base_address.h to access flash related defines. To resolve
 * this some of the values are redefined here with different names, these are
 * marked with comment.
 */

/* Size of a Secure and of a Non-secure image */
#define FLASH_S_PARTITION_SIZE               (0x18000)      /* S partition: 96 kB*/ 
#define FLASH_NS_PARTITION_SIZE              (0x20000)      /* NS partition: 128 kB*/

/* Sector size of the embedded flash hardware */
#define FLASH_AREA_IMAGE_SECTOR_SIZE        (512)           /* 512 B. Flash memory program/erase operations have a page granularity. */
#define FLASH_TOTAL_SIZE                    (0x00040000-(24*FLASH_AREA_IMAGE_SECTOR_SIZE)) /* 244 kB. The last 24 pages (12kB) are reserved for PFR on the 256KB flash. */

/* Sector size of the QSPI flash hardware */
#define QSPI_FLASH_AREA_IMAGE_SECTOR_SIZE   (0x1000)        /* 4 KB */
#define QSPI_FLASH_AREA_PROGRAM_SIZE        (256)
#define QSPI_FLASH_TOTAL_SIZE               (0x4000000)     /* 64 MB */

/* Flash layout info for BL2 bootloader */
#define FLASH_BASE_ADDRESS                  (0x00000000)



#ifdef SB_FILE /* Use signed Secure Binary (SB) image */
#define FLASH_SB_TAIL   0x1000 /* 4 KB */
#else
#define FLASH_SB_TAIL   0x0 /* 0 KB */
#endif

/* Secure + Non-secure image primary slot */
#define FLASH_AREA_0_ID            (1)
#define FLASH_AREA_0_OFFSET        (0x0)
#define FLASH_AREA_0_SIZE          (FLASH_S_PARTITION_SIZE + \
                                    FLASH_NS_PARTITION_SIZE + \
                                    FLASH_SB_TAIL)
                                    
/* Not used*/
#define FLASH_AREA_SCRATCH_ID      (FLASH_AREA_0_ID + 1)
#define FLASH_AREA_SCRATCH_OFFSET  (FLASH_AREA_0_OFFSET + FLASH_AREA_0_SIZE)
#define FLASH_AREA_SCRATCH_SIZE    (0)


/* Internal Trusted Storage (ITS) Service definitions */
#define FLASH_ITS_AREA_OFFSET           (FLASH_AREA_SCRATCH_OFFSET + \
                                         FLASH_AREA_SCRATCH_SIZE)
#define FLASH_ITS_AREA_SIZE             (0xC00) /* 3 KB */

/* NV Counters definitions */
#define FLASH_NV_COUNTERS_AREA_OFFSET   (FLASH_ITS_AREA_OFFSET + \
                                         FLASH_ITS_AREA_SIZE)
#define FLASH_NV_COUNTERS_AREA_SIZE     (FLASH_AREA_IMAGE_SECTOR_SIZE)

/* Offset and size definition in flash area used by assemble.py */
#define SECURE_IMAGE_OFFSET             (0x0)
#define SECURE_IMAGE_MAX_SIZE           FLASH_S_PARTITION_SIZE

#define NON_SECURE_IMAGE_OFFSET         (SECURE_IMAGE_OFFSET + \
                                         SECURE_IMAGE_MAX_SIZE)
#define NON_SECURE_IMAGE_MAX_SIZE       FLASH_NS_PARTITION_SIZE

/* Flash device name used by BL2
 * Name is defined in flash driver file: Driver_Flash.c
 */
#define FLASH_DEV_NAME Driver_EFLASH0

/* Protected Storage (PS) Service definitions
 */
#ifdef PS_QSPI_FS /* QSPI Flash */
    #define FLASH_PS_AREA_SIZE              (0x8000) /* 32 KB */
    
    #define TFM_HAL_PS_FLASH_DRIVER         Driver_QSPI_FLASH0

    /* Base address of dedicated flash area for PS */
    #define TFM_HAL_PS_FLASH_AREA_ADDR      0x0

    /* Number of physical erase sectors per logical FS block */
    #define TFM_HAL_PS_SECTORS_PER_BLOCK    1

    /* Smallest flash programmable unit in bytes */
    #define TFM_HAL_PS_PROGRAM_UNIT         QSPI_FLASH_AREA_PROGRAM_SIZE
    #define PS_FLASH_NAND_BUF_SIZE          (QSPI_FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                             TFM_HAL_PS_SECTORS_PER_BLOCK)
#else  /* On-chip Flash */
    #define FLASH_PS_AREA_SIZE              (0xC00) /* 3 KB */

    #define TFM_HAL_PS_FLASH_DRIVER         Driver_EFLASH0

    /* Base address of dedicated flash area for PS */
    #define TFM_HAL_PS_FLASH_AREA_ADDR      (FLASH_NV_COUNTERS_AREA_OFFSET + \
                                            FLASH_NV_COUNTERS_AREA_SIZE)

    /* Number of physical erase sectors per logical FS block */
    #define TFM_HAL_PS_SECTORS_PER_BLOCK    3

    /* Smallest flash programmable unit in bytes */
    #define TFM_HAL_PS_PROGRAM_UNIT         FLASH_AREA_IMAGE_SECTOR_SIZE
    #define PS_FLASH_NAND_BUF_SIZE          (FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                            TFM_HAL_PS_SECTORS_PER_BLOCK)
#endif

/* Size of dedicated flash area for PS */
#define TFM_HAL_PS_FLASH_AREA_SIZE      FLASH_PS_AREA_SIZE
#define PS_RAM_FS_SIZE                  TFM_HAL_PS_FLASH_AREA_SIZE

/* Internal Trusted Storage (ITS) Service definitions
 * Note: Further documentation of these definitions can be found in the
 * TF-M ITS Integration Guide. The ITS should be in the internal flash, but is
 * allocated in the external flash just for development platforms that don't
 * have internal flash available.
 */
#define TFM_HAL_ITS_FLASH_DRIVER Driver_EFLASH0

/* In this target the CMSIS driver requires only the offset from the base
 * address instead of the full memory address.
 */
/* Base address of dedicated flash area for ITS */
#define TFM_HAL_ITS_FLASH_AREA_ADDR    FLASH_ITS_AREA_OFFSET
/* Size of dedicated flash area for ITS */
#define TFM_HAL_ITS_FLASH_AREA_SIZE    FLASH_ITS_AREA_SIZE
#define ITS_RAM_FS_SIZE                TFM_HAL_ITS_FLASH_AREA_SIZE
/* Number of physical erase sectors per logical FS block */
#define TFM_HAL_ITS_SECTORS_PER_BLOCK   3

/* Smallest flash programmable unit in bytes */
#define TFM_HAL_ITS_PROGRAM_UNIT        FLASH_AREA_IMAGE_SECTOR_SIZE
#define ITS_FLASH_NAND_BUF_SIZE        (FLASH_AREA_IMAGE_SECTOR_SIZE * \
                                        TFM_HAL_ITS_SECTORS_PER_BLOCK)

/* NV Counters definitions */
#define TFM_NV_COUNTERS_AREA_ADDR    FLASH_NV_COUNTERS_AREA_OFFSET
#define TFM_NV_COUNTERS_AREA_SIZE    FLASH_NV_COUNTERS_AREA_SIZE
#define TFM_NV_COUNTERS_SECTOR_ADDR  FLASH_NV_COUNTERS_AREA_OFFSET
#define TFM_NV_COUNTERS_SECTOR_SIZE  FLASH_AREA_IMAGE_SECTOR_SIZE

/* Use Flash memory to store Code data */
#define S_ROM_ALIAS_BASE    (0x10000000)
#define NS_ROM_ALIAS_BASE   (0x00000000)

/* Use SRAM[0-4] memory to store RW data */
#define S_RAM_ALIAS_BASE    (0x30000000)
#define NS_RAM_ALIAS_BASE   (0x20000000)

#define TOTAL_ROM_SIZE 		FLASH_TOTAL_SIZE
#define TOTAL_RAM_SIZE      (0x0001C000)     /* 112 KB RAM for data (without 16KB SRAM X for code)*/

#endif /* __FLASH_LAYOUT_H__ */