/*--------------------------------------------------------------------------*/
/* Copyright 2020 NXP                                                       */
/*                                                                          */
/* NXP Confidential. This software is owned or controlled by NXP and may    */
/* only be used strictly in accordance with the applicable license terms.   */
/* By expressly accepting such terms or by downloading, installing,         */
/* activating and/or otherwise using the software, you are agreeing that    */
/* you have read, and that you agree to comply with and are bound by, such  */
/* license terms. If you do not agree to be bound by the applicable license */
/* terms, then you may not retain, install, activate or otherwise use the   */
/* software.                                                                */
/*--------------------------------------------------------------------------*/

/**
 * @file  common_get_info.c
 * @brief Example of version and configuration load functions.
 *
 * @example common_get_info.c
 * @brief   Example of version and configuration load functions.
 */

#include <mcuxCl_clns.h> // Test the CLNS component-independent functionality
#include <mcuxClCss.h> // Interface to the entire mcuxClCss component
#include <mcuxCsslFlowProtection.h> // Code flow protection
#include <stdbool.h>  // bool type for the example's return code

bool common_get_info(void)
{
    MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(result, token, mcuxClCss_Enable_Async()); // Enable the CSSv2.
    // mcuxClCss_Enable_Async is a flow-protected function: Check the protection token and the return value
    if((MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClCss_Enable_Async) != token) || (MCUXCLCSS_STATUS_OK_WAIT != result))
        return false;
    MCUX_CSSL_FP_FUNCTION_CALL_END();
    
    MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(result, token, mcuxClCss_WaitForOperation(MCUXCLCSS_ERROR_FLAGS_CLEAR)); // Wait for the mcuxClCss_Enable_Async operation to complete.
    // mcuxClCss_WaitForOperation is a flow-protected function: Check the protection token and the return value
    if((MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClCss_WaitForOperation) != token) || (MCUXCLCSS_STATUS_OK != result))
        return false;
    MCUX_CSSL_FP_FUNCTION_CALL_END();

    // Read the CSS hardware version.
    mcuxClCss_HwVersion_t hw_version;
    
    MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(result, token, mcuxClCss_GetHwVersion(&hw_version));
    // mcuxClCss_GetHwVersion is a flow-protected function: Check the protection token and the return value
    if((MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClCss_GetHwVersion) != token) || (MCUXCLCSS_STATUS_OK != result))
        return false;
    MCUX_CSSL_FP_FUNCTION_CALL_END();

    // Access and store hw_version struct elements
    uint32_t revision = hw_version.bits.revision;                                        // Extended revision version 
    uint32_t minor = hw_version.bits.minor;                                              // Minor version
    uint32_t major = hw_version.bits.major;                                              // Major version
    uint32_t level = hw_version.bits.level;                                              // Release level version

    // Read the CSS hardware configuration bitmap.
    mcuxClCss_HwConfig_t config;
    
    MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(result, token, mcuxClCss_GetHwConfig(&config));
    // mcuxClCss_GetHwConfig is a flow-protected function: Check the protection token and the return value
    if((MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClCss_GetHwConfig) != token) || (MCUXCLCSS_STATUS_OK != result))
        return false;
    MCUX_CSSL_FP_FUNCTION_CALL_END();

    // Access and store config struct elements
    uint32_t ciphersup = config.bits.ciphersup;                                          // cipher command not supported
    uint32_t authciphersup = config.bits.authciphersup;                                  // auth_cipher command not supported
    uint32_t ecsignsup = config.bits.ecsignsup;                                          // ecsign command not supported
    uint32_t ecvfysup = config.bits.ecvfysup;                                            // ecvfy command not supported
    uint32_t eckxchsup = config.bits.eckxchsup;                                          // dhkey_xch command is supported
    uint32_t keygensup = config.bits.keygensup;                                          // keygen command not supported
    uint32_t keyinsup = config.bits.keyinsup;                                            // keyin command not supported
    uint32_t keyoutsup = config.bits.keyoutsup;                                          // keyout command  not supported
    uint32_t kdeletesup = config.bits.kdeletesup;                                        // kdelete command not supported
    uint32_t ckdfsup = config.bits.ckdfsup;                                              // ckdf command not supported
    uint32_t hkdfsup = config.bits.hkdfsup;                                              // hkdf command not supported
    uint32_t tlsinitsup = config.bits.tlsinitsup;                                        // tls_init command not supported
    uint32_t hashsup = config.bits.hashsup;                                              // hash command not supported
    uint32_t hmacsup = config.bits.hmacsup;                                              // hmac command not supported
    uint32_t drbgreqsub = config.bits.drbgreqsub;                                        // drbg_req command not supported
    uint32_t dtrgncfgloadsup = config.bits.dtrgncfgloadsup;                              // dtrng_cfg_load command is not supported
    uint32_t dtrngevalsup = config.bits.dtrngevalsup;                                    // dtrng_eval command not supported

    char const* sw_version = mcuxCl_GetVersion();                                         // Read the CLNS version string that uniquely identifies this release of the CLNS.
                                                                                         // MCUXCL_VERSION_MAX_SIZE is the maximum size in bytes of the version string.

    MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(result, token, mcuxClCss_Disable()); // Disable the CSSv2.
    // mcuxClCss_Disable is a flow-protected function: Check the protection token and the return value
    if((MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClCss_Disable) != token) || (MCUXCLCSS_STATUS_OK != result))
        return false;
    MCUX_CSSL_FP_FUNCTION_CALL_END();
    
    return true;
}

