/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "osa_common.h"

#include "app_definitions.h"
#include "board.h"
#include "streamer_pcm.h"
#include "fsl_debug_console.h"
#include "fsl_codec_common.h"

#define PLAYBACK_BUFFER_SIZE (128 * 2U)
#define BUFFER_NUM           (2U)
#define RECORD_BUFFER_SIZE   (128)

pcm_rtos_t pcmHandle = {0};

extern codec_handle_t codecHandle;
extern bool bRecMic;

SDK_ALIGN(dma_descriptor_t s_DmaDescriptorPingpong[2U], 16);
static uint8_t s_buffer[PLAYBACK_BUFFER_SIZE * BUFFER_NUM];
static uint32_t volatile s_writeIndex = 0U;
static uint32_t volatile s_emptyBlock = BUFFER_NUM;

static dmic_transfer_t s_ReceiveXfer[2U] = {
    /* transfer configurations for channel0 */
    {
        .data                   = s_buffer,
        .dataWidth              = sizeof(uint16_t),
        .dataSize               = RECORD_BUFFER_SIZE,
        .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
        .linkTransfer           = &s_ReceiveXfer[1],
    },

    {
        .data                   = &s_buffer[PLAYBACK_BUFFER_SIZE],
        .dataWidth              = sizeof(uint16_t),
        .dataSize               = RECORD_BUFFER_SIZE,
        .dataAddrInterleaveSize = kDMA_AddressInterleave2xWidth,
        .linkTransfer           = &s_ReceiveXfer[0],
    },
};

static void TxCallback(I2S_Type *base, i2s_dma_handle_t *handle, status_t completionStatus, void *userData)
{
    if (s_emptyBlock < BUFFER_NUM)
    {
        s_emptyBlock++;
    }
    pcm_rtos_t *pcm = (pcm_rtos_t *)userData;
    BaseType_t reschedule;
    xSemaphoreGiveFromISR(pcm->semaphoreTX, &reschedule);
    portYIELD_FROM_ISR(reschedule);
}

static void RxCallback(DMIC_Type *base, dmic_dma_handle_t *handle, status_t completionStatus, void *userData)
{
    if (s_emptyBlock)
    {
        s_emptyBlock--;
    }
    pcm_rtos_t *pcm = (pcm_rtos_t *)userData;
    BaseType_t reschedule;
    xSemaphoreGiveFromISR(pcm->semaphoreRX, &reschedule);
    portYIELD_FROM_ISR(reschedule);
}

void streamer_pcm_init(void)
{
    DMA_Init(DEMO_DMA);

    DMA_EnableChannel(DEMO_DMA, DEMO_I2S_TX_CHANNEL);
    DMA_EnableChannel(DEMO_DMA, DEMO_DMIC_RX_CHANNEL);
    DMA_SetChannelPriority(DEMO_DMA, DEMO_I2S_TX_CHANNEL, kDMA_ChannelPriority3);
    DMA_SetChannelPriority(DEMO_DMA, DEMO_DMIC_RX_CHANNEL, kDMA_ChannelPriority2);
    DMA_CreateHandle(&pcmHandle.i2sTxDmaHandle, DEMO_DMA, DEMO_I2S_TX_CHANNEL);
    DMA_CreateHandle(&pcmHandle.dmicRxDmaHandle, DEMO_DMA, DEMO_DMIC_RX_CHANNEL);

    memset(&pcmHandle.dmic_channel_cfg, 0U, sizeof(dmic_channel_config_t));

    pcmHandle.dmic_channel_cfg.divhfclk            = kDMIC_PdmDiv1;
    pcmHandle.dmic_channel_cfg.osr                 = 32U;
    pcmHandle.dmic_channel_cfg.gainshft            = 3U;
    pcmHandle.dmic_channel_cfg.preac2coef          = kDMIC_CompValueZero;
    pcmHandle.dmic_channel_cfg.preac4coef          = kDMIC_CompValueZero;
    pcmHandle.dmic_channel_cfg.dc_cut_level        = kDMIC_DcCut155;
    pcmHandle.dmic_channel_cfg.post_dc_gain_reduce = 1U;
    pcmHandle.dmic_channel_cfg.saturate16bit       = 1U;
    pcmHandle.dmic_channel_cfg.sample_rate         = kDMIC_PhyFullSpeed;
    DMIC_Init(DMIC0);

    DMIC_Use2fs(DMIC0, true);
    DMIC_EnableChannelDma(DMIC0, DEMO_DMIC_CHANNEL, true);
    DMIC_ConfigChannel(DMIC0, DEMO_DMIC_CHANNEL, kDMIC_Left, &pcmHandle.dmic_channel_cfg);
    DMIC_FifoChannel(DMIC0, DEMO_DMIC_CHANNEL, FIFO_DEPTH, true, true);

    DMIC_EnableChannnel(DMIC0, DEMO_DMIC_CHANNEL_ENABLE);

    /* Dummy I2S TX init for the  EAP */
    I2S_TxGetDefaultConfig(&pcmHandle.tx_config);
    I2S_TxInit(DEMO_I2S_TX, &pcmHandle.tx_config);

    pcmHandle.isFirstTx = 1;
    pcmHandle.isFirstRx = 1;
}

pcm_rtos_t *streamer_pcm_open(uint32_t num_buffers)
{
    pcmHandle.semaphoreTX = xSemaphoreCreateBinary();
    return &pcmHandle;
}

pcm_rtos_t *streamer_pcm_rx_open(uint32_t num_buffers)
{
    pcmHandle.semaphoreRX = xSemaphoreCreateBinary();
    return &pcmHandle;
}

void streamer_pcm_start(pcm_rtos_t *pcm)
{
    /* Interrupts already enabled - nothing to do.
     * App/streamer can begin writing data to SAI. */
}

void streamer_pcm_close(pcm_rtos_t *pcm)
{
    /* Stop playback.  This will flush the SAI transmit buffers. */
    if (pcm->i2sTxHandle.state != 0)
    {
        I2S_TransferAbortDMA(DEMO_I2S_TX, &pcm->i2sTxHandle);
    }
    vSemaphoreDelete(pcmHandle.semaphoreTX);
}

void streamer_pcm_rx_close(pcm_rtos_t *pcm)
{
    /* Stop playback.  This will flush the SAI transmit buffers. */
    if (pcm->dmicRxHandle.state != 0)
    {
        DMIC_TransferAbortReceiveDMA(DMIC0, &pcm->dmicRxHandle);
    }
    vSemaphoreDelete(pcmHandle.semaphoreRX);
}

int streamer_pcm_write(pcm_rtos_t *pcm, uint8_t *data, uint32_t size)
{
    if (bRecMic == TRUE)
    {
        if (s_emptyBlock < BUFFER_NUM)
        {
            pcm->i2sTxTransfer.data     = s_buffer + s_writeIndex * PLAYBACK_BUFFER_SIZE;
            pcm->i2sTxTransfer.dataSize = PLAYBACK_BUFFER_SIZE;
            if (I2S_TxTransferSendDMA(DEMO_I2S_TX, &pcm->i2sTxHandle, pcm->i2sTxTransfer) == kStatus_Success)
            {
                if (++s_writeIndex >= BUFFER_NUM)
                {
                    s_writeIndex = 0U;
                }
            }
        }

        if (pcm->isFirstTx)
        {
            pcm->isFirstTx = 0;
        }
        else
        {
            /* Wait for transfer to finish */
            if (xSemaphoreTake(pcm->semaphoreTX, portMAX_DELAY) != pdTRUE)
            {
                return -1;
            }
        }
    }
    else
    {
        pcm->i2sTxTransfer.dataSize = size;
        pcm->i2sTxTransfer.data     = data;

        /* Ensure write size is a multiple of 32, otherwise EDMA will assert
         * failure.  Round down for the last chunk of a file/stream. */
        if (size % 32)
        {
            pcm->i2sTxTransfer.dataSize = size - (size % 32);
        }

        if (pcm->isFirstTx)
        {
            I2S_TxTransferSendDMA(DEMO_I2S_TX, &pcm->i2sTxHandle, pcm->i2sTxTransfer);
            I2S_TxTransferSendDMA(DEMO_I2S_TX, &pcm->i2sTxHandle, pcm->i2sTxTransfer);
            pcm->isFirstTx = 0;
        }
        else
        {
            /* Wait for transfer to finish */
            if (xSemaphoreTake(pcm->semaphoreTX, portMAX_DELAY) != pdTRUE)
            {
                return -1;
            }
            I2S_TxTransferSendDMA(DEMO_I2S_TX, &pcm->i2sTxHandle, pcm->i2sTxTransfer);
        }
    }

    return 0;
}

int streamer_pcm_read(pcm_rtos_t *pcm, uint8_t *data, uint8_t *data2, uint32_t size)
{
    /* Start the first transfer */
    if (pcm->isFirstRx)
    {
        DMIC_TransferReceiveDMA(DEMO_DMIC_RX, &pcm->dmicRxHandle, s_ReceiveXfer, DEMO_DMIC_CHANNEL);
        if (xSemaphoreTake(pcm->semaphoreRX, portMAX_DELAY) != pdTRUE)
        {
            return -1;
        }
        pcm->isFirstRx = 0;
    }
    else
    {
        /* Wait for transfer to finish */
        if (xSemaphoreTake(pcm->semaphoreRX, portMAX_DELAY) != pdTRUE)
        {
            return -1;
        }
    }

    // Enable I2S Tx due to clock availability for the codec (see board schematic).
    if (pcm->dummy_tx_enable)
    {
        I2S_Enable(DEMO_I2S_TX);
    }
    return 0;
}

int streamer_pcm_setparams(
    pcm_rtos_t *pcm, uint32_t sample_rate, uint32_t bit_width, uint8_t num_channels, bool tx, bool dummy_tx)
{
    int ret           = 0;
    int divider       = 0;
    int channel       = 0;
    pcm->sample_rate  = sample_rate;
    pcm->bit_width    = bit_width;
    pcm->num_channels = num_channels;
    pcm->dummy_tx_enable |= dummy_tx;

    if (sample_rate % 8000 == 0 || sample_rate % 6000 == 0)
    {
        const pll_setup_t pll0Setup = {
            .pllctrl = SYSCON_PLL0CTRL_CLKEN_MASK | SYSCON_PLL0CTRL_SELI(2U) | SYSCON_PLL0CTRL_SELP(31U),
            .pllndec = SYSCON_PLL0NDEC_NDIV(125U),
            .pllpdec = SYSCON_PLL0PDEC_PDIV(8U),
            .pllsscg = {0x0U, (SYSCON_PLL0SSCG1_MDIV_EXT(3072U) | SYSCON_PLL0SSCG1_SEL_EXT_MASK)},
            .pllRate = 24576000U,
            .flags   = PLL_SETUPFLAG_WAITLOCK};
        /*!< Configure PLL to the desired values */
        CLOCK_SetPLL0Freq(&pll0Setup);
    }
    else if (sample_rate % 11025 == 0)
    {
        const pll_setup_t pll0Setup = {
            .pllctrl = SYSCON_PLL0CTRL_CLKEN_MASK | SYSCON_PLL0CTRL_SELI(2U) | SYSCON_PLL0CTRL_SELP(31U),
            .pllndec = SYSCON_PLL0NDEC_NDIV(202U),
            .pllpdec = SYSCON_PLL0PDEC_PDIV(8U),
            .pllsscg = {0x0U, (SYSCON_PLL0SSCG1_MDIV_EXT(4561U) | SYSCON_PLL0SSCG1_SEL_EXT_MASK)},
            .pllRate = 22579200U,
            .flags   = PLL_SETUPFLAG_WAITLOCK};
        CLOCK_SetPLL0Freq(&pll0Setup); /*!< Configure PLL0 to the desired values */
    }

    if (bRecMic == TRUE)
    {
        divider = (CLOCK_GetPll0OutFreq() / sample_rate / bit_width / DEMO_CHANNEL_NUM);
    }
    else
    {
        divider = (CLOCK_GetPll0OutFreq() / sample_rate / bit_width / pcm->num_channels);
    }

    if (tx)
    {
        I2S_TxGetDefaultConfig(&pcmHandle.tx_config);
        pcmHandle.tx_config.divider     = divider;
        pcmHandle.tx_config.masterSlave = DEMO_I2S_TX_MODE;
        if (bRecMic == FALSE)
        {
            pcmHandle.tx_config.oneChannel = (pcm->num_channels == 1);
        }
        I2S_TxInit(DEMO_I2S_TX, &pcmHandle.tx_config);
        I2S_TxTransferCreateHandleDMA(DEMO_I2S_TX, &pcmHandle.i2sTxHandle, &pcmHandle.i2sTxDmaHandle, TxCallback,
                                      (void *)&pcmHandle);
    }
    else
    {
        DMIC_TransferCreateHandleDMA(DMIC0, &pcm->dmicRxHandle, RxCallback, (void *)&pcmHandle, &pcm->dmicRxDmaHandle);
        DMIC_InstallDMADescriptorMemory(&pcm->dmicRxHandle, s_DmaDescriptorPingpong, 2U);

        if (dummy_tx)
        {
            I2S_TxGetDefaultConfig(&pcmHandle.tx_config);
            pcmHandle.tx_config.divider     = divider;
            pcmHandle.tx_config.masterSlave = DEMO_I2S_TX_MODE;
            I2S_TxInit(DEMO_I2S_TX, &pcmHandle.tx_config);
        }
    }

    channel = (pcm->num_channels == 1) ? kCODEC_PlayChannelHeadphoneLeft :
                                         kCODEC_PlayChannelHeadphoneLeft | kCODEC_PlayChannelHeadphoneRight;

    ret = CODEC_SetMute(&codecHandle, kCODEC_PlayChannelHeadphoneLeft | kCODEC_PlayChannelHeadphoneRight, true);
    if (ret != kStatus_Success)
    {
        return 1;
    }

    ret = CODEC_SetFormat(&codecHandle, CLOCK_GetPll0OutFreq() / 2, sample_rate, bit_width);
    if (ret != kStatus_Success)
    {
        return 1;
    }

    ret = CODEC_SetMute(&codecHandle, channel, false);
    if (ret != kStatus_Success)
    {
        return 1;
    }

    return 0;
}

void streamer_pcm_getparams(pcm_rtos_t *pcm, uint32_t *sample_rate, uint32_t *bit_width, uint8_t *num_channels)
{
    *sample_rate  = pcm->sample_rate;
    *bit_width    = pcm->bit_width;
    *num_channels = pcm->num_channels;
}

int streamer_pcm_mute(pcm_rtos_t *pcm, bool mute)
{
    status_t ret;
    ret = CODEC_SetMute(&codecHandle, kCODEC_PlayChannelHeadphoneLeft | kCODEC_PlayChannelHeadphoneRight, mute);
    if (ret != kStatus_Success)
    {
        return 1;
    }

    return 0;
}

int streamer_pcm_set_volume(uint32_t volume)
{
    status_t ret;
    ret = CODEC_SetVolume(&codecHandle, kCODEC_PlayChannelHeadphoneLeft | kCODEC_PlayChannelHeadphoneRight, volume);
    if (ret != kStatus_Success)
    {
        return 1;
    }

    return 0;
}
