/*
 * Copyright 2016 - 2017 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdint.h>
#include <stdbool.h>
#include "usb_pd_config.h"
#include "usb_pd.h"
#include "usb_pd_i2c.h"
#include "usb_pd_phy.h"
#include "pd_app.h"
#include "fsl_debug_console.h"
#include "pd_power_interface.h"
#include "pd_command_interface.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

pd_status_t PD_DpmSoftResetCallback(void *callbackParam)
{
    /* reset soft status */
    PRINTF("app soft reset\r\n");
    return kStatus_PD_Success;
}

pd_status_t PD_DpmHardResetCallback(void *callbackParam)
{
    /* reset state, The Sink shall not draw more than iSafe0mA when VBUS is driven to vSafe0V. */
    PRINTF("hard reset request\r\n");
    return kStatus_PD_Success;
}

pd_status_t PD_DpmSrcRDORequestCallback(void *callbackParam, pd_rdo_t rdo, uint8_t *negotiateResult)
{
    pd_app_t *pdAppInstance = (pd_app_t *)callbackParam;

    /* prepare for power supply, The power supply shall be ready to operate at the new power level within
     * tSrcReady */
    if (rdo.bitFields.objectPosition >
        ((pd_power_port_config_t *)pdAppInstance->pdConfigParam->portConfig)->sourceCapCount)
    {
        *negotiateResult = kCommandResult_Reject;
    }
    else
    {
        pdAppInstance->partnerRequestRDO = rdo;
        *negotiateResult                 = kCommandResult_Accept;
    }
    return kStatus_PD_Success;
}

pd_status_t PD_DpmSrcPreContractStillValidCallback(void *callbackParam, uint8_t *isStillValid)
{
    pd_app_t *pdAppInstance = (pd_app_t *)callbackParam;

    /* if pre contract exist, return true */
    if (pdAppInstance->partnerRequestRDO.bitFields.objectPosition != 0)
    {
        *isStillValid = 1;
    }
    else
    {
        *isStillValid = 0;
    }
    return kStatus_PD_Success;
}

pd_status_t PD_DpmSrcRDOResultCallback(void *callbackParam, uint8_t success, uint8_t failResultType)
{
    pd_app_t *pdAppInstance = (pd_app_t *)callbackParam;

    if (success)
    {
        pdAppInstance->contractValid = 1;
        PRINTF("start charge voltage:%dV\r\n", pdAppInstance->partnerRequestRDO.bitFields.objectPosition == 1 ? 5 : 9);
    }
    else
    {
        switch (failResultType)
        {
            case kCommandResult_Reject:
                PRINTF("source reject power negotiation\r\n");
                break;

            default:
                PRINTF("source has error in power negotiation\r\n");
                break;
        }
    }

    return kStatus_PD_Success;
}

pd_status_t PD_DpmSrcGotoMinResultCallback(void *callbackParam, uint8_t success, uint8_t failResultType)
{
    return kStatus_PD_Success;
}

pd_status_t PD_DpmReceivePartnerSnkCapsCallback(void *callbackParam, pd_capabilities_t *caps)
{
    uint8_t index;
    pd_app_t *pdAppInstance = (pd_app_t *)callbackParam;
    pd_sink_pdo_t sinkPDO;
    pdAppInstance->partnerSinkCapNumber = caps->capabilitiesCount;
    for (index = 0; index < pdAppInstance->partnerSinkCapNumber; ++index)
    {
        sinkPDO.PDOValue = pdAppInstance->partnerSinkCaps[index].PDOValue = caps->capabilities[index];
        switch (sinkPDO.commonPDO.pdoType)
        {
            case kPDO_Fixed:
            {
                PRINTF("%d: fixed PDO; ", index + 1);
                PRINTF("vol:%dmV, current:%dmA\r\n", (uint16_t)(sinkPDO.fixedPDO.voltage * PD_PDO_VOLTAGE_UNIT),
                       (uint16_t)(sinkPDO.fixedPDO.operateCurrent * PD_PDO_CURRENT_UNIT));
                break;
            }

            case kPDO_Variable:
            {
                PRINTF("%d: variable PDO; ", index + 1);
                PRINTF("vol:%dmV ~ %dmV, current:%dmA\r\n",
                       (uint16_t)(sinkPDO.variablePDO.maxVoltage * PD_PDO_VOLTAGE_UNIT),
                       (uint16_t)(sinkPDO.variablePDO.minVoltage * PD_PDO_VOLTAGE_UNIT),
                       (uint16_t)(sinkPDO.variablePDO.operateCurrent * PD_PDO_CURRENT_UNIT));
                break;
            }

            case kPDO_Battery:
            {
                PRINTF("%d: battery PDO; ", index + 1);
                PRINTF("vol:%dmV ~ %dmV, power:%dmW\r\n",
                       (uint16_t)(sinkPDO.batteryPDO.maxVoltage * PD_PDO_VOLTAGE_UNIT),
                       (uint16_t)(sinkPDO.batteryPDO.minVoltage * PD_PDO_VOLTAGE_UNIT),
                       (uint16_t)(sinkPDO.batteryPDO.operatePower * PD_PDO_POWER_UNIT));
            }

            default:
                break;
        }
    }

    return kStatus_PD_Success;
}

pd_status_t PD_DpmGetPartnerSnkCapsFailCallback(void *callbackParam, uint8_t failResultType)
{
    switch (failResultType)
    {
        case kCommandResult_Error:
            PRINTF("get snk cap fail");
            break;

        case kCommandResult_NotSupported:
            PRINTF("get snk cap replying not supported");
            break;
    }

    return kStatus_PD_Success;
}
