/*--------------------------------------------------------------------------*/
/* Copyright 2021-2023 NXP                                                  */
/*                                                                          */
/* NXP Confidential. This software is owned or controlled by NXP and may    */
/* only be used strictly in accordance with the applicable license terms.   */
/* By expressly accepting such terms or by downloading, installing,         */
/* activating and/or otherwise using the software, you are agreeing that    */
/* you have read, and that you agree to comply with and are bound by, such  */
/* license terms. If you do not agree to be bound by the applicable license */
/* terms, then you may not retain, install, activate or otherwise use the   */
/* software.                                                                */
/*--------------------------------------------------------------------------*/

/**
 * @file  mcuxClEcc_ParameterSizes.h
 * @brief Definitions of ECC domain parameter, key and signature sizes
 */


#ifndef MCUXCLECC_PARAMETERSIZES_H_
#define MCUXCLECC_PARAMETERSIZES_H_

#include <mcuxClConfig.h> // Exported features flags header

/**
 * @defgroup mcuxClEcc_ParameterSizes mcuxClEcc_ParameterSizes
 * @brief Defines domain parameter, key and signature sizes of @ref mcuxClEcc
 * @ingroup mcuxClEcc
 * @{
 */



/**
 * @addtogroup MCUXCLECC_MONT_CURVE25519_SIZE_
 * MontDH parameter size definitions for Curve25519
 * @{ */
#define MCUXCLECC_MONT_CURVE25519_SIZE_PRIMEP          (32U)
#define MCUXCLECC_MONT_CURVE25519_SIZE_BASEPOINTORDER  (32U)
#define MCUXCLECC_MONT_CURVE25519_SIZE_PRIVATEKEY      (MCUXCLECC_MONT_CURVE25519_SIZE_BASEPOINTORDER)
#define MCUXCLECC_MONT_CURVE25519_SIZE_PUBLICKEY       (MCUXCLECC_MONT_CURVE25519_SIZE_PRIMEP)
#define MCUXCLECC_MONT_CURVE25519_SIZE_SHAREDSECRET    (MCUXCLECC_MONT_CURVE25519_SIZE_PRIMEP)
/** @} */  /* MCUXCLECC_MONT_CURVE25519_SIZE_ */

/**
 * @addtogroup MCUXCLECC_MONT_CURVE448_SIZE_
 * MontDH parameter size definitions for Curve448
 * @{ */
#define MCUXCLECC_MONT_CURVE448_SIZE_PRIMEP            (56U)
#define MCUXCLECC_MONT_CURVE448_SIZE_BASEPOINTORDER    (56U)
#define MCUXCLECC_MONT_CURVE448_SIZE_PRIVATEKEY        (MCUXCLECC_MONT_CURVE448_SIZE_BASEPOINTORDER)
#define MCUXCLECC_MONT_CURVE448_SIZE_PUBLICKEY         (MCUXCLECC_MONT_CURVE448_SIZE_PRIMEP)
#define MCUXCLECC_MONT_CURVE448_SIZE_SHAREDSECRET      (MCUXCLECC_MONT_CURVE448_SIZE_PRIMEP)
/** @} */  /* MCUXCLECC_MONT_CURVE448_SIZE_ */


/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP160K1_SIZE_
 * WeierECC parameter size definitions for secp160k1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_PRIMEP                     (20U)
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_BASEPOINTORDER             (21U)
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP160K1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP160K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP160K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP160K1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP160K1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP160K1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP192K1_SIZE_
 * WeierECC parameter size definitions for secp192k1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_PRIMEP                     (24U)
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_BASEPOINTORDER             (24U)
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP192K1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP192K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP192K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP192K1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP192K1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP192K1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP224K1_SIZE_
 * WeierECC parameter size definitions for secp224k1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_PRIMEP                     (28U)
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_BASEPOINTORDER             (29U)
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP224K1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP224K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP224K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP224K1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP224K1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP224K1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP256K1_SIZE_
 * WeierECC parameter size definitions for secp256k1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_PRIMEP                     (32U)
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_BASEPOINTORDER             (32U)
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP256K1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP256K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP256K1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP256K1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP256K1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP256K1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP192R1_SIZE_
 * WeierECC parameter size definitions for secp192r1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIMEP                     (24U)
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_BASEPOINTORDER             (24U)
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP192R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP192R1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP192R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP192R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP224R1_SIZE_
 * WeierECC parameter size definitions for secp224r1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIMEP                     (28U)
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_BASEPOINTORDER             (28U)
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP224R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP224R1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP224R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP224R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP256R1_SIZE_
 * WeierECC parameter size definitions for secp256r1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIMEP                     (32U)
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_BASEPOINTORDER             (32U)
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP256R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP256R1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP256R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP256R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP384R1_SIZE_
 * WeierECC parameter size definitions for secp384r1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIMEP                     (48U)
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_BASEPOINTORDER             (48U)
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP384R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP384R1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP384R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP384R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_SECP521R1_SIZE_
 * WeierECC parameter size definitions for secp521r1
 * @{ */
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIMEP                     (66U)
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_BASEPOINTORDER             (66U)
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP521R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_PUBLICKEY                  (2U * MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_SECP521R1_SIZE_SIGNATURE                  (2U * MCUXCLECC_WEIERECC_SECP521R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_SECP521R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_NIST_P192_SIZE_
 * WeierECC parameter size definitions for NIST P-192
 * @{ */
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_PRIMEP                     (MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_BASEPOINTORDER             (MCUXCLECC_WEIERECC_SECP192R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP192R1_SIZE_PRIVATEKEY)
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_PUBLICKEY                  (MCUXCLECC_WEIERECC_SECP192R1_SIZE_PUBLICKEY)
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP192R1_SIZE_SHAREDSECRET)
#define MCUXCLECC_WEIERECC_NIST_P192_SIZE_SIGNATURE                  (MCUXCLECC_WEIERECC_SECP192R1_SIZE_SIGNATURE)
/** @} */  /* MCUXCLECC_WEIERECC_NIST_P192_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_NIST_P224_SIZE_
 * WeierECC parameter size definitions for NIST P-224
 * @{ */
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_PRIMEP                     (MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_BASEPOINTORDER             (MCUXCLECC_WEIERECC_SECP224R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP224R1_SIZE_PRIVATEKEY)
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_PUBLICKEY                  (MCUXCLECC_WEIERECC_SECP224R1_SIZE_PUBLICKEY)
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP224R1_SIZE_SHAREDSECRET)
#define MCUXCLECC_WEIERECC_NIST_P224_SIZE_SIGNATURE                  (MCUXCLECC_WEIERECC_SECP224R1_SIZE_SIGNATURE)
/** @} */  /* MCUXCLECC_WEIERECC_NIST_P224_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_NIST_P256_SIZE_
 * WeierECC parameter size definitions for NIST P-256
 * @{ */
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_PRIMEP                     (MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_BASEPOINTORDER             (MCUXCLECC_WEIERECC_SECP256R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP256R1_SIZE_PRIVATEKEY)
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_PUBLICKEY                  (MCUXCLECC_WEIERECC_SECP256R1_SIZE_PUBLICKEY)
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP256R1_SIZE_SHAREDSECRET)
#define MCUXCLECC_WEIERECC_NIST_P256_SIZE_SIGNATURE                  (MCUXCLECC_WEIERECC_SECP256R1_SIZE_SIGNATURE)
/** @} */  /* MCUXCLECC_WEIERECC_NIST_P256_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_NIST_P384_SIZE_
 * WeierECC parameter size definitions for NIST P-384
 * @{ */
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_PRIMEP                     (MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_BASEPOINTORDER             (MCUXCLECC_WEIERECC_SECP384R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP384R1_SIZE_PRIVATEKEY)
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_PUBLICKEY                  (MCUXCLECC_WEIERECC_SECP384R1_SIZE_PUBLICKEY)
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP384R1_SIZE_SHAREDSECRET)
#define MCUXCLECC_WEIERECC_NIST_P384_SIZE_SIGNATURE                  (MCUXCLECC_WEIERECC_SECP384R1_SIZE_SIGNATURE)
/** @} */  /* MCUXCLECC_WEIERECC_NIST_P384_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_NIST_P521_SIZE_
 * WeierECC parameter size definitions for NIST P-521
 * @{ */
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_PRIMEP                     (MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_BASEPOINTORDER             (MCUXCLECC_WEIERECC_SECP521R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_PRIVATEKEY                 (MCUXCLECC_WEIERECC_SECP521R1_SIZE_PRIVATEKEY)
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_PUBLICKEY                  (MCUXCLECC_WEIERECC_SECP521R1_SIZE_PUBLICKEY)
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_SHAREDSECRET               (MCUXCLECC_WEIERECC_SECP521R1_SIZE_SHAREDSECRET)
#define MCUXCLECC_WEIERECC_NIST_P521_SIZE_SIGNATURE                  (MCUXCLECC_WEIERECC_SECP521R1_SIZE_SIGNATURE)
/** @} */  /* MCUXCLECC_WEIERECC_NIST_P521_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP160r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_PRIMEP                       (20U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_BASEPOINTORDER               (20U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP160R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP192r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_PRIMEP                       (24U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_BASEPOINTORDER               (24U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP192R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP224r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_PRIMEP                       (28U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_BASEPOINTORDER               (28U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP224R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP256r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_PRIMEP                       (32U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_BASEPOINTORDER               (32U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP256R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP320r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_PRIMEP                       (40U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_BASEPOINTORDER               (40U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP320R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP384r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_PRIMEP                       (48U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_BASEPOINTORDER               (48U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP384R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_
 * WeierECC parameter size definitions for brainpoolP512r1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_PRIMEP                       (64U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_BASEPOINTORDER               (64U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP512R1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP160t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_PRIMEP                       (20U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_BASEPOINTORDER               (20U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP160T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP192t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_PRIMEP                       (24U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_BASEPOINTORDER               (24U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP192T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP224t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_PRIMEP                       (28U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_BASEPOINTORDER               (28U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP224T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP256t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_PRIMEP                       (32U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_BASEPOINTORDER               (32U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP256T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP320t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_PRIMEP                       (40U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_BASEPOINTORDER               (40U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP320T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP384t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_PRIMEP                       (48U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_BASEPOINTORDER               (48U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP384T1_SIZE_ */

/**
 * @addtogroup MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_
 * WeierECC parameter size definitions for brainpoolP512t1
 * @{ */
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_PRIMEP                       (64U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_BASEPOINTORDER               (64U)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_BRAINPOOLP512T1_SIZE_ */

/** @addtogroup MCUXCLECC_WEIERECC_MAX_SIZE
 * Maximum size definitions for WeierECC parameters (ECC component officially supports up to 640 bit Weierstrass curves)
 * @{ */
#define MCUXCLECC_WEIERECC_MAX_SIZE_PRIMEP                       (80U)
#define MCUXCLECC_WEIERECC_MAX_SIZE_BASEPOINTORDER               (80U)
#define MCUXCLECC_WEIERECC_MAX_SIZE_PRIVATEKEY                   (MCUXCLECC_WEIERECC_MAX_SIZE_BASEPOINTORDER)
#define MCUXCLECC_WEIERECC_MAX_SIZE_PUBLICKEY                    (2U * MCUXCLECC_WEIERECC_MAX_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_MAX_SIZE_SHAREDSECRET                 (MCUXCLECC_WEIERECC_MAX_SIZE_PRIMEP)
#define MCUXCLECC_WEIERECC_MAX_SIZE_SIGNATURE                    (2U * MCUXCLECC_WEIERECC_MAX_SIZE_BASEPOINTORDER)
/** @} */  /* MCUXCLECC_WEIERECC_MAX_SIZE */

/** @addtogroup MCUXCLECC_EDDSA_ED25519_SIZE_
 * EdDSA parameter size definitions for Ed25519
 * @{ */
#define MCUXCLECC_EDDSA_ED25519_SIZE_PRIMEP            (32u)   ///< Byte length of the underlying prime p used in Ed25519.
#define MCUXCLECC_EDDSA_ED25519_SIZE_BASEPOINTORDER    (32u)   ///< Byte length of the base point order n used in Ed25519.
#define MCUXCLECC_EDDSA_ED25519_SIZE_PRIVATEKEY        (32u)   ///< Byte length of an Ed25519 private key.
#define MCUXCLECC_EDDSA_ED25519_SIZE_PRIVATEKEYDATA    (96u)   ///< Byte length of an Ed25519 private key handle data buffer.
#define MCUXCLECC_EDDSA_ED25519_SIZE_PUBLICKEY         (32u)   ///< Byte length of an Ed25519 public key.
#define MCUXCLECC_EDDSA_ED25519_SIZE_SIGNATURE         (64u)   ///< Byte length of an Ed25519 signature.
#define MCUXCLECC_EDDSA_ED25519_SIZE_HASH_PREFIX(contextLen) (34u + (contextLen)) ///< Byte length of an Ed25519 prefix.
#define MCUXCLECC_EDDSA_ED25519_SIZE_SIGNATURE_PROTOCOL_DESCRIPTOR(contextLen) \
    (MCUXCLECC_EDDSA_SIGNATURE_PROTOCOL_DESCRIPTOR_SIZE + MCUXCLECC_EDDSA_ED25519_SIZE_HASH_PREFIX(contextLen))   ///< Byte length of an Ed25519 signature protocol descriptor.

/** @} */  /* MCUXCLECC_EDDSA_ED25519_SIZE_ */

/** @addtogroup MCUXCLECC_EDDSA_ED448_SIZE_
 * EdDSA parameter size definitions for Ed448
 * @{ */
#define MCUXCLECC_EDDSA_ED448_SIZE_PRIMEP              (56u)   ///< Byte length of the underlying prime p used in Ed448.
#define MCUXCLECC_EDDSA_ED448_SIZE_BASEPOINTORDER      (56u)   ///< Byte length of the base point order n used in Ed448.
#define MCUXCLECC_EDDSA_ED448_SIZE_PRIVATEKEY          (57u)   ///< Byte length of an Ed448 private key.
#define MCUXCLECC_EDDSA_ED448_SIZE_PRIVATEKEYDATA      (171u)  ///< Byte length of an Ed448 private key handle data buffer.
#define MCUXCLECC_EDDSA_ED448_SIZE_PUBLICKEY           (57u)   ///< Byte length of an Ed448 public key.
#define MCUXCLECC_EDDSA_ED448_SIZE_SIGNATURE           (114u)  ///< Byte length of an Ed448 signature.
/** @} */  /* MCUXCLECC_EDDSA_ED448_SIZE_ */
/**
 * @}
 */ /* mcuxClEcc_ParameterSizes */

#endif /* MCUXCLECC_PARAMETERSIZES_H_ */
