/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_i2c.h"

#include "pin_mux.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define EXAMPLE_I2C_MASTER_BASE    (I2C4_BASE)
#define I2C_MASTER_CLOCK_FREQUENCY (12000000)
#define EXAMPLE_I2C_MASTER ((I2C_Type *)EXAMPLE_I2C_MASTER_BASE)

#define I2C_MASTER_SLAVE_ADDR_7BIT (0x7EU)
#define I2C_BAUDRATE               (100000) /* 100K */
#define I2C_DATA_LENGTH            (33)     /* MAX is 256 */

#define START_STOP_RED_PIN             29u
#define ARBITRATION_LOST_GREEN_PIN     10u
#define SCLTIMEOUT_BLUE_PIN            9u
#define EVENTTIMEOUT_LED_PIN           2u
#define UNEXPECTED_LED_PIN             3u

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
//volatile uint32_t g_systickCounter;

uint8_t g_master_txBuff[I2C_DATA_LENGTH];
uint8_t g_master_rxBuff[I2C_DATA_LENGTH];

i2c_master_handle_t g_m_handle;

volatile bool g_MasterCompletionFlag = false;
volatile uint32_t error_status = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/
//void SysTick_Handler(void)
//{
//    if (g_systickCounter != 0U)
//    {
//        g_systickCounter--;
//    }
//}

//void SysTick_DelayTicks(uint32_t n)
//{
//    g_systickCounter = n;
//    while (g_systickCounter != 0U)
//    {
//    }
//}

//void testLEDs(void)
//{
//	GPIO_PortToggle(GPIO, PORT0_IDX, (1<<EVENTTIMEOUT_LED_PIN) | (1<<UNEXPECTED_LED_PIN));
//	SysTick_DelayTicks(1000U);
//	GPIO_PortToggle(GPIO, PORT0_IDX, (1<<EVENTTIMEOUT_LED_PIN) | (1<<UNEXPECTED_LED_PIN));
//	SysTick_DelayTicks(1000U);

//	GPIO_PortToggle(GPIO, PORT0_IDX, 1u << START_STOP_RED_PIN);
//	SysTick_DelayTicks(1000U);
//	GPIO_PortToggle(GPIO, PORT0_IDX, 1u << START_STOP_RED_PIN);
//	SysTick_DelayTicks(1000U);


//	GPIO_PortToggle(GPIO, PORT1_IDX, 1u << SCLTIMEOUT_BLUE_PIN);
//	SysTick_DelayTicks(1000U);
//	GPIO_PortToggle(GPIO, PORT1_IDX, 1u << SCLTIMEOUT_BLUE_PIN);
//	SysTick_DelayTicks(1000U);

//	GPIO_PortToggle(GPIO, PORT1_IDX, 1u << ARBITRATION_LOST_GREEN_PIN);
//	SysTick_DelayTicks(1000U);
//	GPIO_PortToggle(GPIO, PORT1_IDX, 1u << ARBITRATION_LOST_GREEN_PIN);
//	SysTick_DelayTicks(1000U);
//}

void showErrorInfo(uint32_t status)
{
	switch(status)
	{
		case kStatus_I2C_StartStopError:
		{
			//GPIO_PinWrite(GPIO, PORT0_IDX, START_STOP_RED_PIN, 0);
			PRINTF("This is IIC Start/Stop error,error code:%d.\r\n",status);
		}
		break;		
		case kStatus_I2C_ArbitrationLost:
		{
      //GPIO_PinWrite(GPIO, PORT1_IDX, ARBITRATION_LOST_GREEN_PIN, 0);
			PRINTF("This is IIC Arbitration Lost error,error code:%d.\r\n",status);
		}
		break;		
		case kStatus_I2C_SCL_Timeout:
		{
      //GPIO_PinWrite(GPIO, PORT1_IDX, SCLTIMEOUT_BLUE_PIN, 0);
			PRINTF("This is IIC SCL Time-out error,error code:%d.\r\n",status);
		}
		break;		
		case kStatus_I2C_Event_Timeout:
		{
			//GPIO_PinWrite(GPIO, PORT0_IDX, EVENTTIMEOUT_LED_PIN, 1);
			PRINTF("This is IIC Event Time-out error,error code:%d.\r\n",status);
		}
		break;		
		case kStatus_I2C_UnexpectedState:
		{
      //GPIO_PinWrite(GPIO, PORT0_IDX, UNEXPECTED_LED_PIN, 0);
			//PRINTF("This is IIC UnexpectedState error,error code:%d.\r\n",status);
		}
		break;		
		default:
		{
			//PRINTF("This is IIC unknown error,error code:%d.\r\n",status);
		}
	}
}

static void i2c_master_callback(I2C_Type *base, i2c_master_handle_t *handle, status_t status, void *userData)
{
    /* Signal transfer success when received success status. */
    if (status == kStatus_Success)
    {//IIC transfer success
			g_MasterCompletionFlag = true;
    }
		else
		{//IIC transfer generates error
			//output IIC error type
			showErrorInfo(status);
		}
}

/*!
 * @brief Main function
 */
int main(void)
{
    i2c_master_transfer_t masterXfer = {0};
    status_t reVal                   = kStatus_Fail;
    uint32_t error_index;
		
		/* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput,
        0,
    };
		
		GPIO_PortInit(GPIO, PORT0_IDX);
    GPIO_PortInit(GPIO, PORT1_IDX);
			
    GPIO_PinInit(GPIO, PORT0_IDX, START_STOP_RED_PIN, &led_config);
		GPIO_PinInit(GPIO, PORT0_IDX, EVENTTIMEOUT_LED_PIN, &led_config);
		GPIO_PinInit(GPIO, PORT0_IDX, UNEXPECTED_LED_PIN, &led_config);
		GPIO_PinInit(GPIO, PORT1_IDX, SCLTIMEOUT_BLUE_PIN, &led_config);
		GPIO_PinInit(GPIO, PORT1_IDX, ARBITRATION_LOST_GREEN_PIN, &led_config);
    
		GPIO_PinWrite(GPIO, PORT0_IDX, START_STOP_RED_PIN, 1);
		GPIO_PinWrite(GPIO, PORT1_IDX, ARBITRATION_LOST_GREEN_PIN, 1);
		GPIO_PinWrite(GPIO, PORT1_IDX, SCLTIMEOUT_BLUE_PIN, 1);
		GPIO_PinWrite(GPIO, PORT0_IDX, EVENTTIMEOUT_LED_PIN, 0);
		GPIO_PinWrite(GPIO, PORT0_IDX, UNEXPECTED_LED_PIN, 1);
		
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
    /* attach 12 MHz clock to FLEXCOMM4 (I2C master) */
    CLOCK_AttachClk(kFRO12M_to_FLEXCOMM4);
    /* enable clock for GPIO*/
    CLOCK_EnableClock(kCLOCK_Gpio0);
    CLOCK_EnableClock(kCLOCK_Gpio1);
		
    /* reset FLEXCOMM for I2C */
    RESET_PeripheralReset(kFC4_RST_SHIFT_RSTn);

    BOARD_InitPins();
    BOARD_BootClockPLL150M();
    BOARD_InitDebugConsole();
    		
    PRINTF("\r\nI2C Communication Error Trigger And Detect For Master_20210412.\r\n");
		
    i2c_master_config_t masterConfig;
    /*
     * masterConfig.debugEnable = false;
     * masterConfig.ignoreAck = false;
     * masterConfig.pinConfig = kI2C_2PinOpenDrain;
     * masterConfig.baudRate_Bps = 100000U;
     * masterConfig.busIdleTimeout_ns = 0;
     * masterConfig.pinLowTimeout_ns = 0;
     * masterConfig.sdaGlitchFilterWidth_ns = 0;
     * masterConfig.sclGlitchFilterWidth_ns = 0;
     */
    I2C_MasterGetDefaultConfig(&masterConfig);
    /* Change the default baudrate configuration */
    masterConfig.baudRate_Bps = I2C_BAUDRATE;
    /* Initialize the I2C master peripheral */
    I2C_MasterInit(EXAMPLE_I2C_MASTER, &masterConfig, I2C_MASTER_CLOCK_FREQUENCY);
		
	  PRINTF("\r\nIIC Module Clock Frequency->%d.\r\n",I2C_MASTER_CLOCK_FREQUENCY);
		PRINTF("\r\nIIC Module Clock Divider->%d.\r\n\r\n",EXAMPLE_I2C_MASTER->CLKDIV);
		
		/* Set up i2c master to send data to slave*/
    /* First data in txBuff is data length of the transmiting data. */
    g_master_txBuff[0] = I2C_DATA_LENGTH - 1U;
    for (uint32_t i = 1U; i < I2C_DATA_LENGTH; i++)
    {
        g_master_txBuff[i] = i - 1;
    }
    PRINTF("Master will send data :");
    for (uint32_t i = 0U; i < I2C_DATA_LENGTH - 1U; i++)
    {
        if (i % 8 == 0)
        {
            PRINTF("\r\n");
        }
        PRINTF("0x%2x  ", g_master_txBuff[i + 1]);
    }
    PRINTF("\r\n\r\n");
		
		/********************************Du Leilei Add*********************************************/
		//Enable time_out, This is very important
    EXAMPLE_I2C_MASTER->CFG = (EXAMPLE_I2C_MASTER->CFG & I2C_CFG_MASK) | I2C_CFG_TIMEOUTEN_MASK;
		//Set time-out for SCL time-out and Event time-out
		EXAMPLE_I2C_MASTER->TIMEOUT &=~ (0xFFF<<4);
		//EXAMPLE_I2C_MASTER->TIMEOUT |= (0x1<<4);
		/********************************Du Leilei Add*********************************************/
		
    /* Create the I2C handle for the non-blocking transfer */
    I2C_MasterTransferCreateHandle(EXAMPLE_I2C_MASTER, &g_m_handle, i2c_master_callback, NULL);

    /* subAddress = 0x01, data = g_master_txBuff - write to slave.
      start + slaveaddress(w) + subAddress + length of data buffer + data buffer + stop*/
    uint8_t deviceAddress     = 0x01U;
    masterXfer.slaveAddress   = I2C_MASTER_SLAVE_ADDR_7BIT;
    masterXfer.direction      = kI2C_Write;
    masterXfer.subaddress     = (uint32_t)deviceAddress;
    masterXfer.subaddressSize = 1;
    masterXfer.data           = g_master_txBuff;
    masterXfer.dataSize       = I2C_DATA_LENGTH;
    masterXfer.flags          = kI2C_TransferDefaultFlag;
		
    /* Send master non-blocking data to slave */
    reVal = I2C_MasterTransferNonBlocking(EXAMPLE_I2C_MASTER, &g_m_handle, &masterXfer);
    /*  Reset master completion flag to false. */
    g_MasterCompletionFlag = false;

    if (reVal != kStatus_Success)
    {
        return -1;
    }

    /*  Wait for transfer completed. */
    while (!g_MasterCompletionFlag)
    {
		}
		
    g_MasterCompletionFlag = false;
    
    PRINTF("Receive sent data from slave :");

    /* subAddress = 0x01, data = g_master_rxBuff - read from slave.
      start + slaveaddress(w) + subAddress + repeated start + slaveaddress(r) + rx data buffer + stop */
    masterXfer.slaveAddress   = I2C_MASTER_SLAVE_ADDR_7BIT;
    masterXfer.direction      = kI2C_Read;
    masterXfer.subaddress     = (uint32_t)deviceAddress;
    masterXfer.subaddressSize = 1;
    masterXfer.data           = g_master_rxBuff;
    masterXfer.dataSize       = I2C_DATA_LENGTH - 1;
    masterXfer.flags          = kI2C_TransferDefaultFlag;
		
    reVal = I2C_MasterTransferNonBlocking(EXAMPLE_I2C_MASTER, &g_m_handle, &masterXfer);

    /*  Reset master completion flag to false. */
    g_MasterCompletionFlag = false;

    if (reVal != kStatus_Success)
    {
        return -1;
    }

    /*  Wait for transfer completed. */
    while (!g_MasterCompletionFlag)
    {
    }
    g_MasterCompletionFlag = false;

    for (uint32_t i = 0U; i < I2C_DATA_LENGTH - 1; i++)
    {
        if (i % 8 == 0)
        {
            PRINTF("\r\n");
        }
        PRINTF("0x%2x  ", g_master_rxBuff[i]);
    }
    PRINTF("\r\n\r\n");

    /* Transfer completed. Check the data.*/
    for (uint32_t i = 0U; i < I2C_DATA_LENGTH - 1; i++)
    {
        if (g_master_rxBuff[i] != g_master_txBuff[i + 1])
        {
            PRINTF("\r\nError occurred in the transfer ! \r\n");
            break;
        }
    }

    PRINTF("\r\nEnd of I2C example .\r\n");
    while (1)
    {
    }
}
