/*
 * Copyright 2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_iap.h"
#include "fsl_iap_ffr.h"
#include "fsl_common.h"
#include "pin_mux.h"
#include "fsl_power.h"

////////////////////////////////////////////////////////////////////////////////
// Definitions
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Prototypes
////////////////////////////////////////////////////////////////////////////////
static void verify_status(status_t status);
static void error_trap();

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////

volatile const uint8_t data_buf[0x00001000] __attribute__((section("test"))) = {1,} ;

#define BUFFER_LEN 512 / 4
static uint32_t s_buffer[BUFFER_LEN];

////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////
int main()
{
    flash_config_t flashInstance;
    static uint32_t status;
    uint32_t failedAddress, failedData;
    volatile uint32_t destAdrss, temp; /* Address of the target location */
    volatile uint32_t i, j;
    /* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput,
        1,
    };
    gpio_pin_config_t key_config = {
        kGPIO_DigitalInput,
        1,
    };
    /* Init board hardware. */
    /* set BOD VBAT level to 1.65V */
    POWER_SetBodVbatLevel(kPOWER_BodVbatLevel1650mv, kPOWER_BodHystLevel50mv, false);
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
    /* enable clock for GPIO*/
    CLOCK_EnableClock(kCLOCK_Gpio0);
    CLOCK_EnableClock(kCLOCK_Gpio1);
    
    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();
    
    
    /* Init output LED GPIO. */
    GPIO_PortInit(GPIO, 0);
    GPIO_PortInit(GPIO, 1);
    GPIO_PinInit(GPIO, BOARD_LED_GREEN_GPIO_PORT, BOARD_LED_GREEN_GPIO_PIN, &led_config);
    GPIO_PinInit(GPIO, BOARD_SW1_GPIO_PORT,       BOARD_SW1_GPIO_PIN,       &key_config);
    
    
    while(GPIO_PinRead(GPIO, BOARD_SW1_GPIO_PORT, BOARD_SW1_GPIO_PIN) == 1)
    {
        GPIO_PortToggle(GPIO, BOARD_LED_GREEN_GPIO_PORT, 1u << BOARD_LED_GREEN_GPIO_PIN);
        for(i=0; i<1000; i++)
            for(j=0; j<1000; j++);
    }
    GPIO_PinWrite(GPIO, BOARD_LED_GREEN_GPIO_PORT, BOARD_LED_GREEN_GPIO_PIN, 1);
    
    /* Print basic information for Flash Driver API.*/
    PRINTF("\r\nFlash driver API tree Demo Application...\r\n");
    /* Initialize flash driver */
    PRINTF("Initializing flash driver...\r\n");
    if (FLASH_Init(&flashInstance) == kStatus_Success)
    {
        PRINTF("Flash init successfull!!. Halting...\r\n");
    }
    else
    {
        error_trap();
    }

    destAdrss = 0x00030000;

    PRINTF("\r\nCalling FLASH_Erase() API...\r\n");
    status = FLASH_Erase(&flashInstance, destAdrss, 512, kFLASH_ApiEraseKey);
    //verify_status(status);
    PRINTF("Erase Done!\r\n");

    destAdrss = 0x00030000;
    /* Verify if the given flash range is successfully erased. */
    PRINTF("Calling FLASH_VerifyErase() API...\r\n");
    status = FLASH_VerifyErase(&flashInstance, destAdrss, 512);
    //verify_status(status);
    
    if (status == kStatus_Success)
    {
        PRINTF("FLASH Verify Erase Successful!\n");
    }
    else
    {
        error_trap();
    }
    
    memset(s_buffer, 0xFF, BUFFER_LEN);
    /* Start programming specified flash region */
    PRINTF("Calling FLASH_Program() API...\r\n");
    status = FLASH_Program(&flashInstance, destAdrss, (uint8_t *)s_buffer, sizeof(s_buffer));
    
    
    destAdrss = 0x00030000;
    /* Verify programming by reading back from flash directly */
    for (i = 0; i < BUFFER_LEN; i++)
    {
        temp = *(volatile uint32_t *)(destAdrss + i * 4);
    }
    /* Verify if the given flash region is successfully programmed with given data */
    PRINTF("Calling FLASH_VerifyProgram() API...\r\n");
    status = FLASH_VerifyProgram(&flashInstance, destAdrss, sizeof(s_buffer), (const uint8_t *)s_buffer, &failedAddress,
                                 &failedData);
    
    PRINTF("Done!\r\n");
    
    NVIC_SystemReset();
    
    while (1)
    {
    }
}

void verify_status(status_t status)
{
    char *tipString = "Unknown status";
    switch (status)
    {
        case kStatus_Success:
            tipString = "Done.";
            break;
        case kStatus_InvalidArgument:
            tipString = "Invalid argument.";
            break;
        case kStatus_FLASH_AlignmentError:
            tipString = "Alignment Error.";
            break;
        case kStatus_FLASH_AccessError:
            tipString = "Flash Access Error.";
            break;
        case kStatus_FLASH_CommandNotSupported:
            tipString = "This API is not supported in current target.";
            break;
        default:
            break;
    }
    PRINTF("%s\r\n\r\n", tipString);
}

/*
 * @brief Gets called when an error occurs.
 */
void error_trap(void)
{
    PRINTF("\r\n\r\n\r\n\t---- HALTED DUE TO FLASH ERROR! ----");
    while (1)
    {
    }
}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
