/*
 * Copyright (c) 2014 Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

// Advanced Encryption Standard
#ifdef DEBUG
#include <stdio.h>
#endif
#define SubBytes(i) (Sbox[(i) >> 24] << 24 | Sbox[(i) >> 16 & 0xff] << 16 | Sbox[(i) >> 8 & 0xff] << 8 | Sbox[(i)&0xff])

#define InvSubBytes(i) \
    (InvSbox[(i) >> 24] << 24 | InvSbox[(i) >> 16 & 0xff] << 16 | InvSbox[(i) >> 8 & 0xff] << 8 | InvSbox[(i)&0xff])

static unsigned char Sbox[] = {
    0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76, 0xca, 0x82, 0xc9,
    0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0, 0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f,
    0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15, 0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07,
    0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75, 0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3,
    0x29, 0xe3, 0x2f, 0x84, 0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58,
    0xcf, 0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8, 0x51, 0xa3,
    0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2, 0xcd, 0x0c, 0x13, 0xec, 0x5f,
    0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73, 0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88,
    0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb, 0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac,
    0x62, 0x91, 0x95, 0xe4, 0x79, 0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a,
    0xae, 0x08, 0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a, 0x70,
    0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e, 0xe1, 0xf8, 0x98, 0x11,
    0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf, 0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42,
    0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
};

static unsigned char InvSbox[] = {
    0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb, 0x7c, 0xe3, 0x39,
    0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb, 0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2,
    0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e, 0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76,
    0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25, 0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc,
    0x5d, 0x65, 0xb6, 0x92, 0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d,
    0x84, 0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06, 0xd0, 0x2c,
    0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b, 0x3a, 0x91, 0x11, 0x41, 0x4f,
    0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73, 0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85,
    0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e, 0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62,
    0x0e, 0xaa, 0x18, 0xbe, 0x1b, 0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd,
    0x5a, 0xf4, 0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f, 0x60,
    0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef, 0xa0, 0xe0, 0x3b, 0x4d,
    0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61, 0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6,
    0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d
};

static unsigned char x2[] = {
    0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c, 0x1e, 0x20, 0x22, 0x24,
    0x26, 0x28, 0x2a, 0x2c, 0x2e, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3a, 0x3c, 0x3e, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4a,
    0x4c, 0x4e, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5a, 0x5c, 0x5e, 0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6c, 0x6e, 0x70,
    0x72, 0x74, 0x76, 0x78, 0x7a, 0x7c, 0x7e, 0x80, 0x82, 0x84, 0x86, 0x88, 0x8a, 0x8c, 0x8e, 0x90, 0x92, 0x94, 0x96,
    0x98, 0x9a, 0x9c, 0x9e, 0xa0, 0xa2, 0xa4, 0xa6, 0xa8, 0xaa, 0xac, 0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc,
    0xbe, 0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc, 0xde, 0xe0, 0xe2,
    0xe4, 0xe6, 0xe8, 0xea, 0xec, 0xee, 0xf0, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfc, 0xfe, 0x1b, 0x19, 0x1f, 0x1d, 0x13,
    0x11, 0x17, 0x15, 0x0b, 0x09, 0x0f, 0x0d, 0x03, 0x01, 0x07, 0x05, 0x3b, 0x39, 0x3f, 0x3d, 0x33, 0x31, 0x37, 0x35,
    0x2b, 0x29, 0x2f, 0x2d, 0x23, 0x21, 0x27, 0x25, 0x5b, 0x59, 0x5f, 0x5d, 0x53, 0x51, 0x57, 0x55, 0x4b, 0x49, 0x4f,
    0x4d, 0x43, 0x41, 0x47, 0x45, 0x7b, 0x79, 0x7f, 0x7d, 0x73, 0x71, 0x77, 0x75, 0x6b, 0x69, 0x6f, 0x6d, 0x63, 0x61,
    0x67, 0x65, 0x9b, 0x99, 0x9f, 0x9d, 0x93, 0x91, 0x97, 0x95, 0x8b, 0x89, 0x8f, 0x8d, 0x83, 0x81, 0x87, 0x85, 0xbb,
    0xb9, 0xbf, 0xbd, 0xb3, 0xb1, 0xb7, 0xb5, 0xab, 0xa9, 0xaf, 0xad, 0xa3, 0xa1, 0xa7, 0xa5, 0xdb, 0xd9, 0xdf, 0xdd,
    0xd3, 0xd1, 0xd7, 0xd5, 0xcb, 0xc9, 0xcf, 0xcd, 0xc3, 0xc1, 0xc7, 0xc5, 0xfb, 0xf9, 0xff, 0xfd, 0xf3, 0xf1, 0xf7,
    0xf5, 0xeb, 0xe9, 0xef, 0xed, 0xe3, 0xe1, 0xe7, 0xe5
};

static unsigned char x3[] = {
    0x00, 0x03, 0x06, 0x05, 0x0c, 0x0f, 0x0a, 0x09, 0x18, 0x1b, 0x1e, 0x1d, 0x14, 0x17, 0x12, 0x11, 0x30, 0x33, 0x36,
    0x35, 0x3c, 0x3f, 0x3a, 0x39, 0x28, 0x2b, 0x2e, 0x2d, 0x24, 0x27, 0x22, 0x21, 0x60, 0x63, 0x66, 0x65, 0x6c, 0x6f,
    0x6a, 0x69, 0x78, 0x7b, 0x7e, 0x7d, 0x74, 0x77, 0x72, 0x71, 0x50, 0x53, 0x56, 0x55, 0x5c, 0x5f, 0x5a, 0x59, 0x48,
    0x4b, 0x4e, 0x4d, 0x44, 0x47, 0x42, 0x41, 0xc0, 0xc3, 0xc6, 0xc5, 0xcc, 0xcf, 0xca, 0xc9, 0xd8, 0xdb, 0xde, 0xdd,
    0xd4, 0xd7, 0xd2, 0xd1, 0xf0, 0xf3, 0xf6, 0xf5, 0xfc, 0xff, 0xfa, 0xf9, 0xe8, 0xeb, 0xee, 0xed, 0xe4, 0xe7, 0xe2,
    0xe1, 0xa0, 0xa3, 0xa6, 0xa5, 0xac, 0xaf, 0xaa, 0xa9, 0xb8, 0xbb, 0xbe, 0xbd, 0xb4, 0xb7, 0xb2, 0xb1, 0x90, 0x93,
    0x96, 0x95, 0x9c, 0x9f, 0x9a, 0x99, 0x88, 0x8b, 0x8e, 0x8d, 0x84, 0x87, 0x82, 0x81, 0x9b, 0x98, 0x9d, 0x9e, 0x97,
    0x94, 0x91, 0x92, 0x83, 0x80, 0x85, 0x86, 0x8f, 0x8c, 0x89, 0x8a, 0xab, 0xa8, 0xad, 0xae, 0xa7, 0xa4, 0xa1, 0xa2,
    0xb3, 0xb0, 0xb5, 0xb6, 0xbf, 0xbc, 0xb9, 0xba, 0xfb, 0xf8, 0xfd, 0xfe, 0xf7, 0xf4, 0xf1, 0xf2, 0xe3, 0xe0, 0xe5,
    0xe6, 0xef, 0xec, 0xe9, 0xea, 0xcb, 0xc8, 0xcd, 0xce, 0xc7, 0xc4, 0xc1, 0xc2, 0xd3, 0xd0, 0xd5, 0xd6, 0xdf, 0xdc,
    0xd9, 0xda, 0x5b, 0x58, 0x5d, 0x5e, 0x57, 0x54, 0x51, 0x52, 0x43, 0x40, 0x45, 0x46, 0x4f, 0x4c, 0x49, 0x4a, 0x6b,
    0x68, 0x6d, 0x6e, 0x67, 0x64, 0x61, 0x62, 0x73, 0x70, 0x75, 0x76, 0x7f, 0x7c, 0x79, 0x7a, 0x3b, 0x38, 0x3d, 0x3e,
    0x37, 0x34, 0x31, 0x32, 0x23, 0x20, 0x25, 0x26, 0x2f, 0x2c, 0x29, 0x2a, 0x0b, 0x08, 0x0d, 0x0e, 0x07, 0x04, 0x01,
    0x02, 0x13, 0x10, 0x15, 0x16, 0x1f, 0x1c, 0x19, 0x1a
};

static unsigned char x9[] = {
    0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77, 0x90, 0x99, 0x82,
    0x8b, 0xb4, 0xbd, 0xa6, 0xaf, 0xd8, 0xd1, 0xca, 0xc3, 0xfc, 0xf5, 0xee, 0xe7, 0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16,
    0x0d, 0x04, 0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c, 0xab, 0xa2, 0xb9, 0xb0, 0x8f, 0x86, 0x9d, 0x94, 0xe3,
    0xea, 0xf1, 0xf8, 0xc7, 0xce, 0xd5, 0xdc, 0x76, 0x7f, 0x64, 0x6d, 0x52, 0x5b, 0x40, 0x49, 0x3e, 0x37, 0x2c, 0x25,
    0x1a, 0x13, 0x08, 0x01, 0xe6, 0xef, 0xf4, 0xfd, 0xc2, 0xcb, 0xd0, 0xd9, 0xae, 0xa7, 0xbc, 0xb5, 0x8a, 0x83, 0x98,
    0x91, 0x4d, 0x44, 0x5f, 0x56, 0x69, 0x60, 0x7b, 0x72, 0x05, 0x0c, 0x17, 0x1e, 0x21, 0x28, 0x33, 0x3a, 0xdd, 0xd4,
    0xcf, 0xc6, 0xf9, 0xf0, 0xeb, 0xe2, 0x95, 0x9c, 0x87, 0x8e, 0xb1, 0xb8, 0xa3, 0xaa, 0xec, 0xe5, 0xfe, 0xf7, 0xc8,
    0xc1, 0xda, 0xd3, 0xa4, 0xad, 0xb6, 0xbf, 0x80, 0x89, 0x92, 0x9b, 0x7c, 0x75, 0x6e, 0x67, 0x58, 0x51, 0x4a, 0x43,
    0x34, 0x3d, 0x26, 0x2f, 0x10, 0x19, 0x02, 0x0b, 0xd7, 0xde, 0xc5, 0xcc, 0xf3, 0xfa, 0xe1, 0xe8, 0x9f, 0x96, 0x8d,
    0x84, 0xbb, 0xb2, 0xa9, 0xa0, 0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71, 0x78, 0x0f, 0x06, 0x1d, 0x14, 0x2b, 0x22,
    0x39, 0x30, 0x9a, 0x93, 0x88, 0x81, 0xbe, 0xb7, 0xac, 0xa5, 0xd2, 0xdb, 0xc0, 0xc9, 0xf6, 0xff, 0xe4, 0xed, 0x0a,
    0x03, 0x18, 0x11, 0x2e, 0x27, 0x3c, 0x35, 0x42, 0x4b, 0x50, 0x59, 0x66, 0x6f, 0x74, 0x7d, 0xa1, 0xa8, 0xb3, 0xba,
    0x85, 0x8c, 0x97, 0x9e, 0xe9, 0xe0, 0xfb, 0xf2, 0xcd, 0xc4, 0xdf, 0xd6, 0x31, 0x38, 0x23, 0x2a, 0x15, 0x1c, 0x07,
    0x0e, 0x79, 0x70, 0x6b, 0x62, 0x5d, 0x54, 0x4f, 0x46
};

static unsigned char xb[] = {
    0x00, 0x0b, 0x16, 0x1d, 0x2c, 0x27, 0x3a, 0x31, 0x58, 0x53, 0x4e, 0x45, 0x74, 0x7f, 0x62, 0x69, 0xb0, 0xbb, 0xa6,
    0xad, 0x9c, 0x97, 0x8a, 0x81, 0xe8, 0xe3, 0xfe, 0xf5, 0xc4, 0xcf, 0xd2, 0xd9, 0x7b, 0x70, 0x6d, 0x66, 0x57, 0x5c,
    0x41, 0x4a, 0x23, 0x28, 0x35, 0x3e, 0x0f, 0x04, 0x19, 0x12, 0xcb, 0xc0, 0xdd, 0xd6, 0xe7, 0xec, 0xf1, 0xfa, 0x93,
    0x98, 0x85, 0x8e, 0xbf, 0xb4, 0xa9, 0xa2, 0xf6, 0xfd, 0xe0, 0xeb, 0xda, 0xd1, 0xcc, 0xc7, 0xae, 0xa5, 0xb8, 0xb3,
    0x82, 0x89, 0x94, 0x9f, 0x46, 0x4d, 0x50, 0x5b, 0x6a, 0x61, 0x7c, 0x77, 0x1e, 0x15, 0x08, 0x03, 0x32, 0x39, 0x24,
    0x2f, 0x8d, 0x86, 0x9b, 0x90, 0xa1, 0xaa, 0xb7, 0xbc, 0xd5, 0xde, 0xc3, 0xc8, 0xf9, 0xf2, 0xef, 0xe4, 0x3d, 0x36,
    0x2b, 0x20, 0x11, 0x1a, 0x07, 0x0c, 0x65, 0x6e, 0x73, 0x78, 0x49, 0x42, 0x5f, 0x54, 0xf7, 0xfc, 0xe1, 0xea, 0xdb,
    0xd0, 0xcd, 0xc6, 0xaf, 0xa4, 0xb9, 0xb2, 0x83, 0x88, 0x95, 0x9e, 0x47, 0x4c, 0x51, 0x5a, 0x6b, 0x60, 0x7d, 0x76,
    0x1f, 0x14, 0x09, 0x02, 0x33, 0x38, 0x25, 0x2e, 0x8c, 0x87, 0x9a, 0x91, 0xa0, 0xab, 0xb6, 0xbd, 0xd4, 0xdf, 0xc2,
    0xc9, 0xf8, 0xf3, 0xee, 0xe5, 0x3c, 0x37, 0x2a, 0x21, 0x10, 0x1b, 0x06, 0x0d, 0x64, 0x6f, 0x72, 0x79, 0x48, 0x43,
    0x5e, 0x55, 0x01, 0x0a, 0x17, 0x1c, 0x2d, 0x26, 0x3b, 0x30, 0x59, 0x52, 0x4f, 0x44, 0x75, 0x7e, 0x63, 0x68, 0xb1,
    0xba, 0xa7, 0xac, 0x9d, 0x96, 0x8b, 0x80, 0xe9, 0xe2, 0xff, 0xf4, 0xc5, 0xce, 0xd3, 0xd8, 0x7a, 0x71, 0x6c, 0x67,
    0x56, 0x5d, 0x40, 0x4b, 0x22, 0x29, 0x34, 0x3f, 0x0e, 0x05, 0x18, 0x13, 0xca, 0xc1, 0xdc, 0xd7, 0xe6, 0xed, 0xf0,
    0xfb, 0x92, 0x99, 0x84, 0x8f, 0xbe, 0xb5, 0xa8, 0xa3
};

static unsigned char xd[] = {
    0x00, 0x0d, 0x1a, 0x17, 0x34, 0x39, 0x2e, 0x23, 0x68, 0x65, 0x72, 0x7f, 0x5c, 0x51, 0x46, 0x4b, 0xd0, 0xdd, 0xca,
    0xc7, 0xe4, 0xe9, 0xfe, 0xf3, 0xb8, 0xb5, 0xa2, 0xaf, 0x8c, 0x81, 0x96, 0x9b, 0xbb, 0xb6, 0xa1, 0xac, 0x8f, 0x82,
    0x95, 0x98, 0xd3, 0xde, 0xc9, 0xc4, 0xe7, 0xea, 0xfd, 0xf0, 0x6b, 0x66, 0x71, 0x7c, 0x5f, 0x52, 0x45, 0x48, 0x03,
    0x0e, 0x19, 0x14, 0x37, 0x3a, 0x2d, 0x20, 0x6d, 0x60, 0x77, 0x7a, 0x59, 0x54, 0x43, 0x4e, 0x05, 0x08, 0x1f, 0x12,
    0x31, 0x3c, 0x2b, 0x26, 0xbd, 0xb0, 0xa7, 0xaa, 0x89, 0x84, 0x93, 0x9e, 0xd5, 0xd8, 0xcf, 0xc2, 0xe1, 0xec, 0xfb,
    0xf6, 0xd6, 0xdb, 0xcc, 0xc1, 0xe2, 0xef, 0xf8, 0xf5, 0xbe, 0xb3, 0xa4, 0xa9, 0x8a, 0x87, 0x90, 0x9d, 0x06, 0x0b,
    0x1c, 0x11, 0x32, 0x3f, 0x28, 0x25, 0x6e, 0x63, 0x74, 0x79, 0x5a, 0x57, 0x40, 0x4d, 0xda, 0xd7, 0xc0, 0xcd, 0xee,
    0xe3, 0xf4, 0xf9, 0xb2, 0xbf, 0xa8, 0xa5, 0x86, 0x8b, 0x9c, 0x91, 0x0a, 0x07, 0x10, 0x1d, 0x3e, 0x33, 0x24, 0x29,
    0x62, 0x6f, 0x78, 0x75, 0x56, 0x5b, 0x4c, 0x41, 0x61, 0x6c, 0x7b, 0x76, 0x55, 0x58, 0x4f, 0x42, 0x09, 0x04, 0x13,
    0x1e, 0x3d, 0x30, 0x27, 0x2a, 0xb1, 0xbc, 0xab, 0xa6, 0x85, 0x88, 0x9f, 0x92, 0xd9, 0xd4, 0xc3, 0xce, 0xed, 0xe0,
    0xf7, 0xfa, 0xb7, 0xba, 0xad, 0xa0, 0x83, 0x8e, 0x99, 0x94, 0xdf, 0xd2, 0xc5, 0xc8, 0xeb, 0xe6, 0xf1, 0xfc, 0x67,
    0x6a, 0x7d, 0x70, 0x53, 0x5e, 0x49, 0x44, 0x0f, 0x02, 0x15, 0x18, 0x3b, 0x36, 0x21, 0x2c, 0x0c, 0x01, 0x16, 0x1b,
    0x38, 0x35, 0x22, 0x2f, 0x64, 0x69, 0x7e, 0x73, 0x50, 0x5d, 0x4a, 0x47, 0xdc, 0xd1, 0xc6, 0xcb, 0xe8, 0xe5, 0xf2,
    0xff, 0xb4, 0xb9, 0xae, 0xa3, 0x80, 0x8d, 0x9a, 0x97
};

static unsigned char xe[] = {
    0x00, 0x0e, 0x1c, 0x12, 0x38, 0x36, 0x24, 0x2a, 0x70, 0x7e, 0x6c, 0x62, 0x48, 0x46, 0x54, 0x5a, 0xe0, 0xee, 0xfc,
    0xf2, 0xd8, 0xd6, 0xc4, 0xca, 0x90, 0x9e, 0x8c, 0x82, 0xa8, 0xa6, 0xb4, 0xba, 0xdb, 0xd5, 0xc7, 0xc9, 0xe3, 0xed,
    0xff, 0xf1, 0xab, 0xa5, 0xb7, 0xb9, 0x93, 0x9d, 0x8f, 0x81, 0x3b, 0x35, 0x27, 0x29, 0x03, 0x0d, 0x1f, 0x11, 0x4b,
    0x45, 0x57, 0x59, 0x73, 0x7d, 0x6f, 0x61, 0xad, 0xa3, 0xb1, 0xbf, 0x95, 0x9b, 0x89, 0x87, 0xdd, 0xd3, 0xc1, 0xcf,
    0xe5, 0xeb, 0xf9, 0xf7, 0x4d, 0x43, 0x51, 0x5f, 0x75, 0x7b, 0x69, 0x67, 0x3d, 0x33, 0x21, 0x2f, 0x05, 0x0b, 0x19,
    0x17, 0x76, 0x78, 0x6a, 0x64, 0x4e, 0x40, 0x52, 0x5c, 0x06, 0x08, 0x1a, 0x14, 0x3e, 0x30, 0x22, 0x2c, 0x96, 0x98,
    0x8a, 0x84, 0xae, 0xa0, 0xb2, 0xbc, 0xe6, 0xe8, 0xfa, 0xf4, 0xde, 0xd0, 0xc2, 0xcc, 0x41, 0x4f, 0x5d, 0x53, 0x79,
    0x77, 0x65, 0x6b, 0x31, 0x3f, 0x2d, 0x23, 0x09, 0x07, 0x15, 0x1b, 0xa1, 0xaf, 0xbd, 0xb3, 0x99, 0x97, 0x85, 0x8b,
    0xd1, 0xdf, 0xcd, 0xc3, 0xe9, 0xe7, 0xf5, 0xfb, 0x9a, 0x94, 0x86, 0x88, 0xa2, 0xac, 0xbe, 0xb0, 0xea, 0xe4, 0xf6,
    0xf8, 0xd2, 0xdc, 0xce, 0xc0, 0x7a, 0x74, 0x66, 0x68, 0x42, 0x4c, 0x5e, 0x50, 0x0a, 0x04, 0x16, 0x18, 0x32, 0x3c,
    0x2e, 0x20, 0xec, 0xe2, 0xf0, 0xfe, 0xd4, 0xda, 0xc8, 0xc6, 0x9c, 0x92, 0x80, 0x8e, 0xa4, 0xaa, 0xb8, 0xb6, 0x0c,
    0x02, 0x10, 0x1e, 0x34, 0x3a, 0x28, 0x26, 0x7c, 0x72, 0x60, 0x6e, 0x44, 0x4a, 0x58, 0x56, 0x37, 0x39, 0x2b, 0x25,
    0x0f, 0x01, 0x13, 0x1d, 0x47, 0x49, 0x5b, 0x55, 0x7f, 0x71, 0x63, 0x6d, 0xd7, 0xd9, 0xcb, 0xc5, 0xef, 0xe1, 0xf3,
    0xfd, 0xa7, 0xa9, 0xbb, 0xb5, 0x9f, 0x91, 0x83, 0x8d
};

static unsigned int Rcon[] = { 0x01000000, 0x02000000, 0x04000000, 0x08000000, 0x10000000,
                               0x20000000, 0x40000000, 0x80000000, 0x1b000000, 0x36000000 };

void Cipher(unsigned char cin[], unsigned int w[], int nr, unsigned char cout[])
{
    unsigned int state[4], temp[4], in[4], out[4];
    unsigned int s00, s01, s02, s03, s10, s11, s12, s13;
    unsigned int s20, s21, s22, s23, s30, s31, s32, s33;
    int i;

    in[0] = cin[0] << 24 | cin[1] << 16 | cin[2] << 8 | cin[3];
    in[1] = cin[4] << 24 | cin[5] << 16 | cin[6] << 8 | cin[7];
    in[2] = cin[8] << 24 | cin[9] << 16 | cin[10] << 8 | cin[11];
    in[3] = cin[12] << 24 | cin[13] << 16 | cin[14] << 8 | cin[15];
#ifdef DEBUG
    printf("input:   %.8x %.8x %.8x %.8x\n", in[0], in[1], in[2], in[3]);
    printf("k_sch:   %.8x %.8x %.8x %.8x\n", w[0], w[1], w[2], w[3]);
#endif
    state[0] = in[0] ^ w[0];
    state[1] = in[1] ^ w[1];
    state[2] = in[2] ^ w[2];
    state[3] = in[3] ^ w[3];

    for (i = 4; i < 4 * nr; i += 4)
    {
#ifdef DEBUG
        printf("start:   %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
        state[0] = SubBytes(state[0]);
        state[1] = SubBytes(state[1]);
        state[2] = SubBytes(state[2]);
        state[3] = SubBytes(state[3]);
#ifdef DEBUG
        printf("s_box:   %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
        temp[0] = (state[0] & 0xff000000) | (state[1] & 0xff0000) | (state[2] & 0xff00) | (state[3] & 0xff);
        temp[1] = (state[1] & 0xff000000) | (state[2] & 0xff0000) | (state[3] & 0xff00) | (state[0] & 0xff);
        temp[2] = (state[2] & 0xff000000) | (state[3] & 0xff0000) | (state[0] & 0xff00) | (state[1] & 0xff);
        temp[3] = (state[3] & 0xff000000) | (state[0] & 0xff0000) | (state[1] & 0xff00) | (state[2] & 0xff);
#ifdef DEBUG
        printf("s_row:   %.8x %.8x %.8x %.8x\n", temp[0], temp[1], temp[2], temp[3]);
#endif
        s00 = temp[0] >> 24 & 0xff;
        s10 = temp[0] >> 16 & 0xff;
        s20 = temp[0] >> 8 & 0xff;
        s30 = temp[0] & 0xff;
        s01 = temp[1] >> 24 & 0xff;
        s11 = temp[1] >> 16 & 0xff;
        s21 = temp[1] >> 8 & 0xff;
        s31 = temp[1] & 0xff;
        s02 = temp[2] >> 24 & 0xff;
        s12 = temp[2] >> 16 & 0xff;
        s22 = temp[2] >> 8 & 0xff;
        s32 = temp[2] & 0xff;
        s03 = temp[3] >> 24 & 0xff;
        s13 = temp[3] >> 16 & 0xff;
        s23 = temp[3] >> 8 & 0xff;
        s33 = temp[3] & 0xff;
        state[0] = (x2[s00] ^ x3[s10] ^ s20 ^ s30) << 24 | (s00 ^ x2[s10] ^ x3[s20] ^ s30) << 16 |
                   (s00 ^ s10 ^ x2[s20] ^ x3[s30]) << 8 | (x3[s00] ^ s10 ^ s20 ^ x2[s30]);
        state[1] = (x2[s01] ^ x3[s11] ^ s21 ^ s31) << 24 | (s01 ^ x2[s11] ^ x3[s21] ^ s31) << 16 |
                   (s01 ^ s11 ^ x2[s21] ^ x3[s31]) << 8 | (x3[s01] ^ s11 ^ s21 ^ x2[s31]);
        state[2] = (x2[s02] ^ x3[s12] ^ s22 ^ s32) << 24 | (s02 ^ x2[s12] ^ x3[s22] ^ s32) << 16 |
                   (s02 ^ s12 ^ x2[s22] ^ x3[s32]) << 8 | (x3[s02] ^ s12 ^ s22 ^ x2[s32]);
        state[3] = (x2[s03] ^ x3[s13] ^ s23 ^ s33) << 24 | (s03 ^ x2[s13] ^ x3[s23] ^ s33) << 16 |
                   (s03 ^ s13 ^ x2[s23] ^ x3[s33]) << 8 | (x3[s03] ^ s13 ^ s23 ^ x2[s33]);
#ifdef DEBUG
        printf("m_col:   %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
        printf("k_sch:   %.8x %.8x %.8x %.8x\n", w[i], w[i + 1], w[i + 2], w[i + 3]);
#endif
        state[0] ^= w[i];
        state[1] ^= w[i + 1];
        state[2] ^= w[i + 2];
        state[3] ^= w[i + 3];
    }

#ifdef DEBUG
    printf("start:   %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
    state[0] = SubBytes(state[0]);
    state[1] = SubBytes(state[1]);
    state[2] = SubBytes(state[2]);
    state[3] = SubBytes(state[3]);
#ifdef DEBUG
    printf("s_box:   %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
    temp[0] = (state[0] & 0xff000000) | (state[1] & 0xff0000) | (state[2] & 0xff00) | (state[3] & 0xff);
    temp[1] = (state[1] & 0xff000000) | (state[2] & 0xff0000) | (state[3] & 0xff00) | (state[0] & 0xff);
    temp[2] = (state[2] & 0xff000000) | (state[3] & 0xff0000) | (state[0] & 0xff00) | (state[1] & 0xff);
    temp[3] = (state[3] & 0xff000000) | (state[0] & 0xff0000) | (state[1] & 0xff00) | (state[2] & 0xff);
#ifdef DEBUG
    printf("s_row:   %.8x %.8x %.8x %.8x\n", temp[0], temp[1], temp[2], temp[3]);
    printf("k_sch:   %.8x %.8x %.8x %.8x\n", w[4 * nr], w[4 * nr + 1], w[4 * nr + 2], w[4 * nr + 3]);
#endif
    out[0] = temp[0] ^ w[4 * nr];
    out[1] = temp[1] ^ w[4 * nr + 1];
    out[2] = temp[2] ^ w[4 * nr + 2];
    out[3] = temp[3] ^ w[4 * nr + 3];
#ifdef DEBUG
    printf("output:  %.8x %.8x %.8x %.8x\n", out[0], out[1], out[2], out[3]);
#endif
    cout[0] = out[0] >> 24 & 0xff;
    cout[1] = out[0] >> 16 & 0xff;
    cout[2] = out[0] >> 8 & 0xff;
    cout[3] = out[0] & 0xff;
    cout[4] = out[1] >> 24 & 0xff;
    cout[5] = out[1] >> 16 & 0xff;
    cout[6] = out[1] >> 8 & 0xff;
    cout[7] = out[1] & 0xff;
    cout[8] = out[2] >> 24 & 0xff;
    cout[9] = out[2] >> 16 & 0xff;
    cout[10] = out[2] >> 8 & 0xff;
    cout[11] = out[2] & 0xff;
    cout[12] = out[3] >> 24 & 0xff;
    cout[13] = out[3] >> 16 & 0xff;
    cout[14] = out[3] >> 8 & 0xff;
    cout[15] = out[3] & 0xff;
}

void InvCipher(unsigned char cin[], unsigned int w[], int nr, unsigned char cout[])
{
    unsigned int state[4], temp[4], in[4], out[4];
    unsigned int s00, s01, s02, s03, s10, s11, s12, s13;
    unsigned int s20, s21, s22, s23, s30, s31, s32, s33;
    int i;

    in[0] = cin[0] << 24 | cin[1] << 16 | cin[2] << 8 | cin[3];
    in[1] = cin[4] << 24 | cin[5] << 16 | cin[6] << 8 | cin[7];
    in[2] = cin[8] << 24 | cin[9] << 16 | cin[10] << 8 | cin[11];
    in[3] = cin[12] << 24 | cin[13] << 16 | cin[14] << 8 | cin[15];
#ifdef DEBUG
    printf("iinput:  %.8x %.8x %.8x %.8x\n", in[0], in[1], in[2], in[3]);
    printf("ik_sch:  %.8x %.8x %.8x %.8x\n", w[4 * nr], w[4 * nr + 1], w[4 * nr + 2], w[4 * nr + 3]);
#endif
    state[0] = in[0] ^ w[4 * nr];
    state[1] = in[1] ^ w[4 * nr + 1];
    state[2] = in[2] ^ w[4 * nr + 2];
    state[3] = in[3] ^ w[4 * nr + 3];

    for (i = 4 * nr - 4; i > 0; i -= 4)
    {
#ifdef DEBUG
        printf("istart:  %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
        temp[0] = (state[0] & 0xff000000) | (state[3] & 0xff0000) | (state[2] & 0xff00) | (state[1] & 0xff);
        temp[1] = (state[1] & 0xff000000) | (state[0] & 0xff0000) | (state[3] & 0xff00) | (state[2] & 0xff);
        temp[2] = (state[2] & 0xff000000) | (state[1] & 0xff0000) | (state[0] & 0xff00) | (state[3] & 0xff);
        temp[3] = (state[3] & 0xff000000) | (state[2] & 0xff0000) | (state[1] & 0xff00) | (state[0] & 0xff);
#ifdef DEBUG
        printf("is_row:  %.8x %.8x %.8x %.8x\n", temp[0], temp[1], temp[2], temp[3]);
#endif
        state[0] = InvSubBytes(temp[0]);
        state[1] = InvSubBytes(temp[1]);
        state[2] = InvSubBytes(temp[2]);
        state[3] = InvSubBytes(temp[3]);
#ifdef DEBUG
        printf("is_box:  %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
        printf("ik_sch:  %.8x %.8x %.8x %.8x\n", w[i], w[i + 1], w[i + 2], w[i + 3]);
#endif
        state[0] ^= w[i];
        state[1] ^= w[i + 1];
        state[2] ^= w[i + 2];
        state[3] ^= w[i + 3];
#ifdef DEBUG
        printf("ik_add:  %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
        s00 = state[0] >> 24 & 0xff;
        s10 = state[0] >> 16 & 0xff;
        s20 = state[0] >> 8 & 0xff;
        s30 = state[0] & 0xff;
        s01 = state[1] >> 24 & 0xff;
        s11 = state[1] >> 16 & 0xff;
        s21 = state[1] >> 8 & 0xff;
        s31 = state[1] & 0xff;
        s02 = state[2] >> 24 & 0xff;
        s12 = state[2] >> 16 & 0xff;
        s22 = state[2] >> 8 & 0xff;
        s32 = state[2] & 0xff;
        s03 = state[3] >> 24 & 0xff;
        s13 = state[3] >> 16 & 0xff;
        s23 = state[3] >> 8 & 0xff;
        s33 = state[3] & 0xff;
        state[0] = (xe[s00] ^ xb[s10] ^ xd[s20] ^ x9[s30]) << 24 | (x9[s00] ^ xe[s10] ^ xb[s20] ^ xd[s30]) << 16 |
                   (xd[s00] ^ x9[s10] ^ xe[s20] ^ xb[s30]) << 8 | (xb[s00] ^ xd[s10] ^ x9[s20] ^ xe[s30]);
        state[1] = (xe[s01] ^ xb[s11] ^ xd[s21] ^ x9[s31]) << 24 | (x9[s01] ^ xe[s11] ^ xb[s21] ^ xd[s31]) << 16 |
                   (xd[s01] ^ x9[s11] ^ xe[s21] ^ xb[s31]) << 8 | (xb[s01] ^ xd[s11] ^ x9[s21] ^ xe[s31]);
        state[2] = (xe[s02] ^ xb[s12] ^ xd[s22] ^ x9[s32]) << 24 | (x9[s02] ^ xe[s12] ^ xb[s22] ^ xd[s32]) << 16 |
                   (xd[s02] ^ x9[s12] ^ xe[s22] ^ xb[s32]) << 8 | (xb[s02] ^ xd[s12] ^ x9[s22] ^ xe[s32]);
        state[3] = (xe[s03] ^ xb[s13] ^ xd[s23] ^ x9[s33]) << 24 | (x9[s03] ^ xe[s13] ^ xb[s23] ^ xd[s33]) << 16 |
                   (xd[s03] ^ x9[s13] ^ xe[s23] ^ xb[s33]) << 8 | (xb[s03] ^ xd[s13] ^ x9[s23] ^ xe[s33]);
    }

#ifdef DEBUG
    printf("istart:  %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
#endif
    temp[0] = (state[0] & 0xff000000) | (state[3] & 0xff0000) | (state[2] & 0xff00) | (state[1] & 0xff);
    temp[1] = (state[1] & 0xff000000) | (state[0] & 0xff0000) | (state[3] & 0xff00) | (state[2] & 0xff);
    temp[2] = (state[2] & 0xff000000) | (state[1] & 0xff0000) | (state[0] & 0xff00) | (state[3] & 0xff);
    temp[3] = (state[3] & 0xff000000) | (state[2] & 0xff0000) | (state[1] & 0xff00) | (state[0] & 0xff);
#ifdef DEBUG
    printf("is_row:  %.8x %.8x %.8x %.8x\n", temp[0], temp[1], temp[2], temp[3]);
#endif
    state[0] = InvSubBytes(temp[0]);
    state[1] = InvSubBytes(temp[1]);
    state[2] = InvSubBytes(temp[2]);
    state[3] = InvSubBytes(temp[3]);
#ifdef DEBUG
    printf("is_box:  %.8x %.8x %.8x %.8x\n", state[0], state[1], state[2], state[3]);
    printf("ik_sch:  %.8x %.8x %.8x %.8x\n", w[0], w[1], w[2], w[3]);
#endif
    out[0] = state[0] ^ w[0];
    out[1] = state[1] ^ w[1];
    out[2] = state[2] ^ w[2];
    out[3] = state[3] ^ w[3];
#ifdef DEBUG
    printf("ioutput: %.8x %.8x %.8x %.8x\n", out[0], out[1], out[2], out[3]);
#endif
    cout[0] = out[0] >> 24 & 0xff;
    cout[1] = out[0] >> 16 & 0xff;
    cout[2] = out[0] >> 8 & 0xff;
    cout[3] = out[0] & 0xff;
    cout[4] = out[1] >> 24 & 0xff;
    cout[5] = out[1] >> 16 & 0xff;
    cout[6] = out[1] >> 8 & 0xff;
    cout[7] = out[1] & 0xff;
    cout[8] = out[2] >> 24 & 0xff;
    cout[9] = out[2] >> 16 & 0xff;
    cout[10] = out[2] >> 8 & 0xff;
    cout[11] = out[2] & 0xff;
    cout[12] = out[3] >> 24 & 0xff;
    cout[13] = out[3] >> 16 & 0xff;
    cout[14] = out[3] >> 8 & 0xff;
    cout[15] = out[3] & 0xff;
}

void KeyExpansion(unsigned char ckey[], int nbits, unsigned int w[])
{
    int i, j;
    unsigned int temp, tempx;

    j = nbits / 32;
    for (i = 0; i < j; i++)
        w[i] = ckey[4 * i] << 24 | ckey[4 * i + 1] << 16 | ckey[4 * i + 2] << 8 | ckey[4 * i + 3];

    for (i = j; i < 4 * j + 28; i++)
    {
        temp = w[i - 1];
        if ((i % j) == 0)
        {
            tempx = temp << 8 | (temp >> 24 & 0xff);
            temp = SubBytes(tempx) ^ Rcon[i / j - 1];
        }
        else if (j == 8 && (i % j) == 4)
            temp = SubBytes(temp);
        w[i] = w[i - j] ^ temp;
    }
}
