/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

// SDK Included Files
#include <stdio.h>
#include <stdlib.h>
#include "fsl_debug_console.h"
#include "qcom_api.h"
#include "wlan_qcom.h"
#include "wlan_qcom_tftp.h"


// ============================================================================
// Defines
// ============================================================================
// Convert IP address in uint32_t to comma separated bytes
#define UINT32_IPADDR_TO_CSV_BYTES(a) (((a) >> 24) & 0xFF), (((a) >> 16) & 0xFF), (((a) >> 8) & 0xFF), ((a)&0xFF)
// Convert comma separated bytes to a uint32_t IP address
#define CSV_BYTES_TO_UINT32_IPADDR(a0, a1, a2, a3) \
    (((uint32_t)(a0)&0xFF) << 24) | (((uint32_t)(a1)&0xFF) << 16) | (((uint32_t)(a2)&0xFF) << 8) | ((uint32_t)(a3)&0xFF)


static int _traceQcomApi = 0;
static void printError(uint32_t value, const char *funcName)
{
    PRINTF("ERROR: %s() returned %d\r\n", funcName, value);
}
static int isValueFailed(int32_t value, int32_t failValue, const char *funcName)
{
    if (value == failValue)
    {
        printError(value, funcName);
    }
    else if (_traceQcomApi)
    {
        PRINTF("%s() OK\r\n", funcName);
    }
    return (value == failValue);
}

// ============================================================================
// TFTP 
// ============================================================================
// IPv4 address of remote peer
#define TFTP_SERVER_IPv4_ADDR0 192
#define TFTP_SERVER_IPv4_ADDR1 168
#define TFTP_SERVER_IPv4_ADDR2 100
#define TFTP_SERVER_IPv4_ADDR3 116
#define TFTP_SERVER_PORT 69

#define OPCODE_LEN 2U
#define	RES0_LEN 1U
#define RES1_LEN 1U

#define PRQ_MODE "octet"
//char *tftp_fname = "firmware.bin";
//char *tftp_mode = "octet";
int32_t tftpSock = 0;
static SOCKADDR_T addr;
uint8_t tftpIPAddr[4];

static int isQcomError(A_STATUS status, const char *funcName)
{
    if (status != A_OK)
    {
        printError(status, funcName);
    }
    else if (_traceQcomApi)
    {
        PRINTF("%s() OK\r\n", funcName);
    }
    return (status != A_OK);
}

int _isTFTPPinged = 0;
/** Ping tftp server ip
 */
uint32_t pingTFTP(void)
{
		_isTFTPPinged = 0;
    // Default is to ping the gateway
    uint32_t addr = CSV_BYTES_TO_UINT32_IPADDR(tftpIPAddr[0],tftpIPAddr[1],tftpIPAddr[2],tftpIPAddr[3]);

    // NOTE: qcom_ping() is a blocking function and has no timeout so it will
    PRINTF("Pinging %d.%d.%d.%d... ", UINT32_IPADDR_TO_CSV_BYTES(addr));
    uint32_t t = A_TIME_GET_MSEC();
    A_STATUS status = qcom_ping(addr, 10);
    uint32_t elapsed = status;
    if (status == 0)
    {
				_isTFTPPinged = 1;
        elapsed = A_TIME_GET_MSEC() - t;
        PRINTF("OK (%d ms)\r\n", elapsed);
    }
    isQcomError(status, "qcom_ping");
    return elapsed;
}
int isTFTPPinged(void)
{
		return _isTFTPPinged;
}



int tftpPRQ(char *file_name)
{	   
		uint32_t pack_len, file_len, mode_len;
		char *pack_buf; 
	
		memset(&addr, 0, sizeof(addr));
    addr.sin_family = ATH_AF_INET;
    addr.sin_addr.s_addr = CSV_BYTES_TO_UINT32_IPADDR(tftpIPAddr[0], tftpIPAddr[1],tftpIPAddr[2],tftpIPAddr[3]);
    addr.sin_port = TFTP_SERVER_PORT;
	
		file_len = strlen(file_name);
		mode_len = strlen(PRQ_MODE);
		pack_len = OPCODE_LEN + file_len + RES0_LEN + mode_len + RES1_LEN;
		pack_buf = custom_alloc(pack_len);
    if (pack_buf == NULL)
    {
        printError(0, "custom_alloc");
        return -1;
    }
	
		//create a socket on udp
		if(!(tftpSock > 0))//if no tftp socket created yet
		{
				tftpSock = qcom_socket(ATH_PF_INET, SOCK_DGRAM_TYPE, 0);
				if (isValueFailed(tftpSock, -1, "qcom_socket"))
				{
						custom_free(pack_buf);
						return -1;
				}
		}
		
		//create PRQ pack
		pack_buf[0] = 0x0;
		pack_buf[1] = OPCODE_PRQ;

		memcpy(pack_buf + OPCODE_LEN, file_name, file_len);
		
		pack_buf[OPCODE_LEN+file_len] = 0;
		
		memcpy(pack_buf+OPCODE_LEN+file_len+RES0_LEN, PRQ_MODE, mode_len);
		
		pack_buf[OPCODE_LEN+file_len+RES0_LEN+mode_len] = 0;
				
		//send PRQ pack
		int sent = qcom_sendto(tftpSock, pack_buf, pack_len, 0, (struct sockaddr *)&addr, sizeof(addr));

		// After sending the buffer, remember to free it!
    custom_free(pack_buf);
		
		return sent;
}

int tftpGet(int timeout, char * *buf, int *buf_len)
{			
		socklen_t remAddrLen;
	
		// Return immediately if tftp socket hasn't been created yet
    if (tftpSock == 0)
        return -9;

		
    // NOTE: There is no qcom_select() function, only t_select() that polls only one socket
    // NOTE: t_select() does NOT take a devId like all the other functions
    QCA_CONTEXT_STRUCT *enetCtx = wlan_get_context();
    int32_t status = t_select(enetCtx, tftpSock, timeout);
    if (status == -1)
        return A_ERROR;

    // Free the ZERO_COPY receive buffer (from previous call of this function)
    if (*buf != NULL)
    {
        zero_copy_free(*buf);
        *buf = NULL;
//        buf = 0;
    }

    // NOTE: when QCA400x library has been compiled with ZERO_COPY, the recvBuf
    // is allocated by the qcom_recv() function and the user must free it afterwards!
    // NOTE: qcom_recvfrom() buffer argument is of type char although it should be void
    int received = qcom_recvfrom(tftpSock, buf, BLOCK_SIZE+4, 0, (struct sockaddr *)&addr, &remAddrLen);
    if (received <= 0)
    {
				PRINTF("qcom_recvfrom() ERROR: %d  ", received);
				return A_ERROR;
    }
		*buf_len = received;
		
		short block_no = (*(*buf+2)<<8) + *(*buf+3);
		if(block_no == 1)
		{
			PRINTF("Received bytes from %d.%d.%d.%d:%d...\r\nBlock:", 
             UINT32_IPADDR_TO_CSV_BYTES(addr.sin_addr.s_addr), addr.sin_port);
		}
		PRINTF(" %d", block_no);
		if(block_no%20==0)//change row when what printed reaches to 20 
		{
			PRINTF("\r\n");
		}
		
		return 0;
}

int tftpACK(char *block_no)
{
		char *pack_buf;
		uint32_t pack_len;
	
		pack_len = 4;
		pack_buf = custom_alloc(pack_len);
    if (pack_buf == NULL)
    {
        printError(0, "custom_alloc");
        return -1;
    }
		
		//ack opcode
		pack_buf[0] = 0x0;
		pack_buf[1] = OPCODE_ACK;
		
		//block No.
		pack_buf[2] = block_no[0];
		pack_buf[3] = block_no[1];	
		
		//send PRQ pack
//		PRINTF("\r\nack to %d.%d.%d.%d:%d...\r\n", 
//             UINT32_IPADDR_TO_CSV_BYTES(addr.sin_addr.s_addr), addr.sin_port);
		int sent = qcom_sendto(tftpSock, pack_buf, pack_len, 0, (struct sockaddr *)&addr, sizeof(addr));
		
		// After sending the buffer, remember to free it!
    custom_free(pack_buf);
		
		return sent;
}

void tftpStop(void)
{
		qcom_socket_close(tftpSock);
		tftpSock = 0;
}

void tftpErr(short err_no)
{
		//reserve
}
