/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>

#include "fsl_common.h"
#include "board.h"

#include "i2c_comm.h"
#include "spi_comm.h"
#include "gpio_ctrl.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CMD_I2C_SLAVE_ADDRESS  0x45 /* 7-bit address. 100_1100. */
#define CMD_I2C_SLAVE_INSTANCE   I2C0
#define CMD_I2C_SLAVE_XFER_BUFF_LEN  (200u)

/* i2c slave cmd list. */
#define CMD_I2C_SLAVE_CMD_WRITE_BUFF  0x0F
#define CMD_I2C_SLAVE_CMD_CONF_SPI  0xF0
#define CMD_I2C_SLAVE_CMD_CLEAR_INT 0xF1
#define CMD_I2C_SLAVE_CMD_IDLE_MODE 0xF2
#define CMD_I2C_SLAVE_CMD_GPIO_WRITE 0xF4
#define CMD_I2C_SLAVE_CMD_GPIO_READ  0xF5
#define CMD_I2C_SLAVE_CMD_GPIO_ENABLE 0xF6
#define CMD_I2C_SLAVE_CMD_GPIO_CONF 0xF7

#define CMD_SPI_MASTER_INSTANCE   SPI0



#define DEBUG_ENABLE_LOG  1

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void cmd_i2c_slave_init(void);

void cmd_spi_master_init(void);
void cmd_spi_master_xfer_done_callback(void *param);

/*******************************************************************************
 * Variables.
 ******************************************************************************/


uint8_t cmd_i2c_slave_xfer_buff[CMD_I2C_SLAVE_XFER_BUFF_LEN]; /* the func code is at the first byte. */
spi_master_xfer_handler_t  cmd_spi_master_xfer_handler_struct;

uint8_t cmd_gpio_pinmask_for_spi_cs = 0x0f; /* controlled by spi command. */
uint8_t cmd_gpio_pinmask_for_gpio = 0x0; /* controlled by gpio commands. */

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    board_init();
    printf("app i2c slave.\r\n");

    gpio_int_pin_init();

    gpio_cs_pin_init( (1u << GPIO_CS_PIN_NUM) - 1u);
    cmd_gpio_pinmask_for_spi_cs = (1u << GPIO_CS_PIN_NUM) - 1u;
    cmd_gpio_pinmask_for_gpio = 0u;

    cmd_spi_master_init();
    cmd_i2c_slave_init();

    while (1)
    {
    }
}


void cmd_i2c_slave_xfer_done_callback(void *param1)
{
    i2c_slave_xfer_done_callback_param_t *xfer_done_param_ptr = (i2c_slave_xfer_done_callback_param_t *)param1;
    uint32_t flags = xfer_done_param_ptr->flags;
    uint8_t rx_cmd = xfer_done_param_ptr->rx_cmd;
    uint8_t *xfer_data = xfer_done_param_ptr->xfer_data;
    uint32_t xfer_len = xfer_done_param_ptr->xfer_len;

    /* receive. */
    if (flags & I2C_SLAVE_XFER_DONE_FLAG_RECEIVE_DATA)
    {
        /* i2c slaveӿڽ, ͨspi master ӿת */

        /* the 1st item from rx is function cmd. */
        switch ( rx_cmd )
        {
            case CMD_I2C_SLAVE_CMD_CONF_SPI:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_CONF_SPI: 0x%-2X.\r\n",xfer_data[0]);
#endif /* DEBUG_ENABLE_LOG */

                uint8_t bitorder = (uint8_t)((xfer_data[0] >> 5u) & 0x1);
                uint8_t cpolcpha = (uint8_t)((xfer_data[0] >> 2u) & 0x3);
                uint8_t speed = (uint8_t)(xfer_data[0] & 0x3);

                spi_conf_master(CMD_SPI_MASTER_INSTANCE,
                                (spi_speed_t)speed,
                                (spi_cpolcpha_t)cpolcpha,
                                (spi_bitorder_t)bitorder );

                break;

            case CMD_I2C_SLAVE_CMD_CLEAR_INT:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_CLEAR_INT\r\n");
#endif /* DEBUG_ENABLE_LOG */

                gpio_int_pin_clear();

                break;

            case CMD_I2C_SLAVE_CMD_IDLE_MODE:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_IDLE_MODE\r\n");
#endif /* DEBUG_ENABLE_LOG */

                POWER_EnterSleep();

                break;

            case CMD_I2C_SLAVE_CMD_GPIO_WRITE:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_GPIO_WRITE\r\n");
#endif /* DEBUG_ENABLE_LOG */

                uint8_t pin_vals = xfer_data[0];
                gpio_cs_pin_write_1(pin_vals);
                gpio_cs_pin_write_0(~pin_vals);

                break;

            case CMD_I2C_SLAVE_CMD_GPIO_READ:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_GPIO_READ\r\n");
#endif /* DEBUG_ENABLE_LOG */

                xfer_data[0] = gpio_cs_pin_read(cmd_gpio_pinmask_for_gpio);

                break;

            case CMD_I2C_SLAVE_CMD_GPIO_ENABLE:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_GPIO_ENABLE\r\n");
#endif /* DEBUG_ENABLE_LOG */

                /* spi would not control the cs pins when they are enabled as gpio. */
                cmd_gpio_pinmask_for_spi_cs = ~xfer_data[0];
                cmd_gpio_pinmask_for_gpio = xfer_data[0];

                break;

            case CMD_I2C_SLAVE_CMD_GPIO_CONF:
#if DEBUG_ENABLE_LOG
                printf("CMD_I2C_SLAVE_CMD_GPIO_CONF\r\n");
#endif /* DEBUG_ENABLE_LOG */

                for (uint8_t i = 0u; i < GPIO_CS_PIN_NUM; i++)
                {
                    uint8_t pin_mode = (xfer_data[0] >> (i*2)) & 0x3;
                    gpio_cs_pin_set_pin_mode(i, (gpio_cs_pin_mode_t)pin_mode);
                }

                break;

            default:
                if (rx_cmd <= CMD_I2C_SLAVE_CMD_WRITE_BUFF)
                {
#if DEBUG_ENABLE_LOG
                    printf("CMD_I2C_SLAVE_CMD_WRITE_BUFF: 0x%-2X, %d\r\n", rx_cmd, xfer_len);
#endif /* DEBUG_ENABLE_LOG */
                    cmd_spi_master_xfer_handler_struct.cs_mask = (rx_cmd & cmd_gpio_pinmask_for_spi_cs);
                    gpio_cs_pin_write_0(cmd_spi_master_xfer_handler_struct.cs_mask);

                    cmd_spi_master_xfer_handler_struct.buff_len = xfer_len;
                    cmd_spi_master_xfer_handler_struct.rx_buff = cmd_i2c_slave_xfer_buff;
                    cmd_spi_master_xfer_handler_struct.tx_buff = cmd_i2c_slave_xfer_buff;
                    cmd_spi_master_xfer_handler_struct.xfer_done_callback = cmd_spi_master_xfer_done_callback;
                    spi_master_start_xfer(CMD_SPI_MASTER_INSTANCE, &cmd_spi_master_xfer_handler_struct);
                }
                break;
        }
    }

    /* transmit. */
    if (flags & I2C_SLAVE_XFER_DONE_FLAG_TRANSMIT_DATA)
    {
#if DEBUG_ENABLE_LOG
      printf("tx done: %d\r\n", xfer_len);
#endif /*  DEBUG_ENABLE_LOG*/

    }
}

void cmd_i2c_slave_init(void)
{
    i2c_init_slave(CMD_I2C_SLAVE_INSTANCE, CMD_I2C_SLAVE_ADDRESS, cmd_i2c_slave_xfer_buff, CMD_I2C_SLAVE_XFER_BUFF_LEN);
    i2c_slave_install_xfer_done_callback(CMD_I2C_SLAVE_INSTANCE, cmd_i2c_slave_xfer_done_callback);
}

void cmd_spi_master_init(void)
{
    spi_init_master(CMD_SPI_MASTER_INSTANCE);
    spi_conf_master(CMD_SPI_MASTER_INSTANCE, eSPI_Speed_58k, eSPI_CpolCpha_00, eSPI_BitOrder_MSB);
}

void cmd_spi_master_xfer_done_callback(void *param)
{
    spi_master_xfer_handler_t * xfer_handler = (spi_master_xfer_handler_t *)param;
    gpio_cs_pin_write_1(xfer_handler->cs_mask);

    gpio_int_pin_set(); /* assert an irq signal to master when spi master xfer is done. */
}

void SPI0_IRQHandler(void)
{
    spi_master_isr_hook(CMD_SPI_MASTER_INSTANCE, &cmd_spi_master_xfer_handler_struct);
}

/* EOF. */

