;/******************************************************************************
;* 
;* Copyright (c) 2009 Freescale Semiconductor;
;* All Rights Reserved                       
;*
;*******************************************************************************
;*
;* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
;* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
;* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
;* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
;* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
;* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
;* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
;* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
;* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
;* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
;* THE POSSIBILITY OF SUCH DAMAGE.
;*
;***************************************************************************//*!
;*
;* @file      intrinsic_math.c
;*
;* @author    r30322
;* 
;* @version   1.0.3.0
;* 
;* @date      Oct-15-2008
;* 
;* @brief     basic mathematical functions assembler file
;*
;* target     HCS08
;*
;* tested on compiler:  
;*            CodeWarrior Development Studio for Microcontrollers V6.2 Release
;*
;*******************************************************************************/

;/*****************************************************************************
;*   External Symbol Definition
;*****************************************************************************/
 XDEF udiv16_8to8
 XDEF udiv16i8to16
 XDEF udiv32i8to16
 XDEF udiv16_16to8
 XDEF udiv16

 XDEF add24_mult16_8_24
 XDEF sub24_mult16_8_24
 XDEF add16
 XDEF uadd8
 XDEF uadd16
 XDEF sub16yx
 XDEF subu16_u16to16
 XDEF usub8
 XDEF usub16yx
 XDEF uadd32
 XDEF add32
 XDEF usub32
 XDEF sub32
 XDEF uadd32_16_32
 XDEF add32_16_32

 XDEF neg8
 XDEF neg16
 XDEF abs16

 XDEF umul16_8to32
 XDEF shl16_u8to16
 XDEF shr16_u8to16
 
 XDEF shl32_u8to32
 XDEF shr32_u8to32
 
 XDEF irq_disable
 XDEF irq_restore



;/*****************************************************************************
;*
;* Module: unsigned char udiv16_8to8(unsigned int x, unsigned char y)
;*
;* Description:
;*     Unsigned dividing 16 bit by 8 bit Ubyte.
;*     The Result is saturated at 0xFF if overflow occures.
;*
;* Returns:   x/y
;*
;* Arguments: x (in) H->H, L->X
;*            y (in) A
;*
;* Range Issues: if y=0 the result is saturated at 0xFF
;*
;* Special Issues:  HCS08S -Cs08 Option backend.The result is saturated
;*
;*****************************************************************************/
udiv16_8to8:               ; 2,X-xH Byte,  3,X-xL Byte, Accum y
           TAX              ;2 y -> X
           PSHX             ;2 x Low -> (SP)
           PULA             ;3 x Low ->A

           DIV              ; sets carry if overflow occured
           BCC  okdiv168    ;/* if not overflow */
zerodiv168:LDA  #0FFh
okdiv168:  RTS


;Debug Note - very usefull function
;/*****************************************************************************
;*
;* Module: unsigned int udiv16i8to16(unsigned int x, unsigned char y)
;*
;* Description: unsigned word and unsigned byte division, return unsigned word
;*   
;*
;* Returns: x/y
;*
;* Global Data:None
;*
;* Arguments: x (in) Low->X, High->H
;*            y (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend.
;*
;*****************************************************************************/
udiv16i8to16:
        PSHX        ; x Low -> @(SP)
        TAX         ; y -> X
        PSHH        ; x
        PULA        ; x High -> A
        CLRH
        DIV         ; 0, x High / y 
        PSHA        ; save out result high byte        
        LDA 2,SP    ; x Low -> A
        DIV
        TAX         ; result Low -> X
        PULH        ; result High -> H
        RTS

;/*****************************************************************************
;*
;* Module: unsigned int udiv32i8to16(unsigned long x, unsigned char y)
;*
;* Description:
;*     Unsigned integer dividing 32 bit by 8 bit from 16 bit Uword. 
;*     The Result is saturated at 0xFFFF if overflow occures.
;*
;* Returns:   x/y
;* Returns:   Out3:Out2:Out1:Out0 = (x3:x2:x1:x0)/(y0) = x/y
;*
;* Arguments: y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;
;*            y (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
udiv32i8to16:
        PSHA        ;
        TAX         ; y -> X
        TST 4,SP    ;
        BNE udiv32Sat
        LDA 5, SP
        PSHA
        PULH
        LDA 6, SP
        DIV         ;
        BCS udiv32Sat
        PSHA
        LDA 8, SP
        DIV
        TAX         ;z0 -> X
        PULH        ;z1 -> H
        PULA
        RTS

udiv32Sat:
        LDHX #0FFFFh
        PULA
        RTS

;/*****************************************************************************
;*
;* Module: unsigned char udiv16_16to8(unsigned int x, unsigned int y)
;*
;* Description:
;*     Unsigned dividing 16 bit by high 8 bit from 16 bit Uword.
;*     Both divisor and divident are scaled  to get high results precision. 
;*     The Result is saturated at 0xFF if overflow occures.
;*
;* Returns:   x*256/y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X;   High->H
;*
;* Range Issues: if y=0 the result is saturated at 0xFF
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
udiv16_16to8:
        PSHH
        PULA
        TSTX             ;/* if (y=0 return 0xFF)*/
        BNE  scalein
        TSTA
        BEQ  zerodiv
scalein:LSLX             ;/* scale divisor      */
        ROLA
        BCS  rangeout    ;/* the divisor>0xFFFF */
        LSL  4,SP        ;/* scale divident     */
        ROL  3,SP
        BCC  scalein     ;/* divident<=0xFFFF   */
        ROR  3,SP
        ROR  4,SP
        CLC
rangeout:                ;/* make division */
        RORA
        TAX
        LDA  3,SP
        PSHA
        PULH
        LDA  4,SP
        DIV
        BCC  divok       ;/* if not overflow */
zerodiv:LDA  #0FFh
divok:  RTS

;/*****************************************************************************
;*
;* Module: unsigned int udiv16(unsigned int x, unsigned int y)
;*
;* Description:
;*     Unsigned dividing 16 bit by 16 bit from 16 bit Uword. 
;*     The Result is saturated at 0xFFFF if overflow occures.
;*
;* Returns:   Out1:Out0 = (x3:x2:00:00)/(y1:y0) = x*256*256/y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X;   High->H
;*
;* Range Issues: if y=0 the result is saturated at 0xFF
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*                 the result precission is  0..+2LSB
;*
;*****************************************************************************/
udiv16:
        ;* Arguments: x (in) x2=x Low->4,SP x3=x High->3,SP
        ;*            y (in) y0=y Low->X;   y1=y High->H        
        PSHX            ;(SP)<-y Low = y0
        PSHH            ;(SP)<-y High = y1
        LDA 6,SP
        PSHA            ;(SP)<-x Low = x2
        LDA 6,SP
        PSHA            ;(SP)<-x Low = x3
        JSR udiv16_16to8;
        AIS #2          ;clear last stacks
        
        PSHA            ;(SP)<-z1 3
        LDHX 2,SP        ;H<-y High = y1, X<-y Low = y0
        JSR mult16_8to24    ;/* y1,y0 *z1 */

        ;/* A<-Out2, H<-Out1, X<-Out0 */
        ;/* rem2:rem1:rem0 = x3:x2:0 - y1:y0*z1 = x Low */
remaindr:PSHX            ;(SP) Out0  6
        PSHH            ;(SP) Out1
        PSHA            ;(SP) Out2

        CLRA            ;x1 = 0
        SUB 3,SP        ;x1-Out0
        PSHA            ;(SP) rem1* 7
        LDA 11,SP        ;x2
        SBC 3,SP        ;x2-Out1
        PSHA            ;(SP) rem2* 8
        LDA 11,SP       ;x3
        SBC 3,SP        ;x3-Out2
        ;PSHA           ;(SP) rem3*
        BCC tstsatd16  ;no result correction

corZrem:DEC 6,SP        ;z1 = dec z1*
        LDA 2,SP       ;(SP) rem1*
        ADD 8,SP       ;y0
        STA 2,SP       ;(SP) rem1
        LDA 1,SP       ;(SP) rem2*
        ADC 7,SP       ;y1
        STA 1,SP       ;(SP) rem2
        BCS remadiv  ;one result correction enough

corZrem2:DEC 6,SP       ;z1 = dec z1*
        LDA 2,SP       ;(SP) rem1*
        ADD 8,SP       ;y0
        STA 2,SP       ;(SP) rem1
        LDA 1,SP       ;(SP) rem2*
        ADC 7,SP       ;y1
        STA 1,SP       ;(SP) rem2

        ;* Arguments: rem2->2,SP rem1->1,SP
        ;*            y (in) y0->X;   y1->H
        ;* (rem2:rem1:0)(y1:y0)
remadiv:LDHX  1,SP      ;HL<-rem2:rem1 
        PSHX
        PSHH
        LDHX  9,SP      ;HL y1:y0
        JSR udiv16_16to8;
        
        TAX             ;z0<-X
        LDA 8,SP
        AIS #10         ;stack "clean-up"
        PSHA
        PULH            ;z1 <-H
        RTS


tstsatd16:BEQ remadiv ;if rem3 is zero 
          LDX  #0FFh  ;if rem3 is not zero but no rem3 underflow 
          PSHX
          PULH
          AIS #8
          RTS

;/*****************************************************************************
;*
;* Module: mult16_8to24
;*
;* Description:
;*     Unsigned multiplication 16 bit by 8 bit. 
;*     The Result is 24 bit.
;*
;* Returns:   Out2:Out1:Out0 = y1:y0*z
;*
;* Arguments: y (in) y1->H, y0->X
;*            z (in) A
;*
;* Range Issues: 24 bit output
;* 
;* Special Issues: IS NOT NOT COMPATIBLE with HCS08S -Cs08 Option backend. 
;*                 Due to 24 bit output
;*
;* Output: A<-Out2, H<-Out1, X<-Out0
;*
;*****************************************************************************/
mult16_8to24:
                            ;z1->A
        PSHA                ;z1 ->(SP) 1
        PSHX                ;y0 ->(SP) 2
        PSHH                ;y1 ->(SP) 3
        PULX                ;y1->X
        MUL                 ;X:A=y1*z1
        PSHA                ;TempA1
        PSHX                ;TempA2
        
        LDA 4,SP           ;A<-z1
        LDX 3,SP           ;X<-y0
        MUL                ;X:A=y0*z1
        
        PSHA                ;(SP)<-Out0
        ;PULH                ;H<-TempB0 = Out0

        TXA                 ;A<-TempB1
        ADD 3,SP            ;TempB1+TempA1
        PSHA                ;(SP)<-Out1

        CLRA
        ADC 3,SP            ;A <- Out2 = 0+TempA2  (TempB2 = 0)
        ;PSHA
        PULH                ;X<-Out1
        PULX                ;X<-Out0
        AIS  #4
        RTS                 ;/* A<-Out2, H<-Out1, X<-Out0 */




        ;09.07.29 LDHX 2,SP        ;H<-y High = y1, X<-y Low = y0
        ;JSR mult16_8to24    ;/* y1,y0 *z1 */

;* Arguments: y (in) y1->4,SP, y0->5,SP
;*            z (in) 3,SP and A

mult16_8to24v2:
        LDA 3,SP;09.07.29 
        LDX 4,SP;09.07.29 
        
        MUL                 ;X*A=y1*z1
        PSHA                ;TempA1
        PSHX                ;TempA2
        
        LDA 4,SP           ;A<-z1
        LDX 3,SP           ;X<-y0
        MUL                ;X:A=y0*z1
        
        PSHA                ;(SP)<-Out0
        ;PULH                ;H<-TempB0 = Out0

        TXA                 ;A<-TempB1
        ADD 3,SP            ;TempB1+TempA1
        PSHA                ;(SP)<-Out1

        CLRA
        ADC 3,SP            ;A <- Out2 = 0+TempA2  (TempB2 = 0)

        ;/* A<-Out2, H<-Out1, X<-Out0 */    
        ;09.07.29 PSHX            ;(SP) Out0  6
        ;09.07.29 PSHH            ;(SP) Out1
        PSHA            ;(SP) Out2
        JMP remaindr


;/*****************************************************************************
;*
;* Module: add24_mult16_8_24
;*
;* Description:
;*     Unsigned addition of 24bit register with multiplication 16 bit by 8 bit. 
;*     The Result is 24 bit.
;*
;* Result:   z2:z1:z0 = z2:z1:z0 + (x1:x0)*y0
;*
;* Arguments: address z Low->4,SP High->3,SP
;*            x (in) Low->X;   High->H
;*            y -> A
;*
;* Range Issues: zSum24 pointer to 24 bit structure
;* 
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated to ffffff
;*
;* Output: Result is in UWord24H16L8 *zSum24
;*
;*****************************************************************************/
add24_mult16_8_24:
        JSR mult16_8to24    ;/* y1,y0 *z1 */
        ;/* A<-Out2, H<-Out1, X<-Out0 */
        ;/* rem2:rem1:rem0 = x3:x2:0 - y1:y0*z1 = x Low */

        ; PSHX            ;(SP) Out0  6
        PSHH            ;(SP) Out1
        PSHA            ;(SP) Out2

        TXA
        LDHX 5,SP
        ADD 2,X         ;z0+Out0
        STA 2,X         ;z0 = z0+Out0
        LDA 2,SP        ;Out1
        ADC 1,X         ;z1+Out1
        STA 1,X         ;z1 = z1+Out1
        LDA 1,SP        ;Out2
        ADC 0,X         ;z2+Out2
        STA 0,X         ;z2 = z2+Out2
        BCC rtsaddmult   ;no result correction

        LDA  #0FFh      ;saturate when overflow
        STA 2,X         ;z0 = ff
        STA 1,X         ;z1 = ff
        STA 0,X         ;z2 = ff
rtsaddmult:
        AIS #2
        RTS

;/*****************************************************************************
;*
;* Module: sub24_mult16_8_24
;*
;* Description:
;*     Unsigned substraction of 24bit register with multiplication 16 bit by 8 bit. 
;*     The Result is 24 bit.
;*
;* Result:   z2:z1:z0 = z2:z1:z0 + (x1:x0)*y0
;*
;* Arguments: address z Low->4,SP High->3,SP
;*            x (in) Low->X;   High->H
;*            y -> A
;*
;* Range Issues: zSum24 pointer to 24 bit structure
;* 
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated to 000000
;*
;* Output: Result is in UWord24H16L8 *zSum24
;*
;*****************************************************************************/    
sub24_mult16_8_24:
        JSR mult16_8to24    ;/* y1,y0 *z1 */
        ;/* A<-Out2, H<-Out1, X<-Out0 */
        ;/* rem2:rem1:rem0 = x3:x2:0 - y1:y0*z1 = x Low */

        PSHX            ;(SP) Out0  6
        PSHH            ;(SP) Out1
        PSHA            ;(SP) Out2

        TXA
        LDHX 6,SP
        LDA 2,X
        SUB 3,SP        ;z0+Out0
        STA 2,X         ;z0 = z0+Out0
        LDA 1,X         ;Out1
        SBC 2,SP        ;z1+Out1
        STA 1,X         ;z1 = z1+Out1
        LDA 0,X         ;Out2
        SBC 1,SP        ;z2+Out2
        STA 0,X         ;z2 = z2+Out2
        BCC rtsubmult   ;no result correction

        CLRA            ;saturate when overflow
        STA 2,X         ;z0 = ff
        STA 1,X         ;z1 = ff
        STA 0,X         ;z2 = ff
rtsubmult:
        AIS #3
        RTS

;/*****************************************************************************
;*
;* Module: signed int add16(signed int x, signed int y);
;*
;* Description:
;*   The function performs the addition x+y with overflow control
;*   and saturation. The 16-bit result is set at +32767 when overflow occurs,
;*   or at -32768 when underflow occurs.
;*
;* Returns:   x + y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X, High->H
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
;  /* (V - Overflow Flag, N - Negative flag) */
;  /* V=1 & N=1 too high */
;  /* V=1 & N=0 too low  */ 
add16:    TXA             ; /* transfer X - low byte to A */
          ADD   4,SP      ; /* add low byte from stack */
          PSHA            ; /* push result to stack */
          PSHH            ; /* push H - high byte to stack */
          PULA            ; /* move high byte from stack to A */
          ADC   4,SP      ; /* add high byte with carry */
          PSHA            ; /* store result */
          PULH            ; /* move result high byte to H */
          PULX            ; /* move result low byte to X */
          BLT   addtlow   ; /* if V=1 or N=1, can be too low */
          BPL   addret1   ; /* if N=0 */
          LDHX  #7FFFh    ; /* set saturation value */
addret1:  RTS             ; /* return from subroutine */
addtlow:  BMI   addret2   ; /* if N=1 */
          LDHX  #8000h	  ; /* set saturation value */
addret2:  RTS             ; /* return from subroutine */


;/*****************************************************************************
;*
;* Module: unsigned char uadd8(unsigned char x, unsigned char y)
;*
;* Description:
;*   The function performs the unsigned addition x+y with overflow control
;*   and saturation. The 8-bit result is set at +255 when overflow occurs,
;*
;* Returns:   x + y
;*
;* Arguments: x (in) X
;*            y (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/ 
uadd8:          PSHX
                TSX
                ADD  ,X         ;          /* x + y */
                PULH
                BCC  uad8uuret1 ;C=0
                LDA  #0FFh      ;C=1 /* too high */
uad8uuret1:     RTS


;/*****************************************************************************
;*
;* Module: unsigned int uadd16(unsigned int x, unsigned int y)
;*
;* Description:
;*   The function performs the addition x+y with overflow control
;*   and saturation. The 16-bit result is set at +65535 when overflow occurs,
;*
;* Returns:   x + y
;*
;* Arguments: x (in) L->4,SP H->3,SP
;*            y (in) L->X;   H->H
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
uadd16:         TXA
                ADD  4,SP       ;          /* x + y */
                TAX
                PSHH
                PULA
                ADC  3,SP
                PSHA
                PULH
                BCC  uaduuret1  ;C=0      /* no overflow */
                LDX  #0FFh      ;C=1      /* too high */
                LDA  #0FFh
                PSHA
                PULH
uaduuret1:      RTS



;/*****************************************************************************
;*
;* Module: signed int sub16yx(signed int x, signed int y)
;*
;* Description:
;*   The function performs the 16-bit subtraction with overflow control and saturation. 
;*   The 16-bit result is set at +32767 on overflow, or at -32768 on underflow
;*
;* Returns:   x - y
;*
;* Arguments: x (in) L->4,SP H->3,SP
;*            y (in) L->X;   H->H
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/

sub16yx:  TXA             ; /* transfer X - low byte to A */
          SUB   4,SP      ; /* add low byte from stack */
          PSHA            ; /* push result to stack */
          PSHH            ; /* push H - high byte to stack */
          PULA            ; /* move high byte from stack to A */
          SBC   4,SP      ; /* add high byte with carry */
          PSHA            ; /* store result */
          PULH            ; /* move result high byte to H */
          PULX            ; /* move result low byte to X */
          BLT   subtlow   ; /* if V=1 or N=1, can be too low */
          BPL   subret1   ; /* if N=0 *
          LDHX  #7FFFh    ; /* set saturation value */
subret1:  RTS             ; /* return from subroutine */
subtlow:  BMI   subret2   ; /* if N=1 */
          LDHX  #8000h    ; /* set saturation value */
subret2:  RTS             ; /* return from subroutine */

;/*****************************************************************************
;*
;* Module: signed int subu16_u16to16(unsigned int x, unsigned int y)
;*
;* Description:
;*   The function performs the subtraction of unsigned x-y to signed result 
;*   with overflow control and saturation. The 16-bit result is set at +32767 
;*   when overflow occurs, or at -32768 when underflow occurs.
;*
;* Returns:   x - y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X;   High->H
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated.
;*                 this function uses unsigned inputs to signed result
;*
;*****************************************************************************/

; Debug NOT TESTED YET!!!
subu16_u16to16: PSHX
                PSHH
                TSX
                LDA  5,X        ;x - y
                SUB  1,X
                PSHA
                LDA  4,X
                SBC  ,X
                PULX
                PSHA
                PULH
                AIS  #2
                BCS  subuutlow    ;C=1             /* can be too low */
                
                BPL  subuuret1  ;V=0 & N=0
                LDX  #0FFh      ;V=1 & N=1 /* too high */
                LDA  #07Fh
                PSHA
                PULH
subuuret1:      RTS
subuutlow:      BMI  subuuret2  ;V=0 & N=1
                CLRX            ;V=1 & N=0 /* too low */
                LDA  #080h
                PSHA
                PULH
subuuret2:      RTS

;/*****************************************************************************
;*
;* Module: signed int subu16yxu16to16(unsigned int x, unsigned int y)
;*
;* Description:
;*   The function performs the subtraction of unsigned x-y to signed result 
;*   with overflow control and saturation. The 16-bit result is set at +32767 
;*   when overflow occurs, or at -32768 when underflow occurs.
;*
;* Returns:   x - y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X;   High->H
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated.
;*                 this function uses unsigned inputs to signed result
;*
;*****************************************************************************/

; Debug NOT TESTED YET!!!
subu16yxu16to16: TXA
                SUB  3,SP
                TAX
                PSHH
                PULH
                SBC  4,SP
                PSHA
                PULH
                AIS  #2
;                BLT  subyxuutlow    ;V^N:      /* can be too low */
                BCS  subuutlow      ;C=1        /* can be too low */
                
                BPL  subyxuuret1    ;V=0 & N=0
                LDX  #0FFh          ;V=1 & N=1  /* too high */
                LDA  #07Fh
                PSHA
                PULH
subyxuuret1:    RTS
subyxuutlow:    BMI  subyxuuret2    ;V=0 & N=1
                CLRX                ;V=1 & N=0  /* too low */
                LDA  #080h
                PSHA
                PULH
subyxuuret2:    RTS


;/*****************************************************************************
;*
;* Module: unsigned char usub8(unsigned char x, unsigned char y)
;*
;* Description:
;*   The function performs the subtraction x-y with overflow control
;*   and saturation. The 8-bit result is set at 0 when underflow occurs.
;*
;* Returns:   x - y
;*
;* Arguments: x (in) X
;*            y (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
usub8:      PSHA
            TXA
            TSX
            SUB  ,X         ;x-y
            PULH
            BCC  usb8uuret1 ;C=0
            CLRA            ;C=1 /* too low */      
usb8uuret1: RTS

;/*****************************************************************************
;*
;* Module: unsigned int 16 usub16yx(unsigned int y, unsigned int x) Debug 09.06.03
;*
;* Description:
;*   The function performs the subtraction x-y with overflow control
;*   and saturation. The 16-bit result is set at at +65535 when overflow occurs,
;*   or at 0 when underflow occurs.
;*
;* Returns:   x - y
;*
;* Arguments: x (in) Low->4,SP High->3,SP
;*            y (in) Low->X;   High->H
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
usub16yx:       TXA
                SUB  4,SP       ;        /* x - y */
                TAX
                PSHH
                PULA
                SBC  3,SP
                PSHA
                PULH
                BCC  usubret    ;C=0    /* no overflow */
                CLRX            ;C=1    /* too low */         
                CLRH
usubret:        RTS

        
;/*****************************************************************************
;*
;* Module: unsigned long uadd32(unsigned long x, unsigned long y)
;*
;* Description:
;*    The function performs unsigned 32-bit addition x+y with overflow control 
;*    and saturation. The 32-bit result is set to 0x7fffffff when overflow occurs.
;*
;* Returns:   z = x + y
;*
;* Arguments: *z(in) L->12,SP H->11,SP
;*            
;*            x (in) L1->9,SP L2->10,SP
;*                   H1->7,SP H2->8,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;                   
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/  
uadd32:   LDHX  11,SP     ; /* load *z to H:X */           
          LDA   10,SP     ; /* x-LSB(2) */
          ADD   6,SP      ; /* x-LSB(2) + y-LSB(2) */
          STA   3,X       ; /* LSB(2) -> z */ 
          LDA   9,SP      ; /* x-LSB(1) */
          ADC   5,SP      ; /* x-LSB(1) + y-LSB(1) */
          STA   2,X       ; /* LSB(1) -> z */ 
          LDA   8,SP      ; /* x-MSB(2) */
          ADC   4,SP      ; /* x-MSB(2) + y-MSB(2) */
          STA   1,X       ; /* MSB(2) -> z */ 
          LDA   7,SP      ; /* x-MSB(1) */
          ADC   3,SP      ; /* x-MSB(1) + y-MSB(1) */
          STA   0,X       ; /* MSB(1) -> z */            
          BCC   uadd32ret ; /* if carry clear -> end,if set -> overflow */                        
          LDA   #0FFh     ; /* load accumulator - 0xFF value */
          STA   0,X       ; /* store FF in MSB(1) */
          STA   1,X       ; /* store FF in MSB(2) */
          STA   2,X       ; /* store FF in LSB(1) */
          STA   3,X       ; /* store FF in LSB(2) */
uadd32ret:RTS             ; /* return from subroutine */


;/*****************************************************************************
;*
;* Module: unsigned long add32(unsigned long x, unsigned long y)
;*
;* Description:
;*    The function performs signed 32-bit addition x+y with overflow control 
;*    and saturation. The 32-bit result is set to 0x7FFFFFFF when overflow occurs
;*    and to 0x80000000 when underflow
;*
;* Returns:   z = x + y
;*
;* Arguments: *z(in) L->12,SP H->11,SP
;*            
;*            x (in) L1->9,SP L2->10,SP
;*                   H1->7,SP H2->8,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;                   
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/ 
add32:    LDHX  11,SP     ; /* load *z to H:X */           
          LDA   10,SP     ; /* x-LSB(2) */
          ADD   6,SP      ; /* x-LSB(2) + y-LSB(2) */
          STA   3,X       ; /* LSB(2) -> z */ 
          LDA   9,SP      ; /* x-LSB(1) */
          ADC   5,SP      ; /* x-LSB(1) + y-LSB(1) */
          STA   2,X       ; /* LSB(1) -> z */ 
          LDA   8,SP      ; /* x-MSB(2) */
          ADC   4,SP      ; /* x-MSB(2) + y-MSB(2) */
          STA   1,X       ; /* MSB(2) -> z */ 
          LDA   7,SP      ; /* x-MSB(1) */
          ADC   3,SP      ; /* x-MSB(1) + y-MSB(1) */       
          BLT  add32low    ;V^N:      /* can be too low */
          BPL  add32ret    ;V=0 & N=0       
          LDA   #0FFh     ; /* load accumulator - 0xFF value */
          STA   1,X       ; /* store 0xFF in MSB(2) */
          STA   2,X       ; /* store 0xFF in LSB(1) */
          STA   3,X       ; /* store 0xFF in LSB(2) */
          LDA  #07Fh     ; /* load accumulator - 0x7F value */
add32ret: STA   0,X       ; /* MSB(1) -> z */     
          RTS             ; /* return from subroutine */


add32low: BMI  add32ret   ;V=0 & N=1
                          ;V=1 & N=0 /* too low */
          CLRA            ; /* clear accumulator - 0x0 value */
          STA   1,X       ; /* store 0x0 in MSB(2) */
          STA   2,X       ; /* store 0x0 in LSB(1) */
          STA   3,X       ; /* store 0x0 in LSB(2) */
          LDA  #080h      ; /* load accumulator - 0x80 value */
          STA   0,X       ; /* store 0x80 in MSB(1) */
          RTS             ; /* return from subroutine */


;/*****************************************************************************
;*
;* Module: unsigned long usub32(unsigned long x, unsigned long y)
;*
;* Description:
;*    The function performs unsigned 32-bit substraction x+y with overflow control 
;*    and saturation. The 32-bit result is set to 0 when underflow occurs.
;*
;* Returns:   z = x - y
;*
;* Arguments: *z(in) L->12,SP H->11,SP
;*            
;*            x (in) L1->9,SP L2->10,SP
;*                   H1->7,SP H2->8,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;                   
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/  
usub32:   LDHX  11,SP     ; /* load *z to H:X */           
          LDA   10,SP     ; /* x-LSB(2) */
          SUB   6,SP      ; /* x-LSB(2) + y-LSB(2) */
          STA   3,X       ; /* LSB(2) -> z */ 
          LDA   9,SP      ; /* x-LSB(1) */
          SBC   5,SP      ; /* x-LSB(1) + y-LSB(1) */
          STA   2,X       ; /* LSB(1) -> z */ 
          LDA   8,SP      ; /* x-MSB(2) */
          SBC   4,SP      ; /* x-MSB(2) + y-MSB(2) */
          STA   1,X       ; /* MSB(2) -> z */ 
          LDA   7,SP      ; /* x-MSB(1) */
          SBC   3,SP      ; /* x-MSB(1) + y-MSB(1) */
          STA   0,X       ; /* MSB(1) -> z */            
          BCC   usub32ret ; /* if carry clear -> end,if set -> overflow */                        
          CLRA            ; /* load accumulator - 0x0 value */
          STA   0,X       ; /* store FF in MSB(1) */
          STA   1,X       ; /* store FF in MSB(2) */
          STA   2,X       ; /* store FF in LSB(1) */
          STA   3,X       ; /* store FF in LSB(2) */
usub32ret:RTS             ; /* return from subroutine */




;/*****************************************************************************
;*
;* Module: unsigned long usub32(unsigned long x, unsigned long y)
;*
;* Description:
;*    The function performs unsigned 32-bit substraction x+y with overflow control 
;*    and saturation. The 32-bit result is set to 0x7FFFFFFF when overflow occurs
;*    and to 0x80000000 when underflow
;*
;* Returns:   z = x - y
;*
;* Arguments: *z(in) L->12,SP H->11,SP
;*            
;*            x (in) L1->9,SP L2->10,SP
;*                   H1->7,SP H2->8,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;                   
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/
sub32:    LDHX  11,SP     ; /* load *z to H:X */           
          LDA   10,SP     ; /* x-LSB(2) */
          SUB   6,SP      ; /* x-LSB(2) + y-LSB(2) */
          STA   3,X       ; /* LSB(2) -> z */ 
          LDA   9,SP      ; /* x-LSB(1) */
          SBC   5,SP      ; /* x-LSB(1) + y-LSB(1) */
          STA   2,X       ; /* LSB(1) -> z */ 
          LDA   8,SP      ; /* x-MSB(2) */
          SBC   4,SP      ; /* x-MSB(2) + y-MSB(2) */
          STA   1,X       ; /* MSB(2) -> z */ 
          LDA   7,SP      ; /* x-MSB(1) */
          SBC   3,SP      ; /* x-MSB(1) + y-MSB(1) */
          ;STA   0,X       ; /* MSB(1) -> z */

          BLT  sub32low    ;V^N:      /* can be too low */
          BPL  sub32ret    ;V=0 & N=0       
          ;STA   0,X       ; /* store 0x7F in MSB(1) */
          LDA   #0FFh     ; /* load accumulator - 0xFF value */
          STA   1,X       ; /* store 0xFF in MSB(2) */
          STA   2,X       ; /* store 0xFF in LSB(1) */
          STA   3,X       ; /* store 0xFF in LSB(2) */
          LDA  #07Fh     ; /* load accumulator - 0x7F value */
sub32ret: STA   0,X       ; /* MSB(1) -> z */
          RTS             ; /* return from subroutine */

sub32low: BMI  sub32ret   ;V=0 & N=1
          CLRA            ; /* clear accumulator - 0x0 value */
          STA   1,X       ; /* store 0x0 in MSB(2) */
          STA   2,X       ; /* store 0x0 in LSB(1) */
          STA   3,X       ; /* store 0x0 in LSB(2) */
          LDA  #080h      ; /* load accumulator - 0x80 value */
          STA   0,X       ; /* store 0x80 in MSB(1) */
          RTS             ; /* return from subroutine */

;/*****************************************************************************
;*
;* Module: unsigned long uadd32_16_32(unsigned long x, unsigned int y)
;*
;* Description:
;*    The function performs unsigned 32-bit addition x+y with overflow control 
;*    and saturation. The 32-bit result is set to 0x7fffffff when overflow occurs.
;*
;* Returns:   z = x + y
;*
;* Arguments: *z(in) L->7,SP H->8,SP
;*                    
;*            x (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;                   
;*
;*            y (in) Low->X, High->H
;*
;* Range Issues: None
;*
;* Special Issues: The result is saturated
;*
;*****************************************************************************/  
uadd32_16_32:   
            PSHX
            PSHH
            LDHX  9,SP     ; /* load *z to H:X */           
            LDA   8,SP     ; /* x-LSB(2) */
            ADD   2,SP      ; /* x-LSB(2) + y-LSB(2) */
            STA   3,X       ; /* LSB(2) -> z */ 
            LDA   7,SP      ; /* x-LSB(1) */
            ADC   1,SP      ; /* x-LSB(1) + y-LSB(1) */
            STA   2,X       ; /* LSB(1) -> z */ 
            LDA   6,SP      ; /* x-MSB(2) */
            ADC   #0        ; /* x-MSB(2) + 0 */
            STA   1,X       ; /* MSB(2) -> z */ 
            LDA   5,SP      ; /* x-MSB(1) */
            ADC   #0        ; /* x-MSB(1) + 0 */
            STA   0,X       ; /* MSB(1) -> z */            
            BCC   uadd3216ret ; /* if carry clear -> end,if set -> overflow */                        
            LDA   #0FFh     ; /* load accumulator - 0xFF value */
            STA   0,X       ; /* store FF in MSB(1) */
            STA   1,X       ; /* store FF in MSB(2) */
            STA   2,X       ; /* store FF in LSB(1) */
            STA   3,X       ; /* store FF in LSB(2) */
uadd3216ret:AIS #2
            RTS             ; /* return from subroutine */

;/*****************************************************************************
;*
;* Module: signed long add32_16_32(signed long x, signed int y)
;*
;* Description:
;*   The function performs the addition of signed 16-bit values 
;*   with signed 32-bit variable. The function could be used for 
;*   summation, accumulation of 16-bit values to 32-bit sum variable.
;*
;* Returns:   x + y
;*
;* Arguments: x (in) LL->6,SP LH->5,SP HL->4,SP HH->3,SP
;*            address of x (in) 7,SP
;*            y (in) H:X          
;*
;* Range Issues: None
;*
;* Special Issues: None
;*
;*****************************************************************************/
add32_16_32:
          TXA             ; /* transfer low byte of int from X to A */
          ADD   6,SP      ; /* add low byte of long */
          STA   6,SP      ; /* overwrite low byte of long on stack */
          PSHH            ; /* push high byte of int from H to stack */
          PULA            ; /* pull pushed value from stack to A */
          TSTA            ; /* test high byte for negative */
          BMI   add32neg  ; /* if N is detected branch to label */
          ADC   5,SP      ; /* add high byte of int to next byte of long with carry */
          STA   5,SP      ; /* store A, overwrite next byte of long on stack */
          CLRA            ; /* fill virtual byte */
          ADC   4,SP      ; /* add virtual byte of int to next byte of long with carry */
          STA   4,SP      ; /* store A, overwrite next byte of long on stack */
          CLRA            ; /* fill virtual byte */
          ADC   3,SP      ; /* add high byte of int to next byte of long with carry */
          STA   3,SP      ; /* store A, overwrite next byte of long on stack */
          BRA   add32res
add32neg: ADC   5,SP      ; /* add high byte of int to next byte of long with carry */
          STA   5,SP      ; /* store A, overwrite next byte of long on stack */
          LDA   #0FFh     ; /* fill virtual byte */
          ADC   4,SP      ; /* add virtual byte of int to next byte of long with carry */
          STA   4,SP      ; /* store A, overwrite next byte of long on stack */
          LDA   #0FFh     ; /* fill virtual byte */
          ADC   3,SP      ; /* add high byte of int to next byte of long with carry */
          STA   3,SP      ; /* store A, overwrite next byte of long on stack */
add32res: LDHX  7,SP      ; /* load address of 32-bit in/out variable to H:X */
          LDA   3,SP      ; /* load A by result byte */
          STA   0,X       ; /* store result byte to location */
          LDA   4,SP      ; /* load A by result byte */
          STA   1,X       ; /* store result byte to location */
          LDA   5,SP      ; /* load A by result byte */
          STA   2,X       ; /* store result byte to location */
          LDA   6,SP      ; /* load A by result byte */
          STA   3,X       ; /* store result byte to location */
          RTS             ; /* return from subroutine */

;/*****************************************************************************
;*
;* Module: SByte neg8(SByte x)
;*
;* Description:
;*   The function performs the negation of x with overflow control
;*   and saturation. The 8-bit result is set at +127 when overflow occurs.
;*
;* Returns:   -x
;*
;* Arguments: x (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
;  /* (V - Overflow Flag, N - Negative flag) */
;  /* V=1 & N=1 too high */
;  /* Overflow can ocure only if negative x is negated */
neg8:  
        NEGA           ;        /* x = -x */
        BPL  endneg8   ;N=0     /* x was negative but overflow did not ocure */
        BLT  endneg8   ;N=1 V=0 /* x was positive */
        DECA           ;N=1 V=1 /* to high */
endneg8:RTS

;/*****************************************************************************
;*
;* Module: signed int neg(signed int x)
;*
;* Description:
;*   The function performs the negation of x with overflow control
;*   and saturation. The 16-bit result is set at +32767 when overflow occurs.
;*
;* Returns:   -x
;*
;* Arguments: x (in) Low->X(A); High->H(X)
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
;  /* (V - Overflow Flag, N - Negative flag) */
;  /* V=1 & N=1 too high */
neg16:  
        TXA
        PSHH
        PULX
        COMX           ;        /* x = -x */
        NEGA
        BNE  endneg
        INCX
        BPL  endneg    ;N=0     /* x was negative but overflow did not ocure */
        BLT  endneg    ;N=1 V=0 /* x was positive */
        DECA           ;N=1 V=1 /* to high */
        DECX
endneg: PSHX
        PULH
        TAX
        RTS

;/*****************************************************************************
;*
;* Module: signed int abs16(signed int x)
;*
;* Description:
;*    The function performs the abs value of a 16-bit input variable.
;*
;* Returns: abs(x)
;*
;* Arguments: x (in) H:X
;*            
;* Range Issues: None
;*
;* Special Issues: None 
;*
;*****************************************************************************/
abs16:    PSHH            ; /* push H on stack */ 
          LDA   1,SP      ; /* load A from stack, condition flags affected */
          BPL   endabs16_8; /* if A is positive jump to label */ end label */
absneg:   COMA
          NEGX            ; /* negate X */
          BNE   endaneg16 ; /* branch of not equal, if Z=0 */
          INCA
          BPL   endaneg16  ; /* N=0, x was negative but overflow did not occur */
          ; BNE   endaneg16 ; /* branch of not equal, if Z=0 */
          BLT   endaneg16 ; /* N=1 V=0, x was positive */
          DECA            ; /* decrement A */
          DECX            ; /* decrement X sets FFFF */
endaneg16:PSHA
          PULH
endabs16_8:
          AIS   #1        ; /* restore stack pointer */
          RTS             ; /* return from subroutine */

;/*****************************************************************************
;*
;* Module: unsigned long umul16_8to32(unsigned int x, unsigned char y)
;*
;* Description:
;*     Unsigned multiply 16 bit by 8 bit Ubyte.
;*     The Result is saturated at 0xFF if overflow occures.
;*
;* Returns:   x*y*256
;*
;* Arguments: @z Low SP,4, @z High SP,3
;             x (in) H->H, L->X
;*            y (in) A
;*
;* Range Issues: if y=0 the result is saturated at 0xFF
;*
;* Special Issues:  HCS08S -Cs08 Option backend.The result is saturated
;*
;*****************************************************************************/
umul16_8to32:
        PSHA        ; y -> @(SP)
        PSHX        ; x Low -> @(SP)
        PSHH
        PULX        ; x High -> X
        MUL         ; x High * y -> X:A
        PSHA        ; First2 -> @(SP)
        PSHX        ; First3 -> @(SP)

        LDA 3,SP    ; x -> A
        LDX 4,SP    ; y -> A
        MUL         ; x Low * y -> X:A
        PSHX        ; Secound2 -> @(SP)
        LDHX 8,SP    ; @z -> HX 
        STA 2,X     ; z1 ->  z1
        PULA        ; Secound2->A
        ADC 2,SP    ; First2 + Secound2 -> 
        STA 1,X     ; z2 -> z2 
        CLRA
        STA 3,X     ; 0 -> z0
        ADC 1,SP    ; First3 + 0
        STA 0,X     ; z3 -> z3
        AIS #4
        RTS

;/*****************************************************************************
;*
;* Module: signed int shl16_u8to16(signed int x, unsigned char n)
;*
;* Description:
;*   The function returns value x arithmetical shifted by n bits result
;*   with overflow control and saturation. The 8-bit result is set at +32767
;*   when overflow occurs, or at -32768 when underflow occurs.
;*
;* Returns:   x * 2^n
;*
;* Arguments: x (in) (in) Low->X;   High->H
;*            n (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
shl16_u8to16: PSHA             ;/* save n                               */
            PSHH             ;
            PULA             ;/* H byte of x                          */
            TST  1,SP        ;/* (if n>0)                             */
            BEQ  shlend      ;/* {                                    */
            TSTA             ;/* if (x=0) return                      */
            BNE  x_not_0
            TSTX
            BEQ  shlend
            TSTA             ;/*   if(x>=0)                           */
x_not_0:    BMI  shlneg      ;/*   {                                  */
shlpos:     LSLX             ;/*       while(n>0)                     */
            ROLA             ;/*     {                                */
            BMI  posout      ;/*       x<<1                           */ 
            DBNZ 1,SP,shlpos ;/*       if (x>32767) {x=+32767; return */
            PSHA             ;//Debug 09.09.23
            PULH             ;//Debug 09.09.23
            PULA             ;/*       n--;                           */
            RTS              ;/*     }                                */
posout:     LDX  #0FFh       ;/*   }                                  */
            LDA  #07Fh
            PSHA
            PULH
            PULA
            RTS
shlneg:     LSLX             ;/*   else                               */
            ROLA             ;/*   {                                  */ 
            BPL  negout      ;/*     while (n>0)                      */       
            DBNZ 1,SP,shlneg ;/*     {                                */
            PSHA
            PULH
            PULA             ;/*       x<<1                           */ 
            RTS              ;/*       if (x<32768) {x=-32768; return */
negout:     CLRX             ;/*       n--;                           */     
            LDA  #080h       ;/*     }                                */
            PSHA
            PULH
shlend:     PULA             ;/*   }                                  */
            RTS              ;/* }

;/*****************************************************************************
;*
;* Module: signed int shr16_u8to16(signed int x, unsigned char n)
;*
;* Description:
;*   The function returns value x arithmetical shifted to the right by n bits result
;*
;* Returns:   x / 2^n
;*
;* Arguments: x (in) (in) Low->X;   High->H
;*            n (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
shr16_u8to16:PSHA             ;/* save n                               */
            PSHH             ;
            PULA             ; /* H byte of x                          */
            TST   1,SP       ; /* test for zero shift */
            BEQ   shr16end   ; /* if zero jump to label */
shrpos:     ASRA         ;/*       while(n>0)                     */
            RORX         ;/*     {                                */
            DBNZ 1,SP,shrpos ;/*       if (x>32767) {x=+32767; return */

            PSHA             ;//Debug 09.09.23
            PULH             ;//Debug 09.09.23
 shr16end:  PULA             ;/*       n--;                           */
            RTS              ;/*     }                                */

;/*****************************************************************************
;*
;* Module: signed long shl32_u8to32(signed long x, unsigned char n)
;*
;* Description:
;*   The function returns value x arithmetical shifted to the left by n bits result
;*   with overflow control and saturation. The 32-bit result is set at 0x7FFFFFFF
;*   when overflow occurs, or at 0x80000000 when underflow occurs.
;*
;* Returns:   x * 2^n
;*
;* Arguments: *z(in) L->8,SP H->7,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;
;*            n (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
shl32_u8to32:;PSHA             ;/* save n                               */
            LDHX  7,SP     ; /* load *z to H:X */   
            ;TST  1,SP        ;/* (if n!=0)                             */
            TSTA
            BEQ  shlend32 ;/* {                                    */
            TST 3,SP        ;/*   if(x>=0)                           */
            BMI  shlneg32    ;/*   {                                  */

shlpos32:   LSL 6,SP         ;/*       while(n>0)                     */
            ROL 5,SP         ;/*     {                                */
            ROL 4,SP         ;/*     {                                */
            ROL 3,SP         ;/*     {                                */
            BMI  posout32    ;/*       x<<1                           */
            DBNZA shlpos32 ;
            ;DBNZ 1,SP,shlpos32 ;/*       if (x>32767) {x=+32767; return */
shlend32:   LDA  6,SP
            STA   3,X       ; /* store 0x0 in MSB(2) */
            LDA   5,SP
            STA   2,X       ; /* store 0x0 in LSB(1) */
            LDA   4,SP
            STA   1,X       ; /* store 0x0 in LSB(2) */
            LDA   3,SP
            STA   0,X       ; /* store 0x80 in MSB(1) */
            ;PULA             ;/*       n--;                           */         
            RTS              ;/*     }                                */

posout32:   LDA   #0FFh     ; /* load accumulator - 0xFF value */
            STA   1,X       ; /* store 0xFF in MSB(2) */
            STA   2,X       ; /* store 0xFF in LSB(1) */
            STA   3,X       ; /* store 0xFF in LSB(2) */
            LDA  #07Fh     ; /* load accumulator - 0x7F value */
            STA   0,X       ; /* MSB(1) -> z */
            ;PULA             ;/*       n--;                           */
            RTS


shlneg32:   LSL 6,SP         ;/*       while(n>0)                     */
            ROL 5,SP         ;/*     {                                */
            ROL 4,SP         ;/*     {                                */
            ROL 3,SP         ;/*     {                                */
            BPL  negout32    ;/*     while (n>0)                      */       
            DBNZA shlneg32   ;
            ;DBNZ 1,SP,shlneg32 ;/*     {                                */
            BRA  shlend32
 
negout32:   CLRA            ; /* clear accumulator - 0x0 value */
            STA   1,X       ; /* store 0x0 in MSB(2) */
            STA   2,X       ; /* store 0x0 in LSB(1) */
            STA   3,X       ; /* store 0x0 in LSB(2) */
            LDA  #080h      ; /* load accumulator - 0x80 value */
            STA   0,X       ; /* store 0x80 in MSB(1) */
            ;PULA             ;/*       n--;                           */
            RTS             ; /* return from subroutine */
            

;/*****************************************************************************
;*
;* Module: signed long shr32_u8to32(signed long x, unsigned char n)
;*
;* Description:
;*   The function returns value x arithmetical shifted to the right by n bits result
;*
;* Returns:   x / 2^n
;*
;* Arguments: *z(in) L->8,SP H->7,SP
;*                    
;*            y (in) L1->5,SP; L2->6,SP;
;*                   H1->3,SP; H2->4,SP;
;*            n (in) A
;*
;* Range Issues: None
;*
;* Special Issues: HCS08S -Cs08 Option backend. The result is saturated
;*
;*****************************************************************************/
shr32_u8to32:;PSHA             ;/* save n                               */ 
            LDHX  7,SP     ; /* load *z to H:X */   
            ;TST  1,SP        ;/* (if n!=0)                             */
            TSTA
            BEQ  shrend32 ;/* {                                    */
shrpos32:   ASR 3,SP         ;/*       while(n>0)                     */
            ROR 4,SP         ;/*     {                                */
            ROR 5,SP         ;/*     {                                */
            ROR 6,SP         ;/*     {                                */
            DBNZA shrpos32 ;
            ;DBNZ 1,SP,shrpos32 ;/*       if (x>32767) {x=+32767; return */
shrend32:   LDA  6,SP
            STA   3,X       ; /* store 0x0 in MSB(2) */
            LDA   5,SP
            STA   2,X       ; /* store 0x0 in LSB(1) */
            LDA   4,SP
            STA   1,X       ; /* store 0x0 in LSB(2) */
            LDA   3,SP
            STA   0,X       ; /* store 0x80 in MSB(1) */
            ;PULA             ;/*       n--;                           */         
            RTS              ;/*     }                                */
 

;/*****************************************************************************
;* Function:        l_sys_irq_disable
;*
;* Description:     Disable interrupts
;*
;* Returns:         Interrupt mask before disabling
;*
;*****************************************************************************/     
irq_disable:
        TPA
        SEI
        RTS

;/*****************************************************************************
;* Function:        l_sys_irq_disable
;*
;* Description:     Disable interrupts
;*
;* Returns:         Interrupt mask before disabling
;*
;*****************************************************************************/ 
irq_restore:
        TAP
        RTS