/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"

#include "fsl_dcp.h"

#include "pin_mux.h"
#include "clock_config.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

typedef enum _dcp_otp_key_select
{
    kDCP_SNVSMKLow  = 1U, /* Use [127:0] from snvs key as dcp key */
    kDCP_SNVSMKHigh = 2U, /* Use [255:128] from snvs key as dcp key */
    kDCP_SWGP2      = 3U, /* Use [127:0] from ocotp key as dcp key */
} dcp_otp_key_select_t;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

#define TEST_ASSERT(a)       \
    if (!(a))                \
    {                        \
        PRINTF("error\r\n"); \
        do                   \
        {                    \
        } while (1);         \
    }

status_t DCP_OTPKeySelect(dcp_otp_key_select_t keySelect)
{
    if (keySelect == kDCP_SNVSMKLow)
    {
        IOMUXC_GPR->GPR3 &= ~(1 << IOMUXC_GPR_GPR3_DCP_KEY_SEL_SHIFT);
        IOMUXC_GPR->GPR10 &= ~(1 << IOMUXC_GPR_GPR10_DCPKEY_OCOTP_OR_KEYMUX_SHIFT);
    }

    else if (keySelect == kDCP_SNVSMKHigh)
    {
        IOMUXC_GPR->GPR3 |= (1 << IOMUXC_GPR_GPR3_DCP_KEY_SEL_SHIFT);
        IOMUXC_GPR->GPR10 &= ~(1 << IOMUXC_GPR_GPR10_DCPKEY_OCOTP_OR_KEYMUX_SHIFT);
    }

    else if (keySelect == kDCP_SWGP2)
    {
        IOMUXC_GPR->GPR3 &= ~(1 << IOMUXC_GPR_GPR3_DCP_KEY_SEL_SHIFT);
        IOMUXC_GPR->GPR10 |= (1 << IOMUXC_GPR_GPR10_DCPKEY_OCOTP_OR_KEYMUX_SHIFT);
    }

    else
    {
        return kStatus_InvalidArgument;
    }

    return kStatus_Success;
}


void TestAesEcb(dcp_key_slot_t keyslot)
{
    static const uint8_t keyAes128[] __attribute__((aligned)) = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                                                                 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c};
    static const uint8_t plainAes128[]                        = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
																																 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a};
    static const uint8_t cipherAes128[]                       = {0x3a, 0xd7, 0x7b, 0xb4, 0x0d, 0x7a, 0x36, 0x60,
																																 0xa8, 0x9e, 0xca, 0xf3, 0x24, 0x66, 0xef, 0x97};

    uint8_t cipher[16];
    uint8_t output[16];
    status_t status;

    dcp_handle_t m_handle;

    m_handle.channel    = kDCP_Channel0;
    m_handle.swapConfig = kDCP_NoSwap;

    m_handle.keySlot = keyslot;

    status = DCP_AES_SetKey(DCP, &m_handle, keyAes128, 16);
    TEST_ASSERT(kStatus_Success == status);

    DCP_AES_EncryptEcb(DCP, &m_handle, plainAes128, cipher, 16);
		if( (keyslot == kDCP_KeySlot0) || 
				(keyslot == kDCP_KeySlot1) || 
				(keyslot == kDCP_KeySlot2) || 
				(keyslot == kDCP_KeySlot3) ||
				(keyslot == kDCP_PayloadKey) ){
			TEST_ASSERT(memcmp(cipher, cipherAes128, 16) == 0);
		}

    DCP_AES_DecryptEcb(DCP, &m_handle, cipher, output, 16);
    TEST_ASSERT(memcmp(output, plainAes128, 16) == 0);

    PRINTF("  AES ECB Test pass.\r\n");
}

void TestAesCbc(dcp_key_slot_t keyslot)
{
    static const uint8_t keyAes128[] __attribute__((aligned)) = {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
                                                                 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c};
    static const uint8_t plainAes128[]                        = {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
                                          0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a};
    static const uint8_t ive[]                                = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                                  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f};

    static const uint8_t cipherAes128[] = {0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
                                           0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d};

    uint8_t cipher[16];
    uint8_t output[16];
    status_t status;

    dcp_handle_t m_handle;

    m_handle.channel    = kDCP_Channel0;
    m_handle.swapConfig = kDCP_NoSwap;

    m_handle.keySlot = keyslot;

    status = DCP_AES_SetKey(DCP, &m_handle, keyAes128, 16);
    TEST_ASSERT(kStatus_Success == status);
		
    DCP_AES_EncryptCbc(DCP, &m_handle, plainAes128, cipher, 16, ive);
    if( (keyslot == kDCP_KeySlot0) || 
				(keyslot == kDCP_KeySlot1) || 
				(keyslot == kDCP_KeySlot2) || 
				(keyslot == kDCP_KeySlot3) ||
				(keyslot == kDCP_PayloadKey) ){
			TEST_ASSERT(memcmp(cipher, cipherAes128, 16) == 0);
		}

    DCP_AES_DecryptCbc(DCP, &m_handle, cipher, output, 16, ive);
    TEST_ASSERT(memcmp(output, plainAes128, 16) == 0);

    PRINTF("  AES CBC Test pass.\r\n");
}
void PrintKeyInfo(dcp_key_slot_t keyslot, dcp_otp_key_select_t keySelect)
{
	switch(keyslot){
		case kDCP_KeySlot0:
			PRINTF("Key Slot0:\r\n");
			break;
		case kDCP_KeySlot1:
			PRINTF("Key Slot1:\r\n");
			break;
		case kDCP_KeySlot2:
			PRINTF("Key Slot2:\r\n");
			break;
		case kDCP_KeySlot3:
			PRINTF("Key Slot3:\r\n");
			break;
		case kDCP_OtpKey:
			PRINTF("OCOTP Key");
			switch(keySelect){
				case kDCP_SNVSMKLow:
					PRINTF("(SNVS Master key Low):\r\n");
					break;
				case kDCP_SNVSMKHigh:
					PRINTF("(SNVS Master key Hight):\r\n");
					break;
				case kDCP_SWGP2:
					PRINTF("(SW_GP2 key):\r\n");
					break;
				default:
					PRINTF("(wrong OCOTP key select.\r\n)");
					break;
			}
			break;
    case kDCP_OtpUniqueKey:
			PRINTF("Unique key:\r\n");
			break;
    case kDCP_PayloadKey:
			PRINTF("Payload key:\r\n");
			break;
		default:
			PRINTF("Wrong key slot.\r\n");
			break;
	}
}

/*!
 * @brief Main function
 */
int main(void)
{
    dcp_config_t dcpConfig;

    /* Init hardware*/
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    /* Data cache must be temporarily disabled to be able to use sdram */
    SCB_DisableDCache();

    PRINTF("DCP key management.\r\n");

		for(dcp_key_slot_t keyslot = kDCP_KeySlot0; keyslot <= kDCP_PayloadKey; keyslot++){
        if(keyslot == kDCP_OtpUniqueKey){
          continue;
        }
				if(keyslot == kDCP_OtpKey){
					for(dcp_otp_key_select_t otpkeyselect = kDCP_SNVSMKLow; otpkeyselect <= kDCP_SWGP2; otpkeyselect++){
						PrintKeyInfo(keyslot, otpkeyselect);
						/* set OTP Key */
						DCP_OTPKeySelect(otpkeyselect);
						DCP_GetDefaultConfig(&dcpConfig);
						/* Initialize DCP */
						DCP_Init(DCP, &dcpConfig);
						TestAesEcb(keyslot);
						TestAesCbc(keyslot);
						/* Deinitialize DCP */
						DCP_Deinit(DCP);
					}
				} else {
					PrintKeyInfo(keyslot, 0);
					/* Reset and initialize DCP */
					DCP_GetDefaultConfig(&dcpConfig);
					DCP_Init(DCP, &dcpConfig);
					TestAesEcb(keyslot);
					TestAesCbc(keyslot);
					/* Deinitialize DCP */
					DCP_Deinit(DCP);
				}
		}

    while (1)
    {
    }
}
