/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdbool.h>
#include "board.h"
#include "fsl_debug_console.h"
#include "serial_manager.h"
#include "fsl_os_abstraction.h"
#include "fsl_port.h"
#include "fsl_gpio.h"
#include "fsl_common.h"
#include "led.h"

#include "fsl_device_registers.h"
#include "button.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "timer_manager.h"

#include "ApplMain.h"
#include "FSCI_main.h"
#include "temperature_sensor.h"
#include "temperature_interface.h"
#include "battery_interface.h"
#include "device_info_interface.h"
#include "ble_sig_defines.h"
#include "panic.h"

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#define UUID128(name, ...) uint8_t name[16] = { __VA_ARGS__ };
#include "gatt_uuid128.h"
#undef UUID128
/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/
#define BUTTON_HANDLE_SIZE (16U + 24U)
#define TIMER_SOURCE_CLOCK CLOCK_GetFreq(kCLOCK_BusClk)

/*******************************************************************************
 * Private memory declarations
 ******************************************************************************/
static TimerHandle_t timerHandle = NULL;
static uint8_t TimerId;
tmsConfig_t tmsServiceConfig;
static basConfig_t basServiceConfig;
static disConfig_t disServiceConfig;

typedef struct advState_tag{
	bool_t      advOn;
}advState_t;

static advState_t mAdvState;
/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
bool_t g_ButtonPress = FALSE;
bool_t g_ButtonLongPress = FALSE;
deviceId_t  mPeerDeviceId = gInvalidDeviceId_c;

uint8_t s_buttonHandleBuffer[BUTTON_HANDLE_SIZE];
uint16_t tempcharHandle;
uint16_t descriptorHandle;
/*******************************************************************************
 * Private functions Prototypes
*************************************************************************************
************************************************************************************/
static void App_SetAdvertisingParameter(void);
static void App_RegisterGenericCallback (gapGenericCallback_t GenericCallback);
static void App_RegisterGattServerCallback(gattServerCallback_t  serverCallback);

button_status_t button_callback(void *buttonHandle, button_callback_message_t *message, void *callbackParam);
static void AdvertisingTimerCallback(TimerHandle_t xTimer);

/*! *********************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
********************************************************************************** */

/*! *********************************************************************************
*************************************************************************************
* Globals
*************************************************************************************
********************************************************************************** */
extern led_handle_t s_ledMonochromeHandle[1];
extern led_flash_config_t ledFlash;
/*******************************************************************************
 ********************************************************************************
 * Public functions
 ******************************************************************************
********************************************************************************/
bleResult_t TempSensor_GATTDB_Init(void)
{
	bleResult_t result = gBleSuccess_c;
	uint16_t handle;
	uint16_t charUUID = gBleSig_Temperature_d;
	uint8_t initVal = 0xB4;

	uint16_t descUUID = gBleSig_CharPresFormatDescriptor_d;
	uint8_t descinitValue[7] = {0x0E, 0xFE, 0x2F, 0x27, 0x00, 0x00, 0x00};
	bleUuid_t temperature_uuid = { .uuid128 = { 0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x00, 0x02, 0xFF, 0x01 } };

	/* Add service */
	result = GattDb_AddPrimaryService(0, gBleUuidType128_c,
									 &temperature_uuid,
									 &handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Service handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	tmsServiceConfig.serviceHandle = handle;
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
												(bleUuid_t*)&charUUID, //char UUID
												gGattCharPropNotify_c, //Properties
	                                            2, //Max Value len
	                                            1, //Initial value len
	                                            (uint8_t*)&initVal, // Init Value
												gPermissionNone_c,
	                                            &handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	tempcharHandle = handle;

	result = GattDb_AddCharacteristicDescriptor(gBleUuidType16_c,
												(bleUuid_t*)&descUUID,
												7, //Val length
												descinitValue, //Init val
												gPermissionFlagReadable_c, // Access Permission
												&handle);

	/* Char Descriptor handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	descriptorHandle = handle;

	result = GattDb_AddCccd(&handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}
	vTaskDelay(pdMS_TO_TICKS(100));
	PRINTF("\r\n---------- GATT-DB: TEMPERATURE SERVICE ADDED OVER FSCI BY SPI/UART COMMUNICATION ----------\r\n");
	return result;
}

bleResult_t Battery_GATTDB_Init(void)
{
	bleResult_t result = gBleSuccess_c;
	uint16_t handle;
	uint16_t BatServUUID = gBleSig_BatteryService_d;
	uint16_t charUUID = gBleSig_BatteryLevel_d;
	uint8_t BatteryinitVal = 0x5A;
	uint16_t descUUID = gBleSig_CharPresFormatDescriptor_d;
	uint8_t descinitValue[7] = {0x04, 0x00, 0xAD, 0x27, 0x01, 0x01, 0x00};

	/* Add battery service */
	result = GattDb_AddPrimaryService(0, gBleUuidType16_c,
									  (bleUuid_t*)&BatServUUID,
									  &handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Service handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	basServiceConfig.serviceHandle = handle;

	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
												(bleUuid_t*)&charUUID, //char UUID
												(gGattCharPropNotify_c | gGattCharPropRead_c), //Properties
	                                            1, //Max Value len
	                                            1, //Initial value len
	                                            (uint8_t*)&BatteryinitVal, // Init Value
												gPermissionFlagReadable_c,
	                                            &handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(50));

	//tempcharHandle = handle;

	result = GattDb_AddCharacteristicDescriptor(gBleUuidType16_c,
												(bleUuid_t*)&descUUID,
												7, //Val length
												descinitValue, //Init val
												gPermissionFlagReadable_c, // Access Permission
												&handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}
	/* Char Descriptor handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	PRINTF("\r\n---------- GATT-DB: BATTERY SERVICE ADDED OVER FSCI BY SPI/UART COMMUNICATION ----------\r\n");
	return result;
}

bleResult_t GAP_GATT_Service_Init()
{
	bleResult_t result;
	uint16_t handle;
	uint16_t GATTServUUID = gBleSig_GenericAttributeProfile_d;
	uint16_t GAPServUUID = gBleSig_GenericAccessProfile_d;
	uint16_t GATTCharUUID = gBleSig_GattServiceChanged_d;
	uint16_t GAPCharUUID = gBleSig_GapDeviceName_d;
	uint8_t InitValueLength[4]  = {0};
	uint8_t GAPValueLength[12] = {"NXP_BLE_TEMP"};

	result = GattDb_AddPrimaryService(0, gBleUuidType16_c,
									 (bleUuid_t*)&GATTServUUID,
									  &handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
													(bleUuid_t*)&GATTCharUUID, //char UUID
													(gGattCharPropNotify_c | gGattCharPropRead_c), //Properties
		                                            4, //Max Value len
		                                            4, //Initial value len
		                                            (uint8_t*)&InitValueLength, // Init Value
													gPermissionNone_c,
		                                            &handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));

	result = GattDb_AddCccd(&handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}
	vTaskDelay(pdMS_TO_TICKS(100));

	PRINTF("\r\n---------- GATT-DB: GATT SERVICE ADDED OVER FSCI BY SPI/UART COMMUNICATION ----------\r\n");
	result = GattDb_AddPrimaryService(0, gBleUuidType16_c,
									  (bleUuid_t*)&GAPServUUID,
									  &handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
													(bleUuid_t*)&GAPCharUUID, //char UUID
													gGattCharPropRead_c, //Properties
													sizeof(GAPValueLength), //Max Value len
													sizeof(GAPValueLength), //Initial value len
													(uint8_t*)&GAPValueLength, // Init Value
													gGattCharPropRead_c,
													&handle);

	/* Char handle is filled by second FSCI response, so wait for second response */
	if (result != gBleSuccess_c)
	{
		return result;
	}
	vTaskDelay(pdMS_TO_TICKS(100));

	PRINTF("\r\n---------- GATT-DB: GAP SERVICE ADDED OVER FSCI BY SPI/UART COMMUNICATION ----------\r\n");
	return result;
}

bleResult_t DeviceInfo_GATTDB_Init(void)
{
	uint16_t handle;
	uint16_t DeviceServUUID = gBleSig_DeviceInformationService_d;
	uint16_t ManuNameStrUUID = gBleSig_ManufacturerNameString_d;
	uint8_t ManufacturerName[3] = {"NXP"};
	uint8_t ModelNum[16] = {"Temp Sensor Demo"};
	uint16_t charModelNo = gBleSig_ModelNumberString_d;
	uint16_t charSerialNo = gBleSig_SerialNumberString_d;
	uint8_t SerialNo[7] = {"BLESN01"};
	uint16_t charHardwareStr = gBleSig_HardwareRevisionString_d;
	uint8_t HardwareStr[9] = {"FRDM-KW36"};
	uint16_t charFWRevStr = gBleSig_FirmwareRevisionString_d;
	uint8_t FWRevStr[5] = {"1.1.1"};
	uint16_t charSWRevStr = gBleSig_SoftwareRevisionString_d;
	uint8_t SWRevStr[5] = {"1.1.4"};
	bleResult_t result;

	result = GattDb_AddPrimaryService(0, gBleUuidType16_c,
									  (bleUuid_t*)&DeviceServUUID,
									  &handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Service handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	disServiceConfig.serviceHandle = handle;

	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
													(bleUuid_t*)&ManuNameStrUUID, //char UUID
													gGattCharPropRead_c, //Properties
													sizeof(ManufacturerName), //Max Value len
													sizeof(ManufacturerName), //Initial value len
													(uint8_t*)&ManufacturerName, // Init Value
													gPermissionFlagReadable_c,
													&handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
														(bleUuid_t*)&charModelNo, //char UUID
														gGattCharPropRead_c, //Properties
														sizeof(ModelNum), //Max Value len
														sizeof(ModelNum), //Initial value len
														(uint8_t*)&ModelNum, // Init Value
														gPermissionFlagReadable_c,
														&handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
														(bleUuid_t*)&charSerialNo, //char UUID
														gGattCharPropRead_c, //Properties
														sizeof(SerialNo), //Max Value len
														sizeof(SerialNo), //Initial value len
														(uint8_t*)&SerialNo, // Init Value
														gPermissionFlagReadable_c,
														&handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
														(bleUuid_t*)&charHardwareStr, //char UUID
														gGattCharPropRead_c, //Properties
														sizeof(HardwareStr), //Max Value len
														sizeof(HardwareStr), //Initial value len
														(uint8_t*)&HardwareStr, // Init Value
														gPermissionFlagReadable_c,
														&handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
														(bleUuid_t*)&charFWRevStr, //char UUID
														gGattCharPropRead_c, //Properties
														sizeof(FWRevStr), //Max Value len
														sizeof(FWRevStr), //Initial value len
														(uint8_t*)&FWRevStr, // Init Value
														gPermissionFlagReadable_c,
														&handle);

	if (result != gBleSuccess_c)
	{
		return result;
	}

	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	result = GattDb_AddCharacteristicDeclarationAndValue(gBleUuidType16_c,
														(bleUuid_t*)&charSWRevStr, //char UUID
														gGattCharPropRead_c, //Properties
														sizeof(SWRevStr), //Max Value len
														sizeof(SWRevStr), //Initial value len
														(uint8_t*)&SWRevStr, // Init Value
														gPermissionFlagReadable_c,
														&handle);
	if (result != gBleSuccess_c)
	{
		return result;
	}
	/* Char handle is filled by second FSCI response, so wait for second response */
	vTaskDelay(pdMS_TO_TICKS(100));
	PRINTF("\r\n---------- GATT-DB: DEVICE INFORMATION SERVICE ADDED OVER FSCI BY SPI/UART COMMUNICATION ----------\r\n");

	return result;
}

/*! *********************************************************************************
* \brief        Application configuration function.
*
********************************************************************************** */
void App_config()
{
	PRINTF("\r\n---------- CONFIGURATION STARTED ----------\r\n");
	mAdvState.advOn = false;
	App_RegisterGenericCallback(App_GenericCallback);
	/* Register for callbacks*/
	App_RegisterGattServerCallback(App_GattServerCallback);
	/* Start services */
	GAP_GATT_Service_Init();
	TempSensor_GATTDB_Init();
	Battery_GATTDB_Init();
	DeviceInfo_GATTDB_Init();
	tmsServiceConfig.initialTemperature = 25;
	Tms_Start(&tmsServiceConfig);
}

/*! *********************************************************************************
* \brief    start advertising function.
*
********************************************************************************** */
void App_Start(void)
{
	if(mPeerDeviceId == gInvalidDeviceId_c)
	{
		/* Device is not connected and not advertising */
		if(!mAdvState.advOn)
		{
			g_ButtonPress = FALSE;
			App_SetAdvertisingParameter();
		}
	}
	else
	{
		g_ButtonPress = FALSE;
		BleApp_SendTemperature();
	}
}

/*! *********************************************************************************
* \brief        Wrapper function to register generic callback function.
*
* \param[in]    GenericCallback callback function.
********************************************************************************** */
static void App_RegisterGenericCallback (gapGenericCallback_t GenericCallback)
{
	GAP_RegisterGenericCallback(GenericCallback);
}

static void App_RegisterGattServerCallback(gattServerCallback_t serverCallback)
{
	Gatt_RegisterCallback(serverCallback);
}

/*! *********************************************************************************
* \brief        Handles GATT server callback from host stack.
*
* \param[in]    deviceId        Peer device ID.
* \param[in]    pServerEvent    Pointer to gattServerEvent_t.
********************************************************************************** */
void BleApp_GattServerCallback (deviceId_t deviceId, gattServerEvent_t* pServerEvent)
{
    switch (pServerEvent->eventType)
    {
        case gEvtCharacteristicCccdWritten_c:
        {
#ifdef DEBUG_PRINT_ENABLE
        	PRINTF("CCCD written \r\n");
#endif

#ifndef gTempTimerSupported
        	BleApp_SendTemperature();
#endif
        }
        break;
    default:
        break;
    }
}

/*! *********************************************************************************
* \brief        Handles BLE generic callback.
*
* \param[in]    pGenericEvent    Pointer to gapGenericEvent_t.
********************************************************************************** */
void BleApp_GenericCallback(gapGenericEvent_t* pGenericEvent)
{
	switch(pGenericEvent->eventType)
	{
		case gAdvertisingParametersSetupComplete_c:
		{
			(void)Gap_SetAdvertisingData(&gAppAdvertisingData, NULL);
		}
		break;

		case gAdvertisingDataSetupComplete_c:
		{
			App_StartAdvertising(AdvertisingCallback, ConnectionCallback);
		}
		break;

		default:
			break;
	}
}

/*! *********************************************************************************
* \brief        Handles advertising timer callback.
*
* \param[in]    xTimer        Timer handle.
********************************************************************************** */
static void AdvertisingTimerCallback(TimerHandle_t xTimer)
{
	/* Stop Advertising */
	if(mAdvState.advOn)
	{
		PRINTF("\r\n---------- STOP ADVERTISEMENT ----------\r\n");
    	LED_SetColor(s_ledMonochromeHandle[0], kLED_White);
		LED_Flash(s_ledMonochromeHandle[0], &ledFlash);
		Gap_StopAdvertising();
	}
}

/*! *********************************************************************************
* \brief        Handles BLE Advertising callback.
*
* \param[in]    pAdvertisingEvent    Pointer to gapAdvertisingEvent_t.
********************************************************************************** */
void AdvertisingCallback(gapAdvertisingEvent_t* pAdvertisingEvent)
{
	switch (pAdvertisingEvent->eventType)
	{
		case gAdvertisingStateChanged_c:
		{
			mAdvState.advOn = !mAdvState.advOn;

			if(mAdvState.advOn)
			{
				PRINTF("\r\n---------- ADVERTISEMENT STARTED ---------- \r\n");

				//Create timer for advertising timeout
				timerHandle = xTimerCreate("Timer", pdMS_TO_TICKS(gAdvTime_c), pdFALSE, &TimerId, AdvertisingTimerCallback);
				if(timerHandle != NULL)
				{
					// Start timer
					xTimerStart(timerHandle, 0);
				}

				LED_SetColor(s_ledMonochromeHandle[0], kLED_Red);
				LED_Flash(s_ledMonochromeHandle[0], &ledFlash);
			}
			else
			{
				//Stop advertising timer */
				xTimerStop(timerHandle, 0);
			}
		}
		break;

		case gAdvertisingCommandFailed_c:
		{
			PRINTF("\r\n---------- ADVERTISING COMMAND FAILED ----------\r\n---------- RESET THE DEVICE ----------\r\n");
			panic(0,0,0,0);
		}
		break;

		default:
			break;
	}
}

/*! *********************************************************************************
* \brief        Handles BLE Connection callback.
*
* \param[in]    peerDeviceId        Peer device ID.
* \param[in]    pConnectionEvent    Pointer to gapConnectionEvent_t.
********************************************************************************** */
void ConnectionCallback(deviceId_t peerDeviceId, gapConnectionEvent_t* pConnectionEvent)
{
	switch (pConnectionEvent->eventType)
	{
		case gConnEvtConnected_c:
		{
			PRINTF("\r\n---------- CONNECTED ----------\r\n");

			/* Advertising stops when connected */
			mAdvState.advOn = false;
			mPeerDeviceId = peerDeviceId;

			Bas_Subscribe(&basServiceConfig, peerDeviceId);
			Tms_Subscribe(peerDeviceId);

			if(xTimerIsTimerActive(timerHandle) != pdFALSE)
			{
				//Stop advertising timer */
				xTimerStop(timerHandle, 0);
			}

#ifdef gTempTimerSupported
			if(xTimerIsTimerActive(sendTempTimerHandle) == pdFALSE)
			{
				//Start temperature packet send timer */
				xTimerStart(sendTempTimerHandle, 0);
			}
#endif

			/* Turn ON LED */
	    	LED_SetColor(s_ledMonochromeHandle[0], kLED_Red);
			LED_TurnOnOff(s_ledMonochromeHandle[0], TRUE);
		}
		break;

		case gConnEvtDisconnected_c:
		{
			PRINTF("\r\n---------- DISCONNECTED ----------\r\n");

			mPeerDeviceId = gInvalidDeviceId_c;
			Bas_Unsubscribe(&basServiceConfig, peerDeviceId);
			Tms_Unsubscribe();

			/* Flash LED */
	    	LED_SetColor(s_ledMonochromeHandle[0], kLED_White);
			LED_Flash(s_ledMonochromeHandle[0], &ledFlash);

			// start advertise
			App_Start();
		}
		break;

		default:
			break;
	}
}

button_status_t button_callback(void *buttonHandle, button_callback_message_t *message, void *callbackParam)
{
    button_status_t status = kStatus_BUTTON_Success;

    switch (message->event)
    {
        case kBUTTON_EventOneClick:
        case kBUTTON_EventShortPress:
        case kBUTTON_EventDoubleClick:
#ifdef DEBUG_PRINT_ENABLE
            PRINTF("Button Pressed \r\n");
#endif
            g_ButtonPress = TRUE;
            break;
        case kBUTTON_EventLongPress:
#ifdef DEBUG_PRINT_ENABLE
            PRINTF("Button Long Pressed \r\n");
#endif
            g_ButtonLongPress = TRUE;
            break;
        case kBUTTON_EventError:
            PRINTF("Button Press Error Event \r\n");
            break;
        default:
            status = kStatus_BUTTON_Error;
            break;
    }

    return status;
}

/*! *********************************************************************************
* \brief        Button Initialization and configures button Interrupt.
*
********************************************************************************** */
void Button_Initialization()
{
	timer_config_t timerConfig;

	timerConfig.instance    = 0U;
	timerConfig.srcClock_Hz = TIMER_SOURCE_CLOCK;
	TM_Init(&timerConfig);

	button_handle_t s_buttonHandle = (button_handle_t)&s_buttonHandleBuffer[0];
	button_config_t buttonConfig;

	buttonConfig.gpio.port = 0;
	buttonConfig.gpio.pin = BOARD_SW3_GPIO_PIN;
	buttonConfig.gpio.pinStateDefault = 1;

	BUTTON_Init(s_buttonHandle, &buttonConfig);

	BUTTON_InstallCallback(s_buttonHandle, button_callback, NULL);
}

/*! *********************************************************************************
* \brief        Configures GAP Advertise parameters. Advertise will start after
*               the parameters are set.
*
********************************************************************************** */
static void App_SetAdvertisingParameter(void)
{
	Gap_SetAdvertisingParameters(&gAdvParams);
}

/*! *********************************************************************************
* \brief        Sends temperature value over-the-air.
*
********************************************************************************** */
void BleApp_SendTemperature(void)
{
	if(xTimerIsTimerActive(timerHandle) != pdFALSE)
	{
		//Stop advertising timer */
		xTimerStop(timerHandle, 0);
	}

    /* Update with initial temperature */
    Tms_RecordTemperatureMeasurement(tmsServiceConfig.serviceHandle, (BOARD_GetTemperature()*100 ));
}
