/*! *********************************************************************************
 * \addtogroup Temperature Custom Profile
 * @{
 ********************************************************************************** */
/*! *********************************************************************************
* The Clear BSD License
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
* All rights reserved.
* 
* \file
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted (subject to the limitations in the disclaimer
* below) provided that the following conditions are met:
* 
* * Redistributions of source code must retain the above copyright notice,
*   this list of conditions and the following disclaimer.
* 
* * Redistributions in binary form must reproduce the above copyright notice,
*   this list of conditions and the following disclaimer in the documentation
*   and/or other materials provided with the distribution.
* 
* * Neither the name of the copyright holder nor the names of its contributors
*   may be used to endorse or promote products derived from this software
*   without specific prior written permission.
* 
* NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE
* GRANTED BY THIS LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT
* HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
* WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
* IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
********************************************************************************** */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
//#include "FunctionLib.h"
#include "ble_general.h"
#include "gattdb_interface.h"
#include "gatt_server.h"
#include "gap_interface.h"
#include "temperature_interface.h"
#include "ble_sig_defines.h"
#include "fsl_debug_console.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/*! Temperature Service - Subscribed Client*/
static deviceId_t mTms_SubscribedClientId;

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/
static void Hts_SendTemperatureMeasurementNotification
(
    uint16_t handle
);
/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Tms_Start (tmsConfig_t *pServiceConfig)
{    
    mTms_SubscribedClientId = gInvalidDeviceId_c;
    
    return Tms_RecordTemperatureMeasurement (pServiceConfig->serviceHandle,
                                             pServiceConfig->initialTemperature);

}

bleResult_t Tms_Stop (tmsConfig_t *pServiceConfig)
{
    return Tms_Unsubscribe();
}

bleResult_t Tms_Subscribe(deviceId_t deviceId)
{
    mTms_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Tms_Unsubscribe()
{
	mTms_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Tms_RecordTemperatureMeasurement (uint16_t serviceHandle, int16_t temperature)
{
    uint16_t  handle;
    bleResult_t result;
    bleUuid_t uuid = Uuid16(gBleSig_Temperature_d);

    /* Get handle of Temperature characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType16_c, &uuid, &handle);

    if (result != gBleSuccess_c)
        return result;

    /*Wait till FSCI second response is filled from FSCIRecvPacketTask, so application get handle value */
    vTaskDelay(pdMS_TO_TICKS(100));

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint16_t), (uint8_t*)&temperature);

    if (result != gBleSuccess_c)
        return result;

    Hts_SendTemperatureMeasurementNotification(handle);

    return gBleSuccess_c;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

static void Hts_SendTemperatureMeasurementNotification
(
  uint16_t handle
)
{
    uint16_t  hCccd;
    bool_t isNotificationActive;
    bleResult_t result;
    /* Get handle of CCCD */
    if (GattDb_FindCccdHandleForCharValueHandle(handle, &hCccd) != gBleSuccess_c)
        return;

    /*Wait till FSCI second response is filled, so application get handle value */
    vTaskDelay(pdMS_TO_TICKS(100));

    if (gBleSuccess_c == Gap_CheckNotificationStatus
        (mTms_SubscribedClientId, hCccd, &isNotificationActive))
	{
    	/*Wait till FSCI second response is filled from FSCIReceivePacketTask, so application get isNotificationActive value */
    	vTaskDelay(pdMS_TO_TICKS(60));

    	if(TRUE == isNotificationActive)
		{
			result = Gatt_SendNotification(mTms_SubscribedClientId, handle);
			if (result == gBleSuccess_c)
			{
				PRINTF("\r\n---------- TEMPERATURE UPDATED SUCCESS ----------\r\n");
			}
		}
	}
}

/*! *********************************************************************************
 * @}
 ********************************************************************************** */
