/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include "stdint.h"

#include "ApplMain.h"
#include "FSCI_main.h"
#include "fsl_debug_console.h"
#include "SPI.h"
#include "UART_main.h"
#include "temperature_sensor.h"
/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/

/*******************************************************************************
 * Private memory declarations
 ******************************************************************************/

/*******************************************************************************
 * Private functions Prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
/* Rx queue handle */
osa_msgq_handle_t rx_msg_queue = NULL;
/*! *********************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
********************************************************************************** */
bool_t is3SecTimerExpired = FALSE;
#ifdef gTempTimerSupported
TimerHandle_t sendTempTimerHandle =  NULL;
#define TEMP_TIMER_INTERVAL 3000
#endif
/*******************************************************************************
 ********************************************************************************
 * Public functions
 ******************************************************************************
********************************************************************************/

/*! *********************************************************************************
* \brief  This function performs a XOR over the message to compute the CRC
*
* \param[in]  pBuffer - pointer to the message
* \param[in]  size - the length of the message
*
* \return  the CRC of the message
*
********************************************************************************** */
uint8_t FSCI_computeChecksum( void *pBuffer, uint16_t size )
{
    uint16_t index;
    uint8_t  checksum = 0;

    for ( index = 0; index < size; index++ )
    {
        checksum ^= ((uint8_t*)pBuffer)[index];
    }

    return checksum;
}

/*! *********************************************************************************
* \brief  Form FSCI packet and send messages over the serial interface
*
* \param[in] opGroup operation Group
* \param[in] opCode operation Code
* \param[in] payload pointer to payload
* \param[in] Length length of the payload
********************************************************************************** */
void FSCI_transmitPayload(uint8_t opGroup, uint8_t opCode, uint8_t *payload, uint16_t Length)
{
	uint8_t *buffer_ptr = NULL;
	packetHdr_t header;
	uint8_t checksum;
	uint16_t buffer_size = 0, index = 0;

	/* Compute size */
	buffer_size = sizeof(packetHdr_t) + Length + gFsci_TailBytes_c;

	/* Allocate buffer */
	buffer_ptr = malloc(buffer_size);
	if(buffer_ptr == NULL)
	{
		return;
	}

	/* Message Header */
	header.startMarker = gFSCI_StartMarker_c;
	header.opGroup = opGroup;
	header.opCode = opCode;
	header.len = Length;

	/* Compute CRC for TX packet, on opcode group, opcode, payload length, and payload fields */
	checksum = FSCI_computeChecksum((uint8_t*)&header + 1, sizeof(header) - 1);

	if((payload != NULL) && (Length != 0))
	{
		checksum ^= FSCI_computeChecksum(payload, Length);
#ifdef DEBUG_PRINT_ENABLE
		PRINTF("\r\nREQ_PAYLOAD:\r\n");
#endif
	}

	memcpy(&buffer_ptr[index], &header, sizeof(header));
	index += sizeof(header);
	memcpy(&buffer_ptr[index], payload, Length);
	index += Length;

	buffer_ptr[index] = checksum;

#ifdef DEBUG_PRINT_ENABLE
	for(int i = 5; i < index; i++)
	{
		PRINTF("%X ", buffer_ptr[i]);
	}
	PRINTF("\r\n");
#endif

#ifdef gUARTSupported
	SendDataUART(buffer_ptr, buffer_size);
#elif gSPISupported
	SPI_SendData(buffer_ptr, buffer_size);
#endif

	// Free buffer once it is send over UART/SPI
	free(buffer_ptr);
	buffer_ptr = NULL;
}

/*! *********************************************************************************
* \brief   Handles all messages received from BLE Abstraction layer.
*
* \param[in]    pMsg  Pointer to msgBLEabsToFSCI_t.
********************************************************************************** */
void FSCI_HandleMessageInput(msgBLEabsToFSCI_t *pMsg)
{
	PRINTF("\r\n REQUEST : opCode = %X opGroup = %X\r\n", pMsg->opCode, pMsg->opGroup);

	switch(pMsg->opGroup)
	{
		case bleGAP:
		{
			if(pMsg->opCode == gapCheckNotificationStatus)
			{
				gapCheckNotificationStatus_t *payload = &pMsg->bleRequestType.checkNotificationStatus;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gapSetAdvParameter)
			{
				gapAdvertisingParameters_t *payload = &pMsg->bleRequestType.advParameter;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gapStartAdvertisement)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gapStopAdvertisement)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gapSetAdvData)
			{
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;

				payload[i++] = pMsg->bleRequestType.setadvData.cAdvDataInc;
				payload[i++] = pMsg->bleRequestType.setadvData.aAdvData.cNumAdStructures;

				for(int index = 0; index < (pMsg->bleRequestType.setadvData.aAdvData.cNumAdStructures); index++)
				{
					payload[i++] = pMsg->bleRequestType.setadvData.aAdvData.aAdStructures[index].length;
					payload[i++] = pMsg->bleRequestType.setadvData.aAdvData.aAdStructures[index].adType;

					for(int len_cnt = 0; len_cnt < (pMsg->bleRequestType.setadvData.aAdvData.aAdStructures[index].length); len_cnt++)
					{
						payload[i++] = (uint8_t)pMsg->bleRequestType.setadvData.aAdvData.aAdStructures[index].aData[len_cnt];
					}
				}
				payload[i++] = pMsg->bleRequestType.setadvData.cScanDataInc;

				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gapDisconnect)
			{
				gapDisconnect_t *payload = &pMsg->bleRequestType.disConnect;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}
		}
		break;

		case bleGATT:
		{
			if(pMsg->opCode == gattServerCallbackRegister)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattSendNotification)
			{
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;
				payload[i++] = pMsg->bleRequestType.sendNotification.deviceId;
				payload[i++] = pMsg->bleRequestType.sendNotification.handle;
				payload[i++] = (uint8_t)(pMsg->bleRequestType.sendNotification.handle >> 8);

				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}
		}
		break;

		case bleGATTDB:
		{
			if(pMsg->opCode == gattDBInit)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBReleaseDB)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBAddPrimaryService)
			{
				/* Need to fill payload in this way, as pServiceUuid is variable in size */
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;
				payload[i++] = (uint8_t)pMsg->bleRequestType.addPrimaryService.desireHandle;
				payload[i++] = (uint8_t)(pMsg->bleRequestType.addPrimaryService.desireHandle >> 8);
				payload[i++] = pMsg->bleRequestType.addPrimaryService.serviceUuidType;
				if(pMsg->bleRequestType.addPrimaryService.serviceUuidType == gBleUuidType16_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid >> 8);
				}
				else if(pMsg->bleRequestType.addPrimaryService.serviceUuidType == gBleUuidType32_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid >> 8);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid >> 16);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addPrimaryService.pServiceUuid >> 24);
				}
				else if(pMsg->bleRequestType.addPrimaryService.serviceUuidType == gBleUuidType128_c)
				{
					memcpy(&payload[i], pMsg->bleRequestType.addPrimaryService.pServiceUuid->uuid128, 16);
				}
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBAddSecondaryService)
			{
				gattDBAddSecondaryServiceReq_t *payload = &pMsg->bleRequestType.addSecondaryService;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBAddCharDescAndValue)
			{
				/* Need to fill payload in this way, as pCharacteristicUuid and maxValueLength
				 * are variable in size */
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;

				payload[i++] = pMsg->bleRequestType.addCharDecandValue.characteristicUuidType;
				if(pMsg->bleRequestType.addCharDecandValue.characteristicUuidType == gBleUuidType16_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid >> 8);
				}
				else if(pMsg->bleRequestType.addCharDecandValue.characteristicUuidType == gBleUuidType32_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid >> 8);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid >> 16);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid >> 24);
				}
				else if(pMsg->bleRequestType.addCharDecandValue.characteristicUuidType == gBleUuidType128_c)
				{
					for(int index = 0; index < 16; index++)
					{
						payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDecandValue.pCharacteristicUuid >> (index * 8));
					}
				}
				payload[i++] = pMsg->bleRequestType.addCharDecandValue.characteristicProperties;
				payload[i++] = (uint8_t)pMsg->bleRequestType.addCharDecandValue.maxValueLength;
				payload[i++] = (uint8_t)(pMsg->bleRequestType.addCharDecandValue.maxValueLength >> 8);
				payload[i++] = (uint8_t)pMsg->bleRequestType.addCharDecandValue.initialValueLength;
				payload[i++] = (uint8_t)(pMsg->bleRequestType.addCharDecandValue.initialValueLength >> 8);
				for(int init_cnt = 0; init_cnt < (pMsg->bleRequestType.addCharDecandValue.initialValueLength); init_cnt++)
				{
					payload[i++] = *(uint8_t*)pMsg->bleRequestType.addCharDecandValue.aInitialValue;
					pMsg->bleRequestType.addCharDecandValue.aInitialValue++;
				}
				payload[i++] = (uint8_t)pMsg->bleRequestType.addCharDecandValue.valueAccessPermissions;

				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBAddCharDescriptor)
			{
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;
				payload[i++] = pMsg->bleRequestType.addCharDescriptor.descriptorUuidType;

				if(pMsg->bleRequestType.addCharDescriptor.descriptorUuidType == gBleUuidType16_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid >> 8);
				}
				else if(pMsg->bleRequestType.addCharDescriptor.descriptorUuidType == gBleUuidType32_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid >> 8);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid >> 16);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid >> 24);
				}
				else if(pMsg->bleRequestType.addCharDescriptor.descriptorUuidType == gBleUuidType128_c)
				{
					for(int index = 0; index < 16; index++)
					{
						payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.addCharDescriptor.pDescriptorUuid >> (index * 8));
					}
				}
				payload[i++] = (uint8_t)pMsg->bleRequestType.addCharDescriptor.descriptorValueLength;
				payload[i++] = (uint8_t)(pMsg->bleRequestType.addCharDescriptor.descriptorValueLength >> 8);

				for(int init_cnt = 0; init_cnt < (pMsg->bleRequestType.addCharDescriptor.descriptorValueLength); init_cnt++)
				{
					payload[i++] = *(uint8_t*)pMsg->bleRequestType.addCharDescriptor.aInitialValue;
					pMsg->bleRequestType.addCharDescriptor.aInitialValue++;
				}
				payload[i++] = (uint8_t)pMsg->bleRequestType.addCharDescriptor.descriptorAccessPermission;

				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBAddCccd)
			{
				uint8_t *payload = NULL;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBWriteAttribute)
			{
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;
				payload[i++] = pMsg->bleRequestType.writeAttribute.handle;
				payload[i++] = (pMsg->bleRequestType.writeAttribute.handle >> 8);
				payload[i++] = pMsg->bleRequestType.writeAttribute.valueLength;
				payload[i++] = (pMsg->bleRequestType.writeAttribute.valueLength >> 8);

				for(int cnt = 0; cnt < pMsg->bleRequestType.writeAttribute.valueLength; cnt++)
				{
					payload[i++] = (uint8_t)(*(uint8_t*)pMsg->bleRequestType.writeAttribute.aValue);
					pMsg->bleRequestType.writeAttribute.aValue++;
				}
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBReadAttribute)
			{
				gattDBReadAttributeReq_t *payload = &pMsg->bleRequestType.readAttribute;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBFindServiceHandle)
			{
				gattDBFindSeriveHandleReq_t *payload = &pMsg->bleRequestType.findServiceHandle;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBFindCharValueHandleInService)
			{
				uint8_t message[pMsg->length];
				uint8_t *payload = message;
				uint8_t i = 0;
				payload[i++] = pMsg->bleRequestType.findCharValueHandleInService.serviceHandle;
				payload[i++] = (pMsg->bleRequestType.findCharValueHandleInService.serviceHandle >> 8);
				payload[i++] = pMsg->bleRequestType.findCharValueHandleInService.characteristicUuidType;

				if(pMsg->bleRequestType.findCharValueHandleInService.characteristicUuidType == gBleUuidType16_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid >> 8);
				}
				else if(pMsg->bleRequestType.findCharValueHandleInService.characteristicUuidType == gBleUuidType32_c)
				{
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid >> 8);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid >> 16);
					payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid >> 24);
				}
				else if(pMsg->bleRequestType.findCharValueHandleInService.characteristicUuidType == gBleUuidType128_c)
				{
					for(int index = 0; index < 16; index++)
					{
						payload[i++] = (uint8_t)(*(uint16_t*)pMsg->bleRequestType.findCharValueHandleInService.pCharacteristicUuid >> (index * 8));
					}
				}
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBFindCccdHandleForCharValHandle)
			{
				gattDBFindCccdHandleForCharValueHandleReq_t *payload = &pMsg->bleRequestType.findCccdHandleForCharValueHandle;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}

			if(pMsg->opCode == gattDBFindDescHandleForCharvalHandle)
			{
				gattDBFindDescHandleForCharValueHandleReq_t *payload = &pMsg->bleRequestType.findDescHandleForCharValueHandle;
				FSCI_transmitPayload(pMsg->opGroup, pMsg->opCode, (uint8_t*)payload, pMsg->length);
			}
		}
		break;

		default:
		{
		}
		break;
	}
}

#ifdef gTempTimerSupported
/*! *********************************************************************************
* \brief        Sending temperature value (at every 3 seconds Interval) timer callback.
*
* \param[in]    xTimer        Timer handle.
********************************************************************************** */
void SendTempTimerCallback(TimerHandle_t xTimer)
{
	if (mPeerDeviceId != gInvalidDeviceId_c)
	{
		is3SecTimerExpired = TRUE;
	}
}
#endif

/*! *********************************************************************************
* \brief   FSCI task that checks continuously whether there is any message
* 		   available in message queue from BLE Abstraction layer.
*
* \param[in]    param        argument passed to the task function.
********************************************************************************** */
void FSCI_SendPacketTask(void *argument)
{
	osa_event_flags_t event = 0;

#ifdef gTempTimerSupported
	sendTempTimerHandle = xTimerCreate("SendTempTimer", pdMS_TO_TICKS(TEMP_TIMER_INTERVAL), pdTRUE, (void*)0, SendTempTimerCallback);
//	if(sendTempTimerHandle != NULL)
//	{
//		// Start sending temperature value timer
//		xTimerStart(sendTempTimerHandle, 0);
//	}
#endif

	while(1)
	{
		OSA_EventWait(&eventHandle, osaEventFlagsAll_c, FALSE, 0, &event);

		if(event & gFSCIEvtMsgFromAbs_c)
		{
			msgBLEabsToFSCI_t *pMsgIn = malloc(MAX_REQUEST_SIZE * sizeof(uint8_t));

			/* Check for exiting messages from BLE Abstraction layer */
			if((OSA_MsgQGet(&tx_msg_queue, pMsgIn, 0)) == KOSA_StatusSuccess)
			{
				if(pMsgIn)
				{
					FSCI_HandleMessageInput(pMsgIn);

					/* Messages must always be freed. */
					free(pMsgIn);
					pMsgIn = NULL;
				}
			}
			else
			{
				vTaskDelay(pdMS_TO_TICKS(5));
			}
		}
		vTaskDelay(pdMS_TO_TICKS(50));
	}
}

/*!
 * @brief FSCIInit function
 *
 * \param[in]   queueHandle  queue handle.
 * \param[in]   queue_length length of queue.
 *
 *********************************************************************************** */
osa_status_t FSCIInit(void)
{
	osa_status_t result = kStatus_Success;

	result = OSA_MsgQCreate(&rx_msg_queue, MAX_QUEUE_LENGTH, MSG_SIZE);

	return result;
}
