/*! *********************************************************************************
 * \defgroup RSSI ranging Application
 * @{
 ********************************************************************************** */
/*! *********************************************************************************
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* Copyright 2016-2018 NXP
* All rights reserved.
*
* \file
*
* This file is the interface file for the RSSI ranging application
*
* SPDX-License-Identifier: BSD-3-Clause
********************************************************************************** */
#ifndef _RSSI_RANGING_H_
#define _RSSI_RANGING_H_

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
// Default numbers of RSSI sampling 
#ifndef gRSSIDefaultSampls
#define gRSSIDefaultSampls      30U
#endif

// Max numbers of RSSI sampling
#ifndef gRSSIMaxSampls
#define gRSSIMaxSampls          100U
#endif

#define gAvgWindow              gRSSIDefaultSampls


/*
 * Standard Deviation Type define
 */
typedef enum{
    SD_sample     = 0,
    SD_poulation  = 1
}StandardDeviationType_t;

/*
 * 
 */
typedef enum{
    GS_CalcAvg            = 0x0001,    /* Calculate the avg after gaussian smoothing */
    GS_SelMidValue        = 0x0002,    /* select the mid value after gaussian smoothing */
    GS_MixedAvgMid        = 0x0003,    /* Mixed the avg and mid value after gaussian smoothing*/
}GaussianMixedSmoothingType_t;

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
* \brief        The distance is calculated by RSSI
*
*  RSSI = A - ( 10 * n * lgD )
*
*  RSSI:Rx RSSI value when ranging
*  A: Rx RSSI value at a 1 m
*  n: attenuation factor
*  D: distance
*            ||
*            ||
*           \  /
*            \/        
*  D = 10 ^ [( A - RSSI ) / (10 * n)]
*
* \param[in]    rssi        RSSI that measured value.
********************************************************************************** */
float RSSI_DistanceCalc(float rssi);

/*! *********************************************************************************
* \brief        Set the gRssiValueAtOneMeter value
*
* \param[in]    refer_value      RSSI value at a 1 m.
********************************************************************************** */
void RSSI_SetRssiReferenceValue(int8_t refer_value);

/*! *********************************************************************************
* \brief        Get the gRssiValueAtOneMeter value
*
* \param[out]   gRssiValueAtOneMeter
********************************************************************************** */
int8_t RSSI_GetRssiReferenceValue(void);

/*! *********************************************************************************
* \brief        Set the gAttenuationFactor value
*
* \param[in]    factory        attenuation factor.
********************************************************************************** */
void RSSI_SetAttenuationFactor(float factory);

/*! *********************************************************************************
* \brief        Get the gAttenuationFactor value
*
* \param[out]   gAttenuationFactor.
********************************************************************************** */
float RSSI_GetAttenuationFactor(void);

/*! *********************************************************************************
* \brief       Calculate the average value of n data within the data pointed by the pointer
*
* \param[in]   pValue             Pointer to  input RSSI data     
* \param[in]   n                  number of data 
*
* \return                         average value.
********************************************************************************** */
float RSSI_CalcAverage (int8_t * pValue , uint32_t n );


/*! *********************************************************************************
* \brief       Sort n data by value, and select the middle value
*      
* \param[in]   pValue             Pointer to input RSSI data     
* \param[in]   n                  number of data 
*
* \return                         middle value.
********************************************************************************** */
float RSSI_MiddleSmoothing (int8_t * pValue , uint32_t n );

/*! *********************************************************************************
* \brief       Gaussian Smoothing
* \description Select the RSSI data which rssi value is within the gaussian interval. 
*
* \param[in]   pInValue           Pointer to input RSSI data     
* \param[in]   n                  number of data 
* \param[out]  pOutValue          Pointer to output RSSI data the gaussian interval
* \param[out]  pCount             data count in the gaussian interval                
*                                                 
* \return                         True if get data in gaussian interval, false otherwise. 
********************************************************************************** */
bool RSSI_GaussianSmoothing (int8_t * pInValue , uint32_t n , int8_t * pOutValue , uint32_t * pCount);

/*! *********************************************************************************
* \brief       Gaussian mixed other filtering
* \description Select the RSSI data which rssi value is within the gaussian interval. 
*              Then calculate the avg or mid value or the avg of the first two.
*
* \param[in]   pValue             Pointer to input RSSI data     
* \param[in]   number             number of data that need to calculate
* \param[in]   MixType
*
* \return                         RSSI value.
********************************************************************************** */
float RSSI_GaussianMixedSmoothing (int8_t * pValue , uint32_t number , GaussianMixedSmoothingType_t MixType);


#endif