/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2021, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "Defines.h"
#include "fsl_smc.h"
#include "fsl_afe.h"
#include "fsl_lptmr.h"
#include "fsl_vref.h"
#include "fsl_pmc.h"
#include "EEPROMDriver.h"
#include "Application.h"
#include "PowerModes.h"
#include "IOControls.h"
#include "MeteringLPRT.h"
#include "Timer.h"
#include "MeteringInterface1Ph.h"
#include "ComPortDriver.h"
#include "UserInterface.h"
#include "Calibration1Ph.h"
#include "lcd.h"
#include "AppInterface.h"
#include "IOControls.h"
#include "clock_config.h"
#include "AppCommon.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
* Prototypes
******************************************************************************/
uint8 TurnLatchOff;
uint8 BattModeTimeout;

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief This function monitors if there's a state of power supply change in the meter,
 * e.g., mains to battery or vice-versa. Initiates actions based on power state changes.
 */
void DoPowerStatus(void)
{
  uint8 PowerMode;
  
  PowerMode = SystemState.PowerMode;
  /* Read the latest status of the power supply - mains or battery */
  CheckPowerStatus();
  if ((TurnLatchOff == TRUE) && (BattModeTimeout == 0))
  {
    /* Meter is in battery mode and now need to go to sleep/low power mode */
    /* Disable the GPIOs and MCU IPS to reduce power consumption */
    ShutDownGPIOs();
    ShutDownIPs();
    DISABLE_LCD();
    /* Finally, go to sleep mode */
    GoToSleep();
  }
  
  if ((PowerMode == POWERMODE_MAINS) && (SystemState.PowerMode == POWERMODE_BAT))
  {
    /* just entered POWERMODE_MAINS to POWERMODE_BAT state */
    ShutDownGPIOs();
    ShutDownIPs();
    DISABLE_LCD();
    SIM->MISC_CTL   = 0x800F0000;
    
    NVIC_SystemReset();
  }
  
  if ((PowerMode == POWERMODE_BAT) && (SystemState.PowerMode == POWERMODE_MAINS))
  {
    NVIC_SystemReset();
  }
}

/*!
 * @brief Intializes MCU system, bus and flash clocks. 
 * PLL/FLL is also initilized based on power states meter or bandwidth requirement 
 * in meter processing.
 * Clocks are limited to meet power consumption(VA) rating of the meter as per meter tender requirements.
 * Mostly, the MCU is not run at highest possible frequencies.
 */
void ClockModuleInit(void)
{
  switch (SystemState.PowerMode)
  {
    case POWERMODE_MAINS:
      
      /* Set system frequency and bus clock to be high clock mode */
      SystemState.BusClk = BUS_CLOCK;
      SystemState.SysClk = SYSTEM_CLOCK; 
      
      while (kSMC_PowerStateRun != SMC_GetPowerModeState(SMC))
        SMC_SetPowerModeRun(SMC);
      {
      }
      
      /* Initialize MCU clock source */
      BOARD_BootClockRUN();
      break;
 
    case POWERMODE_BAT:
      /* Single wire operation has only 1 mode in non-mains mode. Initialize it. */
      SystemState.BusClk = BATTERY_FREQ;
      SystemState.SysClk = BATTERY_FREQ_SYSTEM;  

      while (kSMC_PowerStateRun != SMC_GetPowerModeState(SMC))
        SMC_SetPowerModeRun(SMC);
      {
      }
      
      /* Initialize MCU clock source */
      BOARD_BootClockRUNBat();
    break;
  }

}

/*!
 * @brief Takes the MCU to sleep mode.
 */
void GoToSleep(void)
{
  smc_power_mode_vlls_config_t vlls_config;
  vlls_config.enablePorDetectInVlls0 = true;
  
  /* Check Mains On / PFAIL */
  if (GPIO_GET_MAINS_ON)
  {
    NVIC_SystemReset();
  }
  
  while (kSMC_PowerStateRun != SMC_GetPowerModeState(SMC))
        SMC_SetPowerModeRun(SMC);
  {
  }
  
  while (kSMC_PowerStateVlls != SMC_GetPowerModeState(SMC))
  {
    /* VLLS2 mode */
    vlls_config.subMode = kSMC_StopSub2;
    SMC_PreEnterStopModes();
    SMC_SetPowerModeVlls(SMC, &vlls_config);
  }
}

/*!
 * @brief Disbales few MCU IP functions to reduce power consumption in the MCU 
 * and in the meter.
 */
void ShutDownIPs(void)
{
  pmc_bandgap_buffer_config_t pmConfig;
  
  CLOCK_DisablePll0();
  pmConfig.enable = false;
  pmConfig.enableInLowPowerMode = false;
  PMC_ConfigureBandgapBuffer(PMC, &pmConfig);
}

