/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2021, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "stdio.h"
#include "common.h"
#include "MKM35Z7.h"
#include "defines.h"
#include "fsl_afe.h"
#include "fsl_lptmr.h"
#include "fsl_qtmr.h"
#include "fsl_vref.h"
#include "fsl_cmp.h"
#include "fsl_adc16.h"
#include "fsl_xbar.h"
#include "fsl_crc.h"
#include "math.h"
#include "Application.h"
#include "MeteringLPRT.h"
#include "MeteringInterface1Ph.h"
#include "Calibration1Ph.h"
#include "ComPortDriver.h"
#include "PowerModes.h"
#include "appstructures.h"
#include "UserInterface.h"
#include "Timer.h"
#include "AppInterface.h"
#include "EEPROMDriver.h"
#include "flash_FTFL.h"
#include "AppCommon.h"
#include "lcd.h"
#ifdef RTC_TEMP_COMP_ENABLED 
#include "RTCCompensation.h"
#endif

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
* Prototypes
******************************************************************************/
uint8  ResetPLL;
uint8  NoMetTO;
uint8  SWRCurrent;
tCalibStruct1Ph TempCalibStruct;
uint32 ActAccumulator;
uint32 ReactAccumulator;

float FreqDependentPhErr[nCURRENTS] = { PHASE_PHERR_PERHZ, NEUTRAL_PHERR_PERHZ};

afe_channel_config_t afeChnConfig;
mcg_pll_config_t pllConfig;

static void InitTimerChl2(void);
void InitLPTMR(void);
static void InitAFE(void);
static void UpdateFlashCalib(void);
void InitVREF(void);
static void InitCMP(void);
static uint8 ReadVerifyFlashCalib(void);  
static uint16 ComputeCRC(uint16 *Address, uint8 Size);

void InitCalibration(void);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Enables PGA of Afe channel used for phase current measurement.
 */
void EnableGain(void)
{
  afeChnConfig.pgaGainSelect = kAFE_PgaGain8;
  AFE_SetChannelConfig(AFE, PHASE_CURRENT_AFE_CH, &afeChnConfig);
  mlib1phdata.IsGainEnabled = TRUE;
}

/*!
 * @brief Disables PGA of AFE channel used for phase current measurement.
 */
void DisableGain(void)
{
  afeChnConfig.pgaGainSelect = kAFE_PgaDisable;
  AFE_SetChannelConfig(AFE, PHASE_CURRENT_AFE_CH, &afeChnConfig);
  mlib1phdata.IsGainEnabled = FALSE;
}

/*!
 * @brief Initializes AFE channels for voltage and current channels.
 */   
void InitAFE(void)
{
  vuint8 i=0;
  
  afe_config_t afeConfig;
  
#if (NSAMPLES==6000)
  afeConfig.enableLowPower = false;
  afeConfig.resultFormat = kAFE_ResultFormatRight;
  afeConfig.clockSource = kAFE_ClockSource1;
  afeConfig.clockDivider = kAFE_ClockDivider8;
  afeConfig.startupCount = 30U; /* startupCnt = (Clk_freq/Clk_div)*20e-6 */
  
  AFE_Init(AFE, &afeConfig);
#else
  afeConfig.enableLowPower = true;
  afeConfig.resultFormat = kAFE_ResultFormatRight;
  afeConfig.clockSource = kAFE_ClockSource1;
  afeConfig.clockDivider = kAFE_ClockDivider16;
  afeConfig.startupCount = 15U; /* startupCnt = (Clk_freq/Clk_div)*20e-6 */
  
  AFE_Init(AFE, &afeConfig);
#endif
  for(i=0;i<0xff;i++);
  
  
  mlib1phdata.VrmsSums[0] = 0;
  mlib1phdata.VrmsSums[1] = 0;
  mlib1phdata.nSamps = 0;
  
  afeChnConfig.enableHardwareTrigger      = false;
  afeChnConfig.enableContinuousConversion = true;
  afeChnConfig.channelMode                = kAFE_BypassDisable;
  afeChnConfig.decimatorOversampleRatio   = AFE_OSR;
  afeChnConfig.pgaGainSelect              = kAFE_PgaDisable;
  /* Initialise AFE for Phase current */
  AFE_SetChannelConfig(AFE, PHASE_CURRENT_AFE_CH, &afeChnConfig);
  AFE_SetChannelPhaseDelayValue(AFE, PHASE_CURRENT_AFE_CH, 0U);
  /* Initialise AFE for Neutral current */
  AFE_SetChannelConfig(AFE, NEUTRAL_CURRENT_AFE_CH, &afeChnConfig);
  AFE_SetChannelPhaseDelayValue(AFE, NEUTRAL_CURRENT_AFE_CH, 40U);
  /* Initialise AFE for Voltage */
  AFE_SetChannelConfig(AFE, VOLTAGE_AFE_CH, &afeChnConfig);
  AFE_SetChannelPhaseDelayValue(AFE, VOLTAGE_AFE_CH, 80U);
  
  NVIC_SetPriority(AFE_CH2_IRQn, VOLTAGE_AFE_INTERRUPT_PRIORITY);
  NVIC_EnableIRQ(AFE_CH2_IRQn);
  /* Disable DMA */
  AFE_EnableChannelInterrupts(AFE, (kAFE_Channel2InterruptEnable));
}

void InitializePXBAR(void)
{
    /* XBAR configuration for comparator's output to TMR2 input for trigger */
  XBAR_SetSignalsConnection(XBAR, XBAR_CMPOUT, kXBAR_OutputTmrCh2SecInput);
  
  /* XBAR configuration for Voltage AFE output to TMR3 input for trigger.
  It is a matter of great confusion through documentation. AFE channels
  are numbered from 1 through 4, but elsewhere in XBAR etc, they are referred
  as 0 to 3.
  */
  /* Configure the XBAR signal connections. */
  XBAR_SetSignalsConnection(XBAR, VOLTAGE_AFE_XBAR_COC, kXBAR_OutputTmrCh3SecInput);
}
 
/*!
 * @brief Initializes LPTMR. This timer is used to time the kWh LED pulsing as 
 * per meter constant cakue.
 */
void InitLPTMR(void)
{
  lptmr_config_t lptmrConfig;
  
  /* Setup LPTMR. */
  LPTMR_GetDefaultConfig(&lptmrConfig);
  lptmrConfig.prescalerClockSource = kLPTMR_PrescalerClock_2;
  lptmrConfig.bypassPrescaler      = false;
  
  LPTMR_Init(LPTMR0, &lptmrConfig);
  LPTMR_SetTimerPeriod(LPTMR0, 16U);
  /* Enable timer interrupt */
  LPTMR_EnableInterrupts(LPTMR0, kLPTMR_TimerInterruptEnable);
  NVIC_SetPriority(LPTMR0_LPTMR1_IRQn, LPTMR_INTERRUPT_PRIORITY);
  NVIC_EnableIRQ(LPTMR0_LPTMR1_IRQn);
  /* Start counting */
  LPTMR_StartTimer(LPTMR0);
}

void InitCMP(void)
{
  cmp_config_t mCmpConfigStruct;
  cmp_filter_config_t filterConfig;
  cmp_dac_config_t dacConfig;
  
  /* Initialize CPM1 to compare the AFE Voltage channel and MCU Internal DAC being fed with voltage reference */
  CMP_GetDefaultConfig(&mCmpConfigStruct);
  
  /* Initialize the CMP comparator. */
  CMP_Init(CMP1, &mCmpConfigStruct);
  filterConfig.filterCount = 0x7;
  filterConfig.enableSample = false;
  filterConfig.filterPeriod = 0xFF;
  CMP_SetFilterConfig(CMP1, &filterConfig);
  dacConfig.DACValue = 0U;
  dacConfig.referenceVoltageSource = kCMP_VrefSourceVin2;
  CMP_SetDACConfig(CMP1, &dacConfig);
  CMP_SetInputChannels(CMP1, CMP_IP_PSEL, CMP_IP_NSEL);
}

/* Running at 1/128th of bus clock, the timer will count 48000 counts per 
second. Comparing to 4800 means 100ms, which is the timeout for finding one 
voltage. Enough!!!
*/
void InitTimerChl2(void)
{
  qtmr_config_t qtmrConfig;
  
  /* Connect TMR2 secondary source channel from XBAR output */
  SIM->MISC_CTL |= SIM_MISC_CTL_TMR2SCSEL_MASK;
  /* Setup TMR channel 2 - Comparator for zero crossing */
  QTMR_GetDefaultConfig(&qtmrConfig);
  
  /* Set clock prescaler */
  qtmrConfig.primarySource = FREQ_COUNTER_CLOCK_DIV;
  qtmrConfig.secondarySource = kQTMR_Counter2InputPin;
  QTMR_Init(TMR2, &qtmrConfig);
  QTMR_SetTimerPeriod(TMR2, 0xFFFF);
  QTMR_SetupInputCapture(TMR2,  kQTMR_Counter2InputPin, false, false, kQTMR_RisingEdge);
  
  /* Enable timer compare interrupt */
  QTMR_EnableInterrupts(TMR2, kQTMR_EdgeInterruptEnable);
  
  /* Enable at the NVIC */
  NVIC_SetPriority(TMR2_IRQn, TMR2_INTERRUPT_PRIORITY);
  EnableIRQ(TMR2_IRQn);
  
  /* Start timer */
  QTMR_StartTimer(TMR2, kQTMR_PriSrcRiseEdge);
}

/*!
 * @brief Iniitializes VREF module so that the reference voltage can be used by other 
 * modules(e.g., Comparator with line voltage to generate an o/p to triggger the timer 
 * for frequency calculation.
 */
void InitVREF(void)
{
  uint32_t temp32;
  /* Do necessery initialization in the SIM module */
  temp32 = SIM->MISC_CTL & ~(SIM_MISC_CTL_VREFBUFPD_MASK | SIM_MISC_CTL_VREFBUFINSEL_MASK | SIM_MISC_CTL_VREFBUFOUTEN_MASK);
  temp32 |= SIM_MISC_CTL_VREFBUFPD(0) | SIM_MISC_CTL_VREFBUFINSEL(0) | SIM_MISC_CTL_VREFBUFOUTEN(1);
  SIM->MISC_CTL = temp32;
  
  /* VREF module must be initialized after SIM module                         */    
  vref_config_t config;
  
  /* Get vref default configure */
  VREF_GetDefaultConfig(&config);
#if defined(FSL_FEATURE_VREF_HAS_LOW_REFERENCE) && FSL_FEATURE_VREF_HAS_LOW_REFERENCE
  /* Enable low reference volt */
  config.enableLowRef = true;
#endif /* FSL_FEATURE_VREF_HAS_LOW_REFERENCE */
  /* Initialize vref */
  VREF_Init(VREF, &config);
}

/*!
 * @brief Initializes the MCU resources used to aid the core metrolgy calculation.
 */
void MeteringInit(void)
{
  NoMetTO = NOMET_TO;
  ResetPLL = FALSE;

  if (MeterLibLPRT1Ph_InitParams(&mlib1phdata, NSAMPLES, 600, FreqDependentPhErr, FALSE))
  {
    /* Parameter error */
    sprintf(disp_string,"par");
    lcd_PrintString();
    return;
  }
  
  /* Initialize few default/threshold values for metrology application */
  AppInterfaceInit();
  mlib1phdata.Frequency = 50.0f;
  ActAccumulator = 0;
  ReactAccumulator = 0;
  if (SystemState.MetMode == METMODE_MAINS)
  {
    CalibStruct1Ph.FrequencyCoeff = 1.0;
    InitTimerChl2();
    /* Initialize CPM1 to compare the AFE Voltage channel and MCU Internal DAC being fed with voltage reference */
    InitCMP();

    /* Check a meter condition: if mains_sense GPIO of the meter fell down intermittently */
    if ((!GPIO_GET_MAINS_ON) || (SystemState.PowerMode != POWERMODE_MAINS))
    {
      /* then reset the meter */
      NVIC_SystemReset();
    }

    PMC->REGSC |= PMC_REGSC_BGEN_MASK;
    SIM->CTRL_REG |= SIM_CTRL_REG_PLLVLPEN_MASK;
    MCG->C2 &= ~(MCG_C2_LP_MASK);

    pllConfig.refSrc = kMCG_PllRefRtc;
    pllConfig.enableMode = 0U;
    CLOCK_EnablePll0(&pllConfig);
    
    /* Initialize VREF and AFE */
    InitVREF();
    InitAFE();
    XBAR_Init(XBAR);

    /* Initialize MCU crossbar to trigger TMR2 for comparator's output */
    InitializePXBAR();
	
#ifdef RTC_TEMP_COMP_ENABLED 
    /* Initialize SAR ADC for temperature sensing */
    InitSARADC();
#endif
    
    /* Initialize LPTMR for calibration/pulse LED operation */
    InitLPTMR();
    
    /* Initialize calibration structure */
    InitCalibration();
    
    AFE_DoSoftwareTriggerChannel(AFE, AFE_CR_COMBINED_MASK);
  }
  
  /* Read meter RTC date, time in globals g_stDate, g_stTime and g_UTCTime */
  InitTime( );  
}

void SetCalibPhase(uint8 CalibPhase)
{
  CalibPoint.CalibPhase = CalibPhase;
  CalibStruct1Ph.CalibSign = CALIBNOTDONE;
  NVWriteIIC(CALIBADDRESS, (uint8 *)&CalibStruct1Ph.CalibSign, 4);
}

uint8 ReadVerifyCalib1Ph(uint32 Address)
{
  uint8  i;
  uint16 CalibCRC;
  
  for (i = 0; i < 3; i++)
  {
    /* Read */
    NVReadIIC(Address, (uint8 *)&TempCalibStruct, sizeof(CalibStruct1Ph));
    CalibCRC = ComputeCRC((uint16 *)&TempCalibStruct, 20);
    if (TempCalibStruct.CRC == CalibCRC)
    {
      return TRUE;
    }
  }
  return FALSE;
}

static void UpdateFlashCalib(void)
{
  uint32_t primaskValue = 0U;
  uint32  Address;
  status_t result; 
  
  Address = (uint32)PFlashCalibStruct;
  primaskValue = DisableGlobalIRQ();
  EraseSectors(Address, ERASE_SECTOR_SIZE);
  result = FLASH_Program(&s_flashDriver, Address, (uint8 *)&CalibStruct1Ph, sizeof(CalibStruct1Ph));
  if (kStatus_FTFx_Success != result)
  {
  }
  EnableGlobalIRQ(primaskValue);
}

void InitCalibration(void)
{
  uint8 FoundCalib;

  PFlashCalibStruct = (tCalibStruct1Ph *)APP_METADATA_START_ADDR;
  
  FoundCalib = FALSE;
  if (mlib1phdata.CalibState == CALIBSTATE_PROGRESS)
  {
    return;
  }
  if (ReadVerifyFlashCalib() == TRUE)
  {
    return;
  }

  if (ReadVerifyCalib1Ph(CALIBADDRESS) == TRUE)
  {
    FoundCalib = TRUE;
  }
  CalibStruct1Ph = TempCalibStruct;

  /* Check if other location has valid data */
  if (ReadVerifyCalib1Ph(CALIBBKPADDR) == TRUE)
  {
    CalibStruct1Ph = TempCalibStruct;
    if (FoundCalib == FALSE)
    {
      UpdateCalib(CALIBADDRESS);
    }
    FoundCalib = TRUE;
  }
  else
  {
    /* Backup location is messed up */
    if (FoundCalib == TRUE)
    {
      UpdateCalib(CALIBBKPADDR);
    }
  }

  if (FoundCalib == FALSE)
  {
    /* Didn't find any CRC match, but there could be data */
    if (TempCalibStruct.CalibSign == CALIBDONE)
    {
      CalibStruct1Ph = TempCalibStruct;
      CalibMemwrite1Ph();
    }
    else if (CalibStruct1Ph.CalibSign == CALIBDONE)
    {
      CalibMemwrite1Ph();
    }
    else
    {
      RestoreDefCalib();
    }
  }

  if (FoundCalib == TRUE)
  {
    UpdateFlashCalib();
    ApplyRTCCompensation(CalibStruct1Ph.RTCCompInterval, CalibStruct1Ph.RTCCompValue);
  }
}

void UpdateCalib(uint32 Address)
{
  for ( ; ; )
  {
    NVWriteIIC(Address, (uint8 *)&CalibStruct1Ph, sizeof(CalibStruct1Ph));
    if (ReadVerifyCalib1Ph(Address) == TRUE)
    {
      return;
    }
  }
}

uint8 ReadVerifyFlashCalib(void)
{
  uint16 CalibCRC;
  
  CalibStruct1Ph = *PFlashCalibStruct;

  CalibCRC = ComputeCRC((uint16 *)&CalibStruct1Ph, 20);
  if (CalibStruct1Ph.CRC == CalibCRC)
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}

void CalibMemwrite1Ph(void)
{
  CalibStruct1Ph.CalibSign = CALIBDONE;
  /* Compute CRC */
  CalibStruct1Ph.CRC = ComputeCRC((uint16 *)&CalibStruct1Ph, 20);
  UpdateFlashCalib();

  UpdateCalib(CALIBADDRESS);
  UpdateCalib(CALIBBKPADDR);
}

/*!
 * @brief Init for CRC-16/MAXIM.
 * @details Init CRC peripheral module for CRC-16/MAXIM protocol.
 *          width=16 poly=0x8005 init=0x0000 refin=true refout=true xorout=0xffff check=0x44c2 name="CRC-16/MAXIM"
 *          http://reveng.sourceforge.net/crc-catalogue/
 */
void InitCrc16(CRC_Type *base, uint32_t seed)
{
    crc_config_t config;

    config.polynomial         = 0x8005;
    config.seed               = seed;
    config.reflectIn          = true;
    config.reflectOut         = true;
    config.complementChecksum = true;
    config.crcBits            = kCrcBits16;
    config.crcResult          = kCrcFinalChecksum;

    CRC_Init(base, &config);
}

/*!
 * @brief Application specific CRC calculation function.
 */
static uint16 ComputeCRC(uint16 *Address, uint8 Size)
{
  CRC_Type *base = CRC0;
  InitCrc16(base, 0x0U);
  CRC_WriteData(base, (uint8_t *)Address, Size);
  return CRC_Get16bitResult(base);
}

