/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016-2021, NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "Defines.h"
#include "MeteringLPRT.h"
#include "AppInterface.h"
#include "MeteringInterface1Ph.h"
#include "Calibration1Ph.h"
#include "EEPROMDriver.h"
#include "PowerModes.h"
#include "UpgradeFW.h"
#include "Flash_FTFL.h"
#include "lcd.h"
#include "ComPortDriver.h"
#include "UserInterface.h"
#include "AppCommon.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
 
/*******************************************************************************
* Prototypes
******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Holds any initialization required between this interface layer which 
 * is used between application and metrology core.
 */
void AppInterfaceInit(void)
{
  mlib1phdata.IMax   = 60.0;
  mlib1phdata.IBasic = 10.0;
  mlib1phdata.VHystHigh = 160.0;
  mlib1phdata.VHystLow = 155.0;
  mlib1phdata.MaxPower = 240.0 * mlib1phdata.IMax * 1.6;
}

/*!
 * @brief Restores a default calibration data object for the meter. 
 * These restored values in the calibration data structure can be tuned to close 
 * values as per meter hw.
 */
void RestoreDefCalib(void)
{
  CalibStruct1Ph.FrequencyCoeff = 1.00023889;

  CalibStruct1Ph.VCoeff = 3.39386974E-6;
  CalibStruct1Ph.ICoeff[0] = 4.6775420511E-7;
  CalibStruct1Ph.ICoeff[1] = 3.10757769E-7;
  CalibStruct1Ph.ActPowerCoeff[0] = 1.5870693433E-12;
  CalibStruct1Ph.ActPowerCoeff[1] = 8.37127351E-13;
  CalibStruct1Ph.PhAngle[0] = 2.24286317E-2;
  CalibStruct1Ph.PhAngle[1] = -3.33993435E-2;
  CalibStruct1Ph.VFCoeff = 5.52673412E-13;
  CalibStruct1Ph.IFCoeff[0] = 1.12404917E-13;
  CalibStruct1Ph.IFCoeff[1] = 6.38489369E-14;
  CalibStruct1Ph.ICoeff_LP = 3.73612852E-6;
  CalibStruct1Ph.ActPowerCoeff_LP = 1.2681085344E-11;
  
  CalibStruct1Ph.RTCCompValue = 0;
  CalibStruct1Ph.RTCCompInterval = 0;

  CalibStruct1Ph.CalibSign = CALIBNOTDONE;
}

void ChkVolLvl(void)
{
  if (mlib1phdata.CalibState == CALIBSTATE_PROGRESS)
  {
    return;
  }

  if (mlib1phdata.Vrms > mlib1phdata.VHystHigh)
  {
    EnableGain();
  }
  if (mlib1phdata.Vrms < mlib1phdata.VHystLow)
  {
    DisableGain();
  }
}

/*!
 * @brief Corrects phase angle as per Application requirements.
 */
void CorrectAppPhAngle1Ph(uint8 currenttoUse)
{
  if (currenttoUse == CURRENT_PHASE)
  {
    if ((mlib1phdata.PhAngles[currenttoUse] > 4.8) && (mlib1phdata.PhAngles[currenttoUse] < 6.0))
    {
      /* Example */
      /* mlib1phdata.PhAngles[currenttoUse] += 0.0024; */
    }
  }
  
  if (currenttoUse == CURRENT_NEUTRAL)    
  {
    if(mlib1phdata.Irms[currenttoUse] > 50.0)
    {
      /* Example */
      /* mlib1phdata.PhAngles[currenttoUse] -= 0.0034; */
    }
  }
}

/*!
 * @brief Metrology gives the application callback to fine tune metrology library parameters.
 */
void FudgeParameters1Ph(void)
{
  uint8  i;
  
  if (mlib1phdata.CalibState == CALIBSTATE_PROGRESS)
  {
    return;
  }
  
  if (mlib1phdata.FirstTime == TRUE)
  {
    mlib1phdata.FirstTime = FALSE;
    mlib1phdata.Irms[0] = 0.0;
    mlib1phdata.Irms[1] = 0.0;
  }  

  if (mlib1phdata.Vrms < 10.0)
  {
    mlib1phdata.Vrms = 0.0;
  }

  if (mlib1phdata.Vrms > 400.0)
  {
    if (mlib1phdata.Irms[0] < 0.1)
    {
      mlib1phdata.Irms[0] = 0.0;
    }
    if (mlib1phdata.Irms[1] < 0.1)
    {
      mlib1phdata.Irms[1] = 0.0;
    }
  }
  if ((mlib1phdata.CurToUse == CURRENT_PHASE) && (mlib1phdata.Vrms > 250.0))
  {
    if ((mlib1phdata.ActPowers[CURRENT_PHASE] == 0.0) && (mlib1phdata.AppPowers[CURRENT_PHASE] < 10.0))
    {
      mlib1phdata.Irms[CURRENT_PHASE] = 0.0;
    }
  }
  
  for (i = 0; i < nCURRENTS; i++)
  {
#if 0
    if ((ActPowers[i] == 0.0) && (ReactPowers[i] == 0.0))
    {
      mlib1phdata.Irms[i] = 0.0;
      mlib1phdata.AppPowers[i] = 0.0;
    }
#endif
    if ((mlib1phdata.Irms[i] < 0.150) && (mlib1phdata.Vrms > 100.0))
    {
      mlib1phdata.Irms[i] = mlib1phdata.AppPowers[i]/mlib1phdata.Vrms;
    }
  }
    
  mlib1phdata.CurToUse = CURRENT_PHASE;
  if (CalibStruct1Ph.CalibSign == CALIBDONE)
  {
    if (mlib1phdata.AppPowers[CURRENT_NEUTRAL] > mlib1phdata.AppPowers[CURRENT_PHASE])
    {
      mlib1phdata.CurToUse = CURRENT_NEUTRAL;
    }
  }
}

#define MERASE_OFST 20

/*!
 * @brief Meter specific routine to erase partitions of EEPROM memory.
 */
void MemErase(void)
{
  uint16 i, j;
  uint32 k;
  uint32 Address;
  
  DisplayErase();
  
  j = MERASE_OFST+PAGE_BOUNDARY;
  for (i = MERASE_OFST; i < j; i++)
  {
    UARTBuffer[i] = 0;
  }

  NVWriteIIC(MemSignAddr, (uint8 *)&UARTBuffer[MERASE_OFST], 4);
  
  Address = g_UTCTime;
  for (k = 4; k < 8; k++)
  {
    UARTBuffer[MERASE_OFST+k] = Address & 0xFF;
    Address >>= 8;
  }
    
  DisplayDone();
  lcd_PrintString( );
  SDK_DelayAtLeastUs(5000000/(SYSTEM_CLOCK/1000000), SYSTEM_CLOCK);
}