/* touch.c */
#include "touch.h"
#include "touch_hal.h"
#include <stdlib.h>

#include "led.h"
#include "fsl_mrt.h"

/* variable definitions. */

const uint16_t touch_channel_code[TOUCH_USER_CHANNEL_NUM] =
{
    0x1, 0x2, 0x4, 0x8, 0x10
};

/* timer. */
volatile uint32_t touch_mrt_clock_source_freq_hz;

/* touch. */
volatile int16_t touch_val_raw_index;
volatile int16_t touch_val_raw[TOUCH_USER_CHANNEL_NUM];
volatile int16_t touch_calib_baseline[TOUCH_USER_CHANNEL_NUM] = {0};
volatile int16_t touch_val[TOUCH_USER_CHANNEL_NUM]; /* unified value. */
volatile int16_t touch_val2[TOUCH_USER_CHANNEL_NUM]; /* unified value. */
volatile bool    touch_scan_round_done = false;


/* function declerations. */





/* //setup pins.
 * setup timer.
 * setup touch converter.
 * setup interrupts.
 * start the scan and let the touch values in variable update automatically.
 */
void touch_init(void)
{
    /* variables. */
    touch_val_raw_index = 0u;
    touch_scan_round_done = false;

    /* setup timer. */
    mrt_config_t mrt_config;

    touch_mrt_clock_source_freq_hz = CLOCK_GetFreq(kCLOCK_CoreSysClk);
    mrt_config.enableMultiTask = false;
    MRT_Init(MRT0, &mrt_config);
    MRT_SetupChannelMode(MRT0, kMRT_Channel_0, kMRT_RepeatMode);
    MRT_EnableInterrupts(MRT0, kMRT_Channel_0, kMRT_TimerInterruptEnable);
    NVIC_EnableIRQ(MRT0_IRQn);

    /* setup touch converter. */
    touch_hal_init_capt();
    touch_hal_enable_interrupt();

}

/* start timer. */
void touch_start(void)
{
    touch_scan_round_done = false;
    touch_val_raw_index = 0;
    MRT_StartTimer(MRT0, kMRT_Channel_0, USEC_TO_COUNT(7500u, touch_mrt_clock_source_freq_hz)); /* 8ms. */
}

/* stop the timer.
 * wait for the last conversion.
 */
void touch_pause(void)
{
    MRT_StopTimer(MRT0, kMRT_Channel_0);
}

/* ISR for timer.
 * trigger the capt conv timely.
 */
volatile bool touch_led_on_flag = false;
__weak void touch_one_channel_scan_done_hook(uint8_t channel_index)
{
}
void MRT0_IRQHandler(void)
{
    //uint32_t flags = MRT_GetStatusFlags(MRT0, kMRT_Channel_0);

    /* channel sensing. */
    if (0u != (MRT_GetStatusFlags(MRT0, kMRT_Channel_0) & kMRT_TimerInterruptFlag) )
    {
        MRT_ClearStatusFlags(MRT0, kMRT_Channel_0, kMRT_TimerInterruptFlag);

        /* periodical task. */
        touch_hal_init_pins();/* setup pins for captouch. */
        touch_hal_start_conv(touch_channel_code[touch_val_raw_index]);

        //led_on(1u << touch_val_raw_index); /* led. */
        touch_one_channel_scan_done_hook(touch_val_raw_index);
    }
}

/* ISR for capt conversion done.
 * read the sensing value on conversion done.
 */
__weak void touch_all_channels_scan_done_hook(void)
{
}
void CMP_CAPT_IRQHandler(void)
{
    //touch_val_raw[touch_val_raw_index] = (3 * touch_val_raw[touch_val_raw_index] + touch_hal_read_conv() ) /4;
    touch_val_raw[touch_val_raw_index] = touch_hal_read_conv();
    /* reset the pins after conversion done. */
    touch_hal_reset_pins();

    //touch_val_raw_index = (touch_val_raw_index + 1u) % TOUCH_USER_CHANNEL_NUM;
    if (touch_val_raw_index == (TOUCH_USER_CHANNEL_NUM-1u))
    {
        touch_pause();/* a round is done. */
        touch_scan_round_done = true;
        touch_val_raw_index = 0;
        //led_off((1u << TOUCH_USER_CHANNEL_NUM) -1); /* LED灯效. */
        touch_all_channels_scan_done_hook();
    }
    else
    {
        touch_val_raw_index++;
    }
}




/* 在应用层调用, 获取一个采样截面.
 * 应用要点, 每次捕获到一个采样截面缓存到应用层之后, 就可以启动下次采样.
 * 但此时主线程不需要等这组数据, 而是处理缓存好的采样截面. 此时转换器硬件也同时在采样.
 * 当主线程处理完之前缓存的采样截面之后:
 * - 如果还没采样完成, 那就只能等了, 因为没有数据也干不了活
 * - 如果采样完成, 直接缓存新的采样截面开始处理, 并且启动下一个采样截面的转换.
 */
void touch_wait_data_ready(int16_t *output)
{
    while (!touch_scan_round_done)
    {}

    for (int i = 0u; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        output[i] = touch_val_raw[i];
    }
}

/*
 * parameters input are for the input sensing values array.
 * return the sum(abs(err)).
 * user would check if the calibration is done.
 */
uint16_t touch_calib(int16_t * input)
{
    int16_t err;
    //uint16_t err_sum_abs = 0u;
    uint16_t err_max = 0u;

    for (int i = 0; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        err = input[i] - touch_calib_baseline[i];
        touch_calib_baseline[i] += (err / 4); /* update the baseline. */
        //err_sum_abs += abs(err);
        if (err >= err_max)
        {
            err_max = err;
        }
    }

    //return err_sum_abs;
    return err_max;
}

void touch_calib_update(int16_t *input)
{
    int16_t err;

    for (int i = 0; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        err = input[i] - touch_calib_baseline[i];
        touch_calib_baseline[i] += (err / 4); /* update the baseline. */
    }
}

/* 直接作用于原始样本, 滤除瞬间的脉冲. */
bool touch_verify(int16_t *input, int max_offset)
{
    int err_abs_max = 0, err_abs;

    for (int i = 0; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        err_abs = abs(input[i] - touch_calib_baseline[i]);
        if (err_abs >= err_abs_max)
        {
            err_abs_max = err_abs;
        }
    }

    return (err_abs_max <= max_offset);
}

/* return err code:
 * - 0: available touch key
 * - 1: no touch
 * - 2: noice
 */
#define TOUCH_YES_TOUCH_SHRESHOLD_LOW  4

touch_get_pos_err_t touch_get_pos(int16_t * val_raw, uint32_t pos_range, int32_t * pos)
{
    touch_get_pos_err_t err = eTouch_GetPosErr_UnKnown;
    uint16_t m_touch_val_max_idx = 0;
    uint16_t m_touch_val_min_idx = 0;
    int32_t m_touch_calc_val[2] = {0};
    int16_t len = TOUCH_USER_CHANNEL_NUM;
    uint16_t m_touch_val_calc_idx_start = 0;

    /* apply the calib offset. */
    touch_val[0] = touch_calib_baseline[0] - val_raw[0];
    for (int i = 1u; i < len; i++)
    {
        touch_val[i] = touch_calib_baseline[i] - val_raw[i];

        if (touch_val[i] > touch_val[m_touch_val_max_idx])
        {
            m_touch_val_max_idx = i;
        }

        if (touch_val[i] < touch_val[m_touch_val_min_idx])
        {
            m_touch_val_min_idx = i;
        }
    }

    /* 不在沉默中爆发, 便在沉默中消亡. */
    if (abs(touch_val[m_touch_val_max_idx] - touch_val[m_touch_val_min_idx]) < TOUCH_YES_TOUCH_SHRESHOLD_LOW)
    {
        err = eTouch_GetPosErr_NoTouch;
        return err;
    }

/* 仅仅选取最大值附近的3个采样通道. */
#define TOUCH_POS_CALC_CHANNEL_NUM 3
    if (m_touch_val_max_idx == 0)
    {
        m_touch_val_calc_idx_start = 0;
    }
    else if ( m_touch_val_max_idx >= (len-1) )
    {
        m_touch_val_calc_idx_start = (len-TOUCH_POS_CALC_CHANNEL_NUM);
    }
    else
    {
       m_touch_val_calc_idx_start = m_touch_val_max_idx-(TOUCH_POS_CALC_CHANNEL_NUM/2);
    }

    /* 找到计算范围内通道的最小值 */
    m_touch_val_min_idx = m_touch_val_calc_idx_start;
    for (int i = m_touch_val_calc_idx_start+1;
                i < m_touch_val_calc_idx_start+TOUCH_POS_CALC_CHANNEL_NUM;
                i++)
    {
        if (touch_val[i] < touch_val[m_touch_val_min_idx])
        {
            m_touch_val_min_idx = i;
        }
    }

    /* 将触摸值归一化. */
    for (int i = m_touch_val_calc_idx_start;
                 i < m_touch_val_calc_idx_start+TOUCH_POS_CALC_CHANNEL_NUM;
                 i++)
    {
        touch_val[i] -= touch_val[m_touch_val_min_idx];
    }

    m_touch_calc_val[0] = 0;
    m_touch_calc_val[1] = 0;
    for (int i = m_touch_val_calc_idx_start;
                 i < m_touch_val_calc_idx_start+TOUCH_POS_CALC_CHANNEL_NUM;
                 i++)
    {
        m_touch_calc_val[0] += touch_val[i] *i;
        m_touch_calc_val[1] += touch_val[i];
    }

    *pos = m_touch_calc_val[0] * (pos_range / (len-1)) / m_touch_calc_val[1];
    err = eTouch_GetPosErr_YesTouch;

    return err;
}

volatile int16_t touch_iir_filter_buf_internal[TOUCH_USER_CHANNEL_NUM];

void touch_iir_filter_init(int16_t *input)
{
    for (int i = 0u; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        touch_iir_filter_buf_internal[i] = input[i];
    }
}

void touch_iir_filter_process(int16_t *input, int16_t *output)
{
    for (int i = 0u; i < TOUCH_USER_CHANNEL_NUM; i++)
    {
        output[i] = (3 * touch_iir_filter_buf_internal[i] + input[i]) / 4;
    }
}

/* EOF. */

