/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_sdmmc_host.h"
#include "fsl_sdmmc_event.h"
#include "fsl_sdmmc_host1.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*!
 * @brief SDMMCHOST detect card insert status by host controller.
 * @param base host base address.
 * @param data3 flag indicate use data3 to detect card or not.
 */
static void SDMMC1HOST_DetectCardInsertByHost(SDMMCHOST_TYPE *base, bool data3);

/*!
 * @brief SDMMCHOST detect card status by GPIO.
 */
static bool SDMMC1HOST_DetectCardByGpio(void);

/*!
 * @brief SDMMCHOST transfer function.
 * @param base host base address.
 * @param content transfer configurations.
 */
static status_t SDMMC1HOST_TransferFunction(SDMMCHOST_TYPE *base, SDMMCHOST_TRANSFER *content);

/*!
 * @brief card detect deinit function.
 */
//static void SDMMCHOST_CardDetectDeinit(void);

/*!
 * @brief card detect deinit function.
 * @param host base address.
 * @param host detect card configuration.
 */
//static status_t SDMMCHOST_CardDetectInit(SDMMCHOST_TYPE *base, const sdmmchost_detect_card_t *cd);

/*******************************************************************************
 * Variables
 ******************************************************************************/
/*! @brief Card detect flag. */
static volatile bool s_sdInsertedFlag1 = false;
/*! @brief DMA descriptor table. */
static uint32_t s_sdifDmaTable1[SDIF_DMA_TABLE_WORDS];
/*******************************************************************************
 * Code
 ******************************************************************************/
static bool SDMMCHOST_DetectCard1ByGpio(void)
{
    if (SDMMCHOST_CARD_DETECT_STATUS() != SDMMCHOST_CARD_INSERT_CD_LEVEL)
    {
        s_sdInsertedFlag1 = false;
    }
    else
    {
        s_sdInsertedFlag1 = true;
    }

    return s_sdInsertedFlag1;
}

static void SDMMCHOST_DetectCard1InsertByHost(SDMMCHOST_TYPE *base, bool data3)
{
    if (SDMMCHOST_CARD1_DETECT_INSERT_STATUS(base, data3))
    {
        s_sdInsertedFlag1 = true;
    }
}

/* User defined transfer function. */
static status_t SDMMC1HOST_TransferFunction(SDMMCHOST_TYPE *base, SDMMCHOST_TRANSFER *content)
{
    status_t error = kStatus_Success;

    sdif_dma_config_t dmaConfig;

    memset(s_sdifDmaTable1, 0, sizeof(s_sdifDmaTable1));
    memset(&dmaConfig, 0, sizeof(dmaConfig));

    if (content->data != NULL)
    {
        dmaConfig.enableFixBurstLen     = true;
        dmaConfig.mode                  = kSDIF_ChainDMAMode;
        dmaConfig.dmaDesBufferStartAddr = s_sdifDmaTable1;
        dmaConfig.dmaDesBufferLen       = SDIF_DMA_TABLE_WORDS;
    }

		content->command->flags |= SDIF_CMD_CARD_NUMBER(1);//specify card 1
		
    if (kStatus_Success != SDIF_TransferBlocking(base, &dmaConfig, content))
    {
        error = kStatus_Fail;
    }

    return error;
}

//static status_t SDMMCHOST_CardDetectInit(SDMMCHOST_TYPE *base, const sdmmchost_detect_card_t *cd)
//{
//    sdmmchost_detect_card_type_t cdType = kSDMMCHOST_DetectCardByGpioCD;

//    if (cd != NULL)
//    {
//        cdType = cd->cdType;
//    }

//    if (cdType == kSDMMCHOST_DetectCardByGpioCD)
//    {
//        SDMMCHOST_CARD_DETECT_INIT();
//        SDMMCHOST_DetectCardByGpio();
//    }
//    else
//    {
//        /* enable card detect status */
//        SDMMCHOST_CARD_DETECT_INSERT_ENABLE(base);
//    }

//    return kStatus_Success;
//}

static void SDMMCHOST_Card1DetectDeinit(void)
{
    s_sdInsertedFlag1 = false;
}

status_t SDMMCHOST_WaitCard1DetectStatus(SDMMCHOST_TYPE *base, const sdmmchost_detect_card_t *cd, bool waitCardStatus)
{
    sdmmchost_detect_card_type_t cdType = kSDMMCHOST_DetectCardByGpioCD;

    if (cd != NULL)
    {
        cdType = cd->cdType;
    }

    if (waitCardStatus != s_sdInsertedFlag1)
    {
        /* Wait card inserted. */
        do
        {
            if (cdType == kSDMMCHOST_DetectCardByGpioCD)
            {
                SDMMCHOST_DetectCard1ByGpio();
            }
            else
            {
                SDMMCHOST_DetectCard1InsertByHost(base, cdType == kSDMMCHOST_DetectCardByHostDATA3);
            }

        } while (waitCardStatus != s_sdInsertedFlag1);
    }

    return kStatus_Success;
}

bool SDMMCHOST_IsCard1Present(void)
{
    return s_sdInsertedFlag1;
}

void SDMMCHOST_PowerOffCard1(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    if (pwr != NULL)
    {
        pwr->powerOff();
        SDMMCHOST_Delay(pwr->powerOffDelay_ms);
    }
    else
    {
        /* disable the card power */
        SDIF_EnableCard1Power(base, false);
        /* Delay several milliseconds to make card stable. */
        SDMMCHOST_Delay(500U);
    }
}
void SDMMCHOST_PowerOnCard0(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    /* use user define the power on function  */
    if (pwr != NULL)
    {
        pwr->powerOn();
        SDMMCHOST_Delay(pwr->powerOnDelay_ms);
    }
    else
    {
        /* Enable the card power */
        SDIF_EnableCardPower(base, true);
        /* Delay several milliseconds to make card stable. */
        SDMMCHOST_Delay(500U);
    }
}
void SDMMCHOST_PowerOnCard1(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    /* use user define the power on function  */
    if (pwr != NULL)
    {
        pwr->powerOn();
        SDMMCHOST_Delay(pwr->powerOnDelay_ms);
    }
    else
    {
        /* Enable the card power */
        SDIF_EnableCard1Power(base, true);
        /* Delay several milliseconds to make card stable. */
        SDMMCHOST_Delay(500U);
    }
}

status_t SDMMC1HOST_Init(SDMMCHOST_CONFIG *host, void *userData)
{
    sdif_host_t *sdifHost = (sdif_host_t *)host;

    /* init event timer. */
//    SDMMCEVENT_InitTimer();

//    /* Initialize SDIF. */
    sdifHost->config.endianMode            = kSDMMCHOST_EndianModeLittle;
    sdifHost->config.responseTimeout       = 0xFFU;
    sdifHost->config.cardDetDebounce_Clock = 0xFFFFFFU;
    sdifHost->config.dataTimeout           = 0xFFFFFFU;
//    SDIF_Init(sdifHost->base, &(sdifHost->config));

    /* Define transfer function. */
    sdifHost->transfer = SDMMC1HOST_TransferFunction;

//    SDMMCHOST_CardDetectInit(sdifHost->base, (userData == NULL) ? NULL : (((sdmmhostcard_usr_param_t *)userData)->cd));

    return kStatus_Success;
}

void SDMMC1HOST_Reset(SDMMCHOST_TYPE *base)
{
    /* reserved for future */
}

void SDMMC1HOST_Deinit(void *host)
{
    sdif_host_t *sdifHost = (sdif_host_t *)host;
//    SDIF_Deinit(sdifHost->base);
    SDMMCHOST_Card1DetectDeinit();
}

void SDMMC1HOST_ErrorRecovery(SDMMCHOST_TYPE *base)
{
    /* reserved for future */
}
