/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "board.h"
#include "fsl_flexio_uart.h"

#include "pin_mux.h"
#include "clock_config.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BOARD_FLEXIO_BASE FLEXIO1
#define FLEXIO_UART_TX_PIN 21U
#define FLEXIO_UART_RX_PIN 22U
#define FLEXIO_IRDA_TX_PIN 3U
#define FLEXIO_IRDA_RX_PIN 26U

/* Select USB1 PLL (480 MHz) as flexio clock source */
#define FLEXIO_CLOCK_SELECT (3U)
/* Clock pre divider for flexio clock source */
#define FLEXIO_CLOCK_PRE_DIVIDER (4U)
/* Clock divider for flexio clock source */
#define FLEXIO_CLOCK_DIVIDER (7U)
#define FLEXIO_CLOCK_FREQUENCY \
    (CLOCK_GetFreq(kCLOCK_Usb1PllClk) / (FLEXIO_CLOCK_PRE_DIVIDER + 1U) / (FLEXIO_CLOCK_DIVIDER + 1U))

#define DEMO_FLEXIO_CLOCK_FREQUENCY FLEXIO_CLOCK_FREQUENCY

#define DEMO_FLEXIO_FREQUENCY (48000U)
#define FLEXIO_MAX_FREQUENCY (DEMO_FLEXIO_CLOCK_FREQUENCY / 2U)
#define FLEXIO_MIN_FREQUENCY (DEMO_FLEXIO_CLOCK_FREQUENCY / 256U)
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
FLEXIO_UART_Type uartDev;
uint8_t txbuff[]   = "Flexio uart polling example\r\nBoard will send back received characters\r\n";
uint8_t rxbuff[20] = {0};
/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */

static void flexio_irda_init(uint32_t freq_Hz)
{
  
  /******************************* IRDA TX ***********************************/
#if 1
    assert((freq_Hz < FLEXIO_MAX_FREQUENCY) && (freq_Hz > FLEXIO_MIN_FREQUENCY));

    uint32_t lowerValue = 0; /* Number of clock cycles in high logic state in one period */
    uint32_t upperValue = 0; /* Number of clock cycles in low logic state in one period */
    uint32_t sum        = 0; /* Number of clock cycles in one period */
    flexio_timer_config_t fxioTimerConfig_TX;
    uint32_t duty = 0;

    /* Configure the timer DEMO_FLEXIO_TIMER_CH for generating PWM */
    fxioTimerConfig_TX.triggerSelect   = FLEXIO_TIMER_TRIGGER_SEL_PININPUT(FLEXIO_UART_TX_PIN);//FLEXIO_TIMER_TRIGGER_SEL_SHIFTnSTAT(0U);
    fxioTimerConfig_TX.triggerSource   = kFLEXIO_TimerTriggerSourceInternal;
    fxioTimerConfig_TX.triggerPolarity = kFLEXIO_TimerTriggerPolarityActiveLow;
    fxioTimerConfig_TX.pinConfig       = kFLEXIO_PinConfigOutput;
    fxioTimerConfig_TX.pinPolarity     = kFLEXIO_PinActiveHigh;
    fxioTimerConfig_TX.pinSelect       = FLEXIO_IRDA_TX_PIN; /* Set pwm output */
    fxioTimerConfig_TX.timerMode       = kFLEXIO_TimerModeDual8BitPWM;
    fxioTimerConfig_TX.timerOutput     = kFLEXIO_TimerOutputOneNotAffectedByReset;
    fxioTimerConfig_TX.timerDecrement  = kFLEXIO_TimerDecSrcOnFlexIOClockShiftTimerOutput;
    fxioTimerConfig_TX.timerDisable    = kFLEXIO_TimerDisableOnTimerCompare;
    fxioTimerConfig_TX.timerEnable     = kFLEXIO_TimerEnableOnTriggerHigh;
    fxioTimerConfig_TX.timerReset      = kFLEXIO_TimerResetNever;
    fxioTimerConfig_TX.timerStart      = kFLEXIO_TimerStartBitDisabled;
    fxioTimerConfig_TX.timerStop       = kFLEXIO_TimerStopBitEnableOnTimerDisable;

    uint32_t divider, timPwm;
    uint16_t pwmHighTime, pwmLowTime;
    divider = ((DEMO_FLEXIO_CLOCK_FREQUENCY /BOARD_DEBUG_UART_BAUDRATE )-2)/2;
    if(divider>255)
    {
        divider = 255;
    }
    pwmHighTime = (divider*3/16*2) - 1;
    pwmLowTime = divider - pwmHighTime;
    timPwm  = ((pwmLowTime)<<8)|((pwmHighTime));
    fxioTimerConfig_TX.timerCompare = timPwm;//((pwmHighTime - 1) << 8U) | (pwmLowTime - 1);

    FLEXIO_SetTimerConfig(BOARD_FLEXIO_BASE, 2, &fxioTimerConfig_TX);
#endif
    
#if 1
    /************************** IRDA RX *******************************/
    flexio_timer_config_t fxioTimerConfig_RX;
    fxioTimerConfig_RX.triggerSelect   = FLEXIO_TIMER_TRIGGER_SEL_PININPUT(FLEXIO_IRDA_RX_PIN);
    fxioTimerConfig_RX.triggerSource   = kFLEXIO_TimerTriggerSourceInternal;
    fxioTimerConfig_RX.triggerPolarity = kFLEXIO_TimerTriggerPolarityActiveHigh;
    fxioTimerConfig_RX.pinConfig       = kFLEXIO_PinConfigOutput;
    fxioTimerConfig_RX.pinPolarity     = kFLEXIO_PinActiveLow;
    fxioTimerConfig_RX.pinSelect       = FLEXIO_UART_RX_PIN; 
    fxioTimerConfig_RX.timerMode       = kFLEXIO_TimerModeDual8BitBaudBit;
    fxioTimerConfig_RX.timerOutput     = kFLEXIO_TimerOutputOneNotAffectedByReset;
    fxioTimerConfig_RX.timerDecrement  = kFLEXIO_TimerDecSrcOnFlexIOClockShiftTimerOutput;
    fxioTimerConfig_RX.timerDisable    = kFLEXIO_TimerDisableOnTimerCompare;
    fxioTimerConfig_RX.timerEnable     = kFLEXIO_TimerEnableOnTriggerHigh;
    fxioTimerConfig_RX.timerReset      = kFLEXIO_TimerResetOnTimerTriggerRisingEdge;
    fxioTimerConfig_RX.timerStart      = kFLEXIO_TimerStartBitDisabled;
    fxioTimerConfig_RX.timerStop       = kFLEXIO_TimerStopBitDisabled;
    
//    divider = (DEMO_FLEXIO_CLOCK_FREQUENCY /BOARD_DEBUG_UART_BAUDRATE) + 2;
//    fxioTimerConfig_RX.timerCompare = divider;
    
    uint16_t timerDiv = FLEXIO_CLOCK_FREQUENCY / BOARD_DEBUG_UART_BAUDRATE;
    timerDiv = timerDiv - 1;
    //uint16_t timerCmp = ((uint32_t)(8 * 2 - 1)) << 8U;
    //timerCmp |= timerDiv;

    fxioTimerConfig_RX.timerCompare = timerDiv;
    FLEXIO_SetTimerConfig(BOARD_FLEXIO_BASE, 3, &fxioTimerConfig_RX);
#endif
}
int main(void)
{
    uint8_t ch;
    flexio_uart_config_t config;
    status_t result = kStatus_Success;

    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();

    /* Clock setting for Flexio */
    CLOCK_SetMux(kCLOCK_Flexio1Mux, FLEXIO_CLOCK_SELECT);
    CLOCK_SetDiv(kCLOCK_Flexio1PreDiv, FLEXIO_CLOCK_PRE_DIVIDER);
    CLOCK_SetDiv(kCLOCK_Flexio1Div, FLEXIO_CLOCK_DIVIDER);

    /*
     * config.enableUart = true;
     * config.enableInDoze = false;
     * config.enableInDebug = true;
     * config.enableFastAccess = false;
     * config.baudRate_Bps = 115200U;
     * config.bitCountPerChar = kFLEXIO_UART_8BitsPerChar;
     */
    FLEXIO_UART_GetDefaultConfig(&config);
    config.baudRate_Bps = BOARD_DEBUG_UART_BAUDRATE;
    config.enableUart   = true;

    uartDev.flexioBase      = BOARD_FLEXIO_BASE;
    uartDev.TxPinIndex      = FLEXIO_UART_TX_PIN;
    uartDev.RxPinIndex      = FLEXIO_UART_RX_PIN;
    uartDev.shifterIndex[0] = 0U;
    uartDev.shifterIndex[1] = 1U;
    uartDev.timerIndex[0]   = 0U;
    uartDev.timerIndex[1]   = 1U;

    result = FLEXIO_UART_Init(&uartDev, &config, FLEXIO_CLOCK_FREQUENCY);
    if (result != kStatus_Success)
    {
        return -1;
    }
    flexio_irda_init(BOARD_DEBUG_UART_BAUDRATE);

    FLEXIO_UART_WriteBlocking(&uartDev, txbuff, sizeof(txbuff) - 1);
    char IRDA = 'U';
    FLEXIO_UART_WriteBlocking(&uartDev, &IRDA, 1);
    while (1)
    {
        FLEXIO_UART_ReadBlocking(&uartDev, &ch, 1);
        FLEXIO_UART_WriteBlocking(&uartDev, &IRDA, 1);
        
    }
}
