/*
 * Copyright 2014-2015 Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 */

#include "fsl_device_registers.h"
#include "bootloader_common.h"
#include "bootloader/bootloader.h"
#include "memory/memory.h"
#include "flexspi_nor_memory.h"
#include "bootloader/bl_context.h"
#include "microseconds/microseconds.h"
#include "utilities/fsl_rtos_abstraction.h"
#include "utilities/fsl_assert.h"
#include <string.h>

#if BL_FEATURE_FLEXSPI_NOR_MODULE

#ifndef FLEXSPI_NOR_INSTANCE
#define FLEXSPI_NOR_INSTANCE BL_FEATURE_FLEXSPI_NOR_MODULE_PERIPHERAL_INSTANCE
#endif

#ifndef FLEXSPI_NOR_ERASE_VERIFY
#define FLEXSPI_NOR_ERASE_VERIFY BL_FEATURE_FLEXSPI_NOR_MODULE_ERASE_VERIFY
#endif

#define SECTOR_SIZE			0x1000
#define PAGE_SIZE                       256
extern AT_QUICKACCESS_SECTION_CODE(status_t flexspi_nor_flash_erase_sector(FLEXSPI_Type *base, uint32_t address));
extern AT_QUICKACCESS_SECTION_CODE(status_t flexspi_nor_flash_page_program(FLEXSPI_Type *base, uint32_t dstAddr, uint32_t *src, uint32_t size));
extern AT_QUICKACCESS_SECTION_CODE(void AHBPrefetchDisable(FLEXSPI_Type *base));
extern AT_QUICKACCESS_SECTION_CODE(void AHBPrefetchEnable(FLEXSPI_Type *base));
extern AT_QUICKACCESS_SECTION_CODE(status_t flexspi_nor_flash_read(FLEXSPI_Type *base, uint32_t dstAddr, uint32_t *src, uint32_t size));
#define FLASH_AMBA_BASE BL_FLEXSPI_AMBA_BASE

////////////////////////////////////////////////////////////////////////////////
// Definitions
////////////////////////////////////////////////////////////////////////////////
enum
{
    kFlashDefaultPattern = 0xFF,
};

//! @brief FLEXSPI NOR memory feature information
//!
//! An instance is maintained in this file, will is used to keep key information for write and flush
//! operations.
typedef struct _flexspi_nor_mem_context
{
    bool isConfigured; //!< The state which indicates whether FlexSPI block is successfully
//! configured.
                              // BL_FEATURE_FLEXSPI_ENCRYPT_PROGRAMMING
    bool isAddingToBuffer;            //!< State used of determine whether it is the first write at
                                      //!< Start address of one page
    uint32_t writeAddress;            //!< This address is used to record the address which is used
                                      //!< to write the whole page into FlexSPI memory
    uint32_t offset;                  //!< A variable which is used to indicate if the buffer is
                                      //!< full.
    uint32_t nextStartAddress;        //!< A variable is used to indicate if recent two writes
                                      //!< are continuous
    uint8_t buffer[kFlexSpiNorMemory_MaxPageSize]; //!< A buffer which is used to buffer a full
                                                   //!< page of data

} flexspi_nor_mem_context_t;

////////////////////////////////////////////////////////////////////////////////
// Prototypes
////////////////////////////////////////////////////////////////////////////////


//! @brief Convert flexspi amba address to physical address
static uint32_t flexspi_get_phy_address(uint32_t mapAddr);
//! @brief Page read function for flexspi nor module driver
static status_t flexspi_nor_memory_page_program(uint32_t address, uint32_t *src);
//! @brief Erase all fcuntion for flexspi nor module driver
//static status_t flexspi_nor_memory_erase_all(void);
//! @brief Erase function for flexspi nor module driver
static status_t flexspi_nor_memory_erase(uint32_t address, uint32_t length);
//! @brief Read Serial NOR memory
//static status_t flexspi_nor_memory_read(uint32_t *dst, uint32_t start, uint32_t bytes);
//! @brief verify if serial nor memory is erased.
static bool is_flexspi_nor_mem_erased(uint32_t start, uint32_t length);
// Check wether the data in Serial NOR matches the data to check
static bool flexspi_nor_memory_check(uint32_t start, uint8_t *data_to_check, uint32_t bytes);

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////


//! @brief Context of Flexspi operation.
static flexspi_nor_mem_context_t s_flexspiNorContext = {
    .isConfigured = false, .isAddingToBuffer = false,

};
//! @brief Interface to flexspi memory operations
const memory_region_interface_t g_flexspiMemoryInterface = {
    .init = flexspi_nor_mem_init,
    .read = flexspi_nor_mem_read,
    .write = flexspi_nor_mem_write,
#if !BL_MIN_PROFILE
    .fill = flexspi_nor_mem_fill,
#endif // #if !BL_MIN_PROFILE
    .erase = flexspi_nor_mem_erase,
    .flush = flexspi_nor_mem_flush,
};


////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////
static uint32_t flexspi_get_phy_address(uint32_t mapAddr)
{
    return mapAddr - 0x60000000;
}
// See flexspi_nor_memory.h for documentation on this function.
status_t flexspi_nor_mem_write(uint32_t address, uint32_t length, const uint8_t *buffer)
{
    // Note: the check for "length != 0" and "range not in reserved region" is done in mem_write().
    assert(length);
    assert(buffer);
    status_t status;

    uint32_t writeLength;

    while (length)
    {
        // If buffer is invalid, it means it is a new write operation.
        if (!s_flexspiNorContext.isAddingToBuffer)
        {
            // If address is page aligned, it means it is a valid start address, else return an error status
            if (address & (PAGE_SIZE - 1))
            {
                return kStatus_FlexSPINOR_WriteAlignmentError;
            }

            // Start buffering datashuo
            s_flexspiNorContext.isAddingToBuffer = true;
            s_flexspiNorContext.offset = 0;
            s_flexspiNorContext.writeAddress = address;
        }
        else
        {
            // In this case, it means recent two writes are not continuous, should flush last cached data into memory,
            // then switch to processsing of this write operation.
            if ((s_flexspiNorContext.offset + s_flexspiNorContext.writeAddress) != address)
            {
                // flush cached data into target memory,
                status = flexspi_nor_mem_flush();
                if (status != kStatus_Success)
                {
                    return status;
                }
                // Start processing this write
                continue;
            }
            // Otherwise, it means recent two writes are continuous, continue to buffer data until whole page gets
            // buffered.
        }

        if (s_flexspiNorContext.offset + length < PAGE_SIZE)
        {
            writeLength = length;
        }
        else
        {
            writeLength = PAGE_SIZE - s_flexspiNorContext.offset;
        }

        // Copy data to internal buffer
        memcpy(&s_flexspiNorContext.buffer[s_flexspiNorContext.offset], &buffer[0], writeLength);
        s_flexspiNorContext.offset += writeLength;
        address += writeLength;
        buffer += writeLength;
        length -= writeLength;

        assert(s_flexspiNorContext.offset <= PAGE_SIZE);
        // If the buffer is full, it is time to flush cached data to target memory.
        if (s_flexspiNorContext.offset == PAGE_SIZE)
        {
            status = flexspi_nor_mem_flush();
            if (status != kStatus_Success)
            {
                return status;
            }
        }
    }

    return kStatus_Success;
}

bool is_flexspi_nor_mem_erased(uint32_t start, uint32_t length)
{
    uint32_t *buf_32;
    buf_32 = (uint32_t *)start;
    for (uint32_t i = 0; i < length / 4; i++)
    {
        if (*buf_32++ != 0xFFFFFFFFUL)
        {
            return false;
        }
    }
    return true;
}

bool flexspi_nor_memory_check(uint32_t start, uint8_t *data_to_check, uint32_t length)
{
    uint32_t buffer[PAGE_SIZE / sizeof(uint32_t)];

    uint32_t read_length;
    while (length)
    {
        read_length = length < sizeof(s_flexspiNorContext.buffer) ? length : sizeof(s_flexspiNorContext.buffer);
        uint32_t phy_address = flexspi_get_phy_address(start);
        flexspi_nor_flash_read(FLEXSPI, phy_address,(uint32_t *)&buffer,read_length);

        if (memcmp(buffer, data_to_check, read_length) != 0)
        {
            return false;
        }

        length -= read_length;
        start += read_length;
        data_to_check += read_length;
    }

    return true;
}
static uint32_t flash_errorcount = 0;
// See flexspi_nor_memory.h for documentation on this function.
status_t flexspi_nor_mem_flush(void)
{
    status_t status = kStatus_Success;

    if (s_flexspiNorContext.isAddingToBuffer)
    {
        s_flexspiNorContext.isAddingToBuffer = false;
        // Fill unused region with 0xFFs.
        if (s_flexspiNorContext.offset != PAGE_SIZE)
        {
            memset(&s_flexspiNorContext.buffer[s_flexspiNorContext.offset], 0xFF,
                   PAGE_SIZE - s_flexspiNorContext.offset);
        }

#if BL_FEATURE_FLASH_CHECK_CUMULATIVE_WRITE
        if (!is_flexspi_nor_mem_erased(s_flexspiNorContext.writeAddress, PAGE_SIZE))
        {
            return kStatusMemoryCumulativeWrite;
        }
#endif // BL_FEATURE_FLASH_CHECK_CUMULATIVE_WRITE

       // Write whole page to spi flash
       status =
            flexspi_nor_memory_page_program(s_flexspiNorContext.writeAddress, (uint32_t *)s_flexspiNorContext.buffer);
        // Clear related states.
        s_flexspiNorContext.isAddingToBuffer = false;
        s_flexspiNorContext.offset = 0;
        if (status != kStatus_Success)
        {
            return status;
        }
        if (SCB->CCR & SCB_CCR_DC_Msk)
        {
            SCB_InvalidateDCache_by_Addr((uint32_t *)s_flexspiNorContext.writeAddress,
                                         PAGE_SIZE);
        }
        // Verify whether the data has been programmed to Serial NOR flash successfully.
        if (!flexspi_nor_memory_check(s_flexspiNorContext.writeAddress, s_flexspiNorContext.buffer,
                                      PAGE_SIZE))
        {
            flash_errorcount++;
            return kStatus_FlexSPINOR_CommandFailure;
        }
    }
    return status;
}
status_t flexspi_nor_mem_read(uint32_t address, uint32_t length, uint8_t *buffer)
{
    uint32_t phy_address = flexspi_get_phy_address(address);
    return flexspi_nor_flash_read(BL_FLEXSPI_BASE_ADDRESS,phy_address,(uint32_t *)buffer,length);
}

status_t flexspi_nor_memory_erase(uint32_t address, uint32_t length)
{
    uint32_t eraseAddress;
    status_t status;
    lock_acquire();
    eraseAddress = flexspi_get_phy_address(address);
    uint32_t eraseSector = (length+SECTOR_SIZE-1)/SECTOR_SIZE;
    
    for(uint32_t i=0;i<eraseSector;i++)
    {
        status = flexspi_nor_flash_erase_sector(BL_FLEXSPI_BASE_ADDRESS,eraseAddress+i*SECTOR_SIZE);
    }
    lock_release();
    return status;
}
// See flexspi_nor_memory.h for documentation on this function.
status_t flexspi_nor_mem_erase(uint32_t address, uint32_t length)
{
    assert(length);

       uint32_t sectorSize = SECTOR_SIZE;
    uint32_t alignedAddress = ALIGN_DOWN(address, sectorSize);
    uint32_t alignedLength = ALIGN_UP(address + length, sectorSize) - alignedAddress;

    status_t status = flexspi_nor_memory_erase(alignedAddress, alignedLength);
    SCB_InvalidateDCache_by_Addr((uint32_t *)alignedAddress, alignedLength);
    if (status != kStatus_Success)
    {
        return status;
    }
   
#if FLEXSPI_NOR_ERASE_VERIFY
    if (!is_flexspi_nor_mem_erased(alignedAddress, alignedLength))
    {
        return kStatus_FlexSPINOR_CommandFailure;
    }
#endif // #if FLEXSPI_NOR_ERASE_VERIFY

    return kStatus_Success;
}

static status_t flexspi_nor_memory_page_program(uint32_t address, uint32_t *src)
{
    uint32_t phy_address = flexspi_get_phy_address(address);
  
    lock_acquire();
    status_t status = flexspi_nor_flash_page_program(BL_FLEXSPI_BASE_ADDRESS,phy_address, src,PAGE_SIZE);
    lock_release();
    return status;
}





#endif // #if BL_FEATURE_FLEXSPI_NOR_MODULE
////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
