/*
 * Copyright (c) 2013-2015 Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "bootloader_common.h"
#include "bootloader/bootloader.h"
#include "memory/memory.h"
#include "sbloader/sbloader.h"
#include "property/property.h"
#include "utilities/fsl_assert.h"
#include "utilities/fsl_rtos_abstraction.h"
#include <string.h>
#include <stdint.h>
#include "flexspi_nor_memory.h"



#if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#if !BL_DEVICE_IS_LPC_SERIES
#include "fsl_flash.h"
#else
#include "flashiap_wrapper/fsl_flashiap_wrapper.h"
#endif
#endif // #if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#include "fsl_device_registers.h"
#if BL_FEATURE_QSPI_MODULE
#include "qspi.h"
#endif // #if BL_FEATURE_QSPI_MODULE
#if BL_FEATURE_OTFAD_MODULE
#include "fsl_otfad_driver.h"
#endif // #if BL_FEATURE_OTFAD_MODULE
#if BL_FEATURE_OCOTP_MODULE
//#include "ocotp/fsl_ocotp.h"
#endif // #if BL_FEATURE_OCOTP_MODULE
#if defined(BL_DEVICE_IS_LPC_SERIES) && defined(OTP_API)
#include "otp/otp.h"
#endif
#if BL_FEATURE_RELIABLE_UPDATE
#include "bootloader/bl_reliable_update.h"
#endif

#if BL_FEATURE_GEN_KEYBLOB
#include "bootloader/bl_keyblob.h"
#endif

#if BL_FEATURE_KEY_PROVISIONING
#include "authentication/key_store.h"
#include "authentication/key_store_hal.h"
/* the RAM address of key store comes from hal (key_store_hal_lpc54s018.c) */
extern skboot_key_store_t *const s_keyStore;


#endif

//! @addtogroup command
//! @{

//! @name State machine
//@{
static status_t handle_command(uint8_t *packet, uint32_t packetLength);
static status_t handle_data(bool *hasMoreData);
//@}

//! @name Command handlers
//@{
void handle_reset(uint8_t *packet, uint32_t packetLength);
void handle_flash_erase_all(uint8_t *packet, uint32_t packetLength);
void handle_flash_erase_all_unsecure(uint8_t *packet, uint32_t packetLength);
void handle_flash_erase_region(uint8_t *packet, uint32_t packetLength);
void handle_receive_sb_file(uint8_t *packet, uint32_t packetLength);
void handle_read_memory(uint8_t *packet, uint32_t packetLength);
void handle_fill_memory(uint8_t *packet, uint32_t packetLength);
void handle_set_property(uint8_t *packet, uint32_t packetLength);
void handle_get_property(uint8_t *packet, uint32_t packetLength);
void handle_write_memory(uint8_t *packet, uint32_t packetLength);
void handle_execute(uint8_t *packet, uint32_t packetLength);
void handle_call(uint8_t *packet, uint32_t packetLength);
void handle_flash_security_disable(uint8_t *packet, uint32_t packetLength);
void handle_flash_program_once(uint8_t *packet, uint32_t length);
void handle_flash_read_once(uint8_t *packet, uint32_t length);
void handle_flash_read_resource(uint8_t *packet, uint32_t length);
void handle_configure_memory(uint8_t *packet, uint32_t packetLength);
void handle_reliable_update(uint8_t *packet, uint32_t packetLength);
void handle_key_provisioning(uint8_t *packet, uint32_t packetLength);
#if BL_FEATURE_GEN_KEYBLOB
void handle_generate_key_blob(uint8_t *packet, uint32_t packetLength);
status_t handle_key_blob_data(bool *hasMoreData);
#endif
//@}

//! @name Command responses
//@{
void send_read_memory_response(uint32_t commandStatus, uint32_t length);
#if BL_FEATURE_GEN_KEYBLOB
void send_generate_key_blob_response(uint32_t commandStatus, uint32_t length);
#endif
void send_generic_response(uint32_t commandStatus, uint32_t commandTag);
void send_get_property_response(uint32_t commandStatus, uint32_t *value, uint32_t numValues);
void send_flash_read_once_response(uint32_t commandStatus, uint32_t *value, uint32_t byteCount);
void send_flash_read_resource_response(uint32_t commandStatus, uint32_t length);
void send_key_provisioning_response(uint32_t commandStatus, uint32_t length);
//@}

//! @name Data phase
//@{
static void reset_data_phase(void);
void finalize_data_phase(status_t status);
status_t handle_data_producer(bool *hasMoreData);
status_t handle_data_consumer(bool *hasMoreData);
status_t handle_data_bidirection(bool *hasMoreData);
//@}

////////////////////////////////////////////////////////////////////////////////
// Definitions
////////////////////////////////////////////////////////////////////////////////

//#define TEST_SENDER_ABORT
//#define TEST_RECEIVER_ABORT

enum _secure_commands
{
    //! @brief Bitmask of commands allowed when flash security is enabled.
    //!
    //! This bitmask uses the same format as the AvailableCommands property. This is,
    //! the bit number for a given command is the command's tag value minus one.
    kCommandsAllowedWhenSecure = (HAS_CMD(kCommandTag_FlashSecurityDisable) | HAS_CMD(kCommandTag_GetProperty) |
                                  HAS_CMD(kCommandTag_Reset) | HAS_CMD(kCommandTag_SetProperty) |
                                  HAS_CMD(kCommandTag_FlashEraseAllUnsecure) | HAS_CMD(kCommandTag_ReceiveSbFile)
#if BL_FEATURE_KEY_PROVISIONING
                                  | HAS_CMD(kCommandTag_KeyProvisioning) 
#endif // BL_FEATURE_KEY_PROVISIONING
                                  )
};

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////

//! @brief Command handler table.
const command_handler_entry_t g_commandHandlerTable[] = {
// cmd handler              // data handler or NULL
    { 0, NULL },              // kCommandTag_FlashEraseAll = 0x01
    { handle_flash_erase_region, NULL },           // kCommandTag_FlashEraseRegion = 0x02
    { handle_read_memory, handle_data_producer },  // kCommandTag_ReadMemory = 0x03
    { handle_write_memory, handle_data_consumer }, // kCommandTag_WriteMemory = 0x04
    {0},//{ handle_fill_memory, NULL },                  // kCommandTag_FillMemory = 0x05
    {0},
                                              // BL_FEATURE_FLASH_SECURITY
    { handle_get_property, NULL },                    // kCommandTag_GetProperty = 0x07
    {0},//{ handle_receive_sb_file, handle_data_consumer }, // kCommandTag_ReceiveSbFile = 0x08
    {0},//{ handle_execute, NULL },                         // kCommandTag_Execute = 0x09
    {0},//{ handle_call, NULL },                            // kCommandTag_Call = 0x0a
    {0},//{ handle_reset, NULL },                           // kCommandTag_Reset = 0x0b
    {0},//{ handle_set_property, NULL },                    // kCommandTag_SetProperty = 0x0c
    {0}, // kCommandTag_FlashEraseAllUnsecure = 0x0d


    { 0 },
    { 0 },
    { 0 },
       // ((BL_DEVICE_IS_LPC_SERIES) && defined(OTP_API))

    { 0 }, // kCommandTag_ConfigureMemory = 0x11

       // BL_FEATURE_EXPAND_MEMORY || BL_FEATURE_SPI_NOR_EEPROM_MODULE || BL_FEATURE_SPIFI_NOR_MODULE
    { 0 }, // kCommandTag_ReliableUpdate = 0x12
    { 0 },
    { 0 },
};

//! @brief Command processor state data.
command_processor_data_t g_commandData;

// See bl_command.h for documentation on this interface.
command_interface_t g_commandInterface = { bootloader_command_init, bootloader_command_pump,
                                           (command_handler_entry_t *)&g_commandHandlerTable, &g_commandData };

#if BL_FEATURE_EXPAND_PACKET_SIZE
static uint8_t s_dataProducerPacket[kMaxBootloaderPacketSize];
#endif // BL_FEATURE_EXPAND_PACKET_SIZE

#if BL_FEATURE_KEY_PROVISIONING
uint8_t s_userKeyBuffer[kPUF_KeySizeMax];
#endif
////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////

// See bl_command.h for documentation on this function.
status_t bootloader_command_init()
{
    command_processor_data_t *data = g_bootloaderContext.commandInterface->stateData;

    data->state = kCommandState_CommandPhase;
    return kStatus_Success;
}

// See bl_command.h for documentation on this function.
status_t bootloader_command_pump()
{
    status_t status = kStatus_Success;
    bool hasMoreData = false;

    if (g_bootloaderContext.activePeripheral->packetInterface)
    {
        switch (g_bootloaderContext.commandInterface->stateData->state)
        {
            default:
            case kCommandState_CommandPhase:
                status = g_bootloaderContext.activePeripheral->packetInterface->readPacket(
                    g_bootloaderContext.activePeripheral, &g_bootloaderContext.commandInterface->stateData->packet,
                    &g_bootloaderContext.commandInterface->stateData->packetLength, kPacketType_Command);
                if ((status != kStatus_Success) && (status != kStatus_AbortDataPhase) && (status != kStatus_Ping))
                {
                    debug_printf("Error: readPacket returned status 0x%x\r\n", status);
                    break;
                }
                if (g_bootloaderContext.commandInterface->stateData->packetLength == 0)
                {
                    // No command packet is available. Return success.
                    break;
                }
                status = handle_command(g_bootloaderContext.commandInterface->stateData->packet,
                                        g_bootloaderContext.commandInterface->stateData->packetLength);
                if (status != kStatus_Success)
                {
                    debug_printf("Error: handle_command returned status 0x%x\r\n", status);
                    break;
                }
                g_bootloaderContext.commandInterface->stateData->state = kCommandState_DataPhase;
                break;

            case kCommandState_DataPhase:
                status = handle_data(&hasMoreData);
                if (status != kStatus_Success)
                {
                    debug_printf("Error: handle_data returned status 0x%x\r\n", status);
                    g_bootloaderContext.commandInterface->stateData->state = kCommandState_CommandPhase;
                    break;
                }
                g_bootloaderContext.commandInterface->stateData->state =
                    hasMoreData ? kCommandState_DataPhase : kCommandState_CommandPhase;
                break;
        }
    }

    return status;
}

//! @brief Find command handler entry.
//!
//! @retval NULL if no entry found.
static const command_handler_entry_t *find_entry(uint8_t tag)
{
    if (tag < kFirstCommandTag || tag > kLastCommandTag)
    {
        return 0; // invalid command
    }
    const command_handler_entry_t *entry =
        &g_bootloaderContext.commandInterface->handlerTable[(tag - kFirstCommandTag)];

    return entry;
}

//! @brief Handle a command transaction.
static status_t handle_command(uint8_t *packet, uint32_t packetLength)
{
    command_packet_t *commandPacket = (command_packet_t *)packet;
    uint8_t commandTag = commandPacket->commandTag;
    status_t status = kStatus_Success;

    // Look up the handler entry and save it for the data phaase.
    g_bootloaderContext.commandInterface->stateData->handlerEntry = find_entry(commandTag);

    if (g_bootloaderContext.commandInterface->stateData->handlerEntry &&
        g_bootloaderContext.commandInterface->stateData->handlerEntry->handleCommand)
    {
#if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#if !BL_DEVICE_IS_LPC_SERIES
        // Get flash security state.
        // Note: Both Main and Secondary flash share the same security state
        //  So it doesn't matter what index of allFlashState[] we use for this FLASH API.
        ftfx_security_state_t securityState;
        status = g_bootloaderContext.flashDriverInterface->flash_get_security_state(
            g_bootloaderContext.allFlashState, &securityState);
        if (status == kStatus_Success)
        {
            // If flash security is enabled, make sure the command is one that is allowed. If
            // it's not, then we return an error response.
            if ((securityState != kFTFx_SecurityStateNotSecure) &&
                !IS_CMD_AVAILABLE(kCommandsAllowedWhenSecure, commandTag))
            {
                // Security is enabled and the command is not one of the few that can be
                // run, so return a security violation error.
                debug_printf("Error: command 0x%x not available due to flash security\r\n", commandPacket->commandTag);
                status = kStatus_SecurityViolation;
            }
            else
            {
#endif // !BL_DEVICE_IS_LPC_SERIES
#endif // #if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
                // Process the command normally.
                g_bootloaderContext.commandInterface->stateData->handlerEntry->handleCommand(packet, packetLength);
                return kStatus_Success;
#if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#if !BL_DEVICE_IS_LPC_SERIES
            }
        }
#endif // !BL_DEVICE_IS_LPC_SERIES
#endif // #if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
    }
    else
    {
        // We don't recognize this command, so return an error response.
        debug_printf("unknown command 0x%x\r\n", commandPacket->commandTag);
        status = kStatus_UnknownCommand;
    }

    // Should only get to this point if an error occurred before running the command handler.
    send_generic_response(status, commandTag);
    return status;
}

//! @brief Handle a data transaction.
static status_t handle_data(bool *hasMoreData)
{
    if (g_bootloaderContext.commandInterface->stateData->handlerEntry)
    {
        // Run data phase if present, otherwise just return success.
        *hasMoreData = 0;
        return g_bootloaderContext.commandInterface->stateData->handlerEntry->handleData ?
                   g_bootloaderContext.commandInterface->stateData->handlerEntry->handleData(hasMoreData) :
                   kStatus_Success;
    }

    debug_printf("Error: no handler entry for data phase\r\n");
    return kStatus_Success;
}

////////////////////////////////////////////////////////////////////////////////
// Command Handlers
////////////////////////////////////////////////////////////////////////////////

//! @brief Reset command handler.
void handle_reset(uint8_t *packet, uint32_t packetLength)
{
    command_packet_t *commandPacket = (command_packet_t *)packet;
    send_generic_response(kStatus_Success, commandPacket->commandTag);

    // Wait for the ack from the host to the generic response
    g_bootloaderContext.activePeripheral->packetInterface->finalize(g_bootloaderContext.activePeripheral);

    // Prepare for shutdown.
    shutdown_cleanup(kShutdownType_Reset);
#if defined(BL_FEATURE_6PINS_PERIPHERAL) && BL_FEATURE_6PINS_PERIPHERAL
    shutdown_cleanup(kShutdownType_Cleanup);
#endif // BL_FEATURE_6PINS_PERIPHERAL

    NVIC_SystemReset();
    // Does not get here.
    assert(0);
}
typedef struct
{
    uint8_t bugfix; //!< bugfix version [7:0]
    uint8_t minor;  //!< minor version [15:8]
    uint8_t major;  //!< major version [23:16]
    char name;      //!< name [31:24]
}BL_VER;

static BL_VER sBl_ver;
static uint32_t s_propertyReturnValue;
// See property.h for documentation on this function.
status_t bootloader_property_get(uint8_t tag, uint32_t id, const void **value, uint32_t *valueSize)
{
    // Set default value size, may be modified below.
    uint32_t returnSize = sizeof(uint32_t);
    const void *returnValue;
    switch (tag)
    {
        case kPropertyTag_BootloaderVersion:
                sBl_ver.bugfix = 1;
                sBl_ver.minor = 0;
                sBl_ver.major = 1;
                sBl_ver.name  = 'R';
                *valueSize = sizeof(sBl_ver);
                returnValue = &sBl_ver;
            break;

        case kPropertyTag_AvailablePeripherals:
            //returnValue = &propertyStore->availablePeripherals;
            break;

        case kPropertyTag_RAMStartAddress:
            break;

        case kPropertyTag_RAMSizeInBytes:
            break;

        case kPropertyTag_AvailableCommands:
            //returnValue = &propertyStore->availableCommands;
            break;

        case kPropertyTag_MaxPacketSize:
            // Read the max packet size from the active peripheral.
            s_propertyReturnValue = g_bootloaderContext.activePeripheral->packetInterface->getMaxPacketSize(
                g_bootloaderContext.activePeripheral);
            returnValue = &s_propertyReturnValue;
            break;

        case kPropertyTag_ReservedRegions:
           // returnSize = sizeof(propertyStore->reservedRegions);
           // returnValue = propertyStore->reservedRegions;
            break;

        case kPropertyTag_FlashSecurityState:
        {
           // s_propertyReturnValue = (get_hab_status() != kHabStatus_Open);
            //returnValue = &s_propertyReturnValue;
            break;
        }

        case kPropertyTag_UniqueDeviceId:
            //returnSize = kUniqueId_SizeInBytes;
            //returnValue = &propertyStore->UniqueDeviceId;
            break;

        case kPropertyTag_TargetVersion:
            //returnValue = &propertyStore->targetVersion.version;
            break;
        case kPropertyTag_ExternalMemoryAttributes:
            break;

        default:
            return kStatus_UnknownProperty;
    }

    // Set the return size.
    if (valueSize)
    {
        *valueSize = returnSize;
    }

    // Set the return value
    if (value)
    {
        *value = returnValue;
    }

    return kStatus_Success;
}
//! @brief Get Property command handler.
void handle_get_property(uint8_t *packet, uint32_t packetLength)
{
    get_property_packet_t *command = (get_property_packet_t *)packet;
    uint32_t *value = NULL;
    uint32_t valueSize = 0;
    status_t status = kStatus_Success;
    // Make sure the property's size is no more than the size of the max number of return parameters.
    assert(valueSize <= (kMaxPropertyReturnValues * sizeof(uint32_t)));
    

    
    bootloader_property_get(command->propertyTag, command->memoryId,(const void **)&value, &valueSize);
                                                        

    // Currently there are no property responses that contain a data phase.
    g_bootloaderContext.commandInterface->stateData->dataPhase.count = 0;
    send_get_property_response(status, value, (valueSize / sizeof(uint32_t)));
}
//! @brief Reset data phase variables.
static void reset_data_phase()
{
    memset(&g_bootloaderContext.commandInterface->stateData->dataPhase, 0,
           sizeof(g_bootloaderContext.commandInterface->stateData->dataPhase));
}

//! @brief Flash Erase All command handler.
void handle_flash_erase_all(uint8_t *packet, uint32_t packetLength)
{
    flash_erase_all_packet_t *commandPacket = (flash_erase_all_packet_t *)packet;
    status_t status = kStatus_Success;


    send_generic_response(status, commandPacket->commandPacket.commandTag);
}

//! @brief Flash Erase All Unsecure command handler.
void handle_flash_erase_all_unsecure(uint8_t *packet, uint32_t packetLength)
{
    command_packet_t *commandPacket = (command_packet_t *)packet;
    status_t status = kStatus_Success;

// Call flash erase all unsecure implementation.
#if BL_FEATURE_ERASEALL_UNSECURE
    status = flash_mem_erase_all_unsecure();
#if BL_FEATURE_SUPPORT_DFLASH 
    if (g_bootloaderContext.dflashDriverInterface != NULL)
    {
        status += flexNVM_mem_erase_all_unsecure();   
    }
#endif // BL_FEATURE_SUPPORT_DFLASH     
#endif //BL_FEATURE_ERASEALL_UNSECURE

    send_generic_response(status, commandPacket->commandTag);
}

//! @brief Flash Erase Region command handler.
void handle_flash_erase_region(uint8_t *packet, uint32_t packetLength)
{
    flash_erase_region_packet_t *command = (flash_erase_region_packet_t *)packet;
    status_t status = kStatus_Success;
    debug_printf("start execute flash erase command!\r\n");
    flexspi_nor_mem_erase(command->startAddress,command->byteCount);
    send_generic_response(status, command->commandPacket.commandTag);
}

//! @brief Write Memory command handler.
void handle_write_memory(uint8_t *packet, uint32_t packetLength)
{
    write_memory_packet_t *command = (write_memory_packet_t *)packet;

    debug_printf("start execute flash write memory!\r\n");
    // Start the data phase.
    reset_data_phase();
    g_bootloaderContext.commandInterface->stateData->dataPhase.memoryId = command->memoryId;
    g_bootloaderContext.commandInterface->stateData->dataPhase.count = command->byteCount;
    g_bootloaderContext.commandInterface->stateData->dataPhase.address = command->startAddress;
    g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag = kCommandTag_WriteMemory;
    send_generic_response(kStatus_Success, command->commandPacket.commandTag);
}


//! @brief Read Memory command handler.
void handle_read_memory(uint8_t *packet, uint32_t packetLength)
{
    read_memory_packet_t *command = (read_memory_packet_t *)packet;

    // Start the data phase.
    reset_data_phase();
    g_bootloaderContext.commandInterface->stateData->dataPhase.memoryId = command->memoryId;
    g_bootloaderContext.commandInterface->stateData->dataPhase.count = command->byteCount;
    g_bootloaderContext.commandInterface->stateData->dataPhase.address = command->startAddress;
    g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag = kCommandTag_ReadMemory;
    send_read_memory_response(kStatus_Success, command->byteCount);
}

//! @brief Complete the data phase, optionally send a response.
void finalize_data_phase(status_t status)
{
    g_bootloaderContext.commandInterface->stateData->dataPhase.address = 0;
    g_bootloaderContext.commandInterface->stateData->dataPhase.count = 0;

    // Force to write cached data to target memory
    if (g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag == kCommandTag_WriteMemory)
    {
        //status_t flushStatus = g_bootloaderContext.memoryInterface->flush();
          status = flexspi_nor_mem_flush();
        // Update status only if the last operation result is successfull in order to reflect
        // real result of the write operation.
       
    }

    // Send final response packet.
    send_generic_response(status, g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag);

}


//! @brief Handle data phase, direction is defined in dataPhase.option.
status_t handle_data_bidirection(bool *hasMoreData)
{
    if (g_bootloaderContext.commandInterface->stateData->dataPhase.option == kCmd_DataPhase_Option_Consumer)
    {
        return handle_data_consumer(hasMoreData);
    }
    else if (g_bootloaderContext.commandInterface->stateData->dataPhase.option == kCmd_DataPhase_Option_Producer)
    {
        return handle_data_producer(hasMoreData);
    }
    else // Skip the data phase.
    {
        *hasMoreData = false;
        return kStatus_Success;
    }
}

//! @brief Handle data phase with data consumer (read from host).
status_t handle_data_consumer(bool *hasMoreData)
{
    if (g_bootloaderContext.commandInterface->stateData->dataPhase.count == 0)
    {
        // No data phase.
        *hasMoreData = false;
        finalize_data_phase(kStatus_Success);
        return kStatus_Success;
    }

    *hasMoreData = true;
    uint32_t remaining = g_bootloaderContext.commandInterface->stateData->dataPhase.count;
    uint32_t dataAddress = g_bootloaderContext.commandInterface->stateData->dataPhase.address;
    uint8_t *packet;
    uint32_t packetLength = 0;
    status_t status;

    // Read the data packet.
    status = g_bootloaderContext.activePeripheral->packetInterface->readPacket(
        g_bootloaderContext.activePeripheral, &packet, &packetLength, kPacketType_Data);
    if (status != kStatus_Success)
    {
        // Abort data phase due to error.
        debug_printf("consumer abort data phase due to status 0x%x\r\n", status);
        g_bootloaderContext.activePeripheral->packetInterface->abortDataPhase(g_bootloaderContext.activePeripheral);
        finalize_data_phase(status);
        *hasMoreData = false;
        return kStatus_Success;
    }
    if (packetLength == 0)
    {
        // Sender requested data phase abort.
        debug_printf("Data phase aborted by sender\r\n");
        finalize_data_phase(kStatus_AbortDataPhase);
        *hasMoreData = false;
        return kStatus_Success;
    }

    //
    // Write the data to the destination address.
    //

    packetLength = MIN(packetLength, remaining);

    if (g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag == kCommandTag_ReceiveSbFile)
    {
        // not support sb file in temporary
        status = kStatus_Success;
    }
    else
    {
        // Consumer is memory interface.
        //status = g_bootloaderContext.memoryInterface->write(dataAddress, packetLength, packet, memoryId);
        status = flexspi_nor_mem_write(dataAddress, packetLength, packet);
        dataAddress += packetLength;
    }

    remaining -= packetLength;

    if (remaining == 0)
    {
        finalize_data_phase(status);
        *hasMoreData = false;
    }
    else if (status != kStatus_Success)
    {
        // Abort data phase due to error.
        debug_printf("Data phase error 0x%x, aborting\r\n", status);
        g_bootloaderContext.activePeripheral->packetInterface->abortDataPhase(g_bootloaderContext.activePeripheral);
        finalize_data_phase(status);
        *hasMoreData = false;
    }
    else
    {
        g_bootloaderContext.commandInterface->stateData->dataPhase.count = remaining;
        g_bootloaderContext.commandInterface->stateData->dataPhase.address = dataAddress;
    }

    return kStatus_Success;
}

//! @brief Handle data phase with data producer (send to host).
status_t handle_data_producer(bool *hasMoreData)
{
    if (g_bootloaderContext.commandInterface->stateData->dataPhase.count == 0)
    {
        // No data phase.
        *hasMoreData = false;
        finalize_data_phase(kStatus_Success);
        return kStatus_Success;
    }

    *hasMoreData = true;
    uint32_t remaining = g_bootloaderContext.commandInterface->stateData->dataPhase.count;
    uint32_t dataAddress = g_bootloaderContext.commandInterface->stateData->dataPhase.address;
    uint8_t *data = g_bootloaderContext.commandInterface->stateData->dataPhase.data;
    uint8_t commandTag = g_bootloaderContext.commandInterface->stateData->dataPhase.commandTag;
    status_t status = kStatus_Success;

    // Initialize the data packet to send.
    uint32_t packetSize;
#if BL_FEATURE_EXPAND_PACKET_SIZE
    uint8_t *packet = s_dataProducerPacket;
    uint32_t packetBufferSize =
        g_bootloaderContext.activePeripheral->packetInterface->getMaxPacketSize(g_bootloaderContext.activePeripheral);
    packetSize = MIN(packetBufferSize, remaining);
#else
    uint8_t packet[kMinPacketBufferSize];
    packetSize = MIN(kMinPacketBufferSize, remaining);
#endif // BL_FEATURE_EXPAND_PACKET_SIZE

    // Copy the data into the data packet.
    if (data)
    {
        // Copy data using compiler-generated memcpy.
        memcpy(packet, data, packetSize);
        data += packetSize;
        status = kStatus_Success;
    }
    else
    {
        if (commandTag == kCommandTag_ReadMemory)
        {
            //memcpy(packet, (uint8_t *)dataAddress, packetSize);
            status = flexspi_nor_mem_read(dataAddress, packetSize, packet);
        }
        dataAddress += packetSize;
    }

    if (status != kStatus_Success)
    {
        debug_printf("Error: %s returned status 0x%x, abort data phase\r\n",
                     (commandTag == kCommandTag_ReadMemory) ? "read memory" : "flash read resource", status);
        // Send zero length packet to tell host we are aborting data phase
        g_bootloaderContext.activePeripheral->packetInterface->writePacket(
            g_bootloaderContext.activePeripheral, (const uint8_t *)packet, 0, kPacketType_Data);
        finalize_data_phase(status);
        *hasMoreData = false;
        return kStatus_Success;
    }
    remaining -= packetSize;

#ifdef TEST_SENDER_ABORT
// Disble IAR "statement is unreachable" error
#pragma diag_suppress = Pe111
    // Send zero length packet to abort data phase.
    g_bootloaderContext.activePeripheral->packetInterface->writePacket(g_bootloaderContext.activePeripheral,
                                                                       (const uint8_t *)packet, 0, kPacketType_Data);
    finalize_data_phase(kStatus_AbortDataPhase);
    *hasMoreData = false;
    return kStatus_Success;
#endif // TEST_SENDER_ABORT;

    status = g_bootloaderContext.activePeripheral->packetInterface->writePacket(
        g_bootloaderContext.activePeripheral, (const uint8_t *)packet, packetSize, kPacketType_Data);

    if (remaining == 0)
    {
        finalize_data_phase(status);
        *hasMoreData = false;
    }
    else if (status != kStatus_Success)
    {
        debug_printf("writePacket aborted due to status 0x%x\r\n", status);
        finalize_data_phase(status);
        *hasMoreData = false;
    }
    else
    {
        g_bootloaderContext.commandInterface->stateData->dataPhase.count = remaining;
        g_bootloaderContext.commandInterface->stateData->dataPhase.address = dataAddress;
    }

    return kStatus_Success;
}

//! @brief Send a generic response packet.
void send_generic_response(uint32_t commandStatus, uint32_t commandTag)
{
    generic_response_packet_t responsePacket;
    responsePacket.commandPacket.commandTag = kCommandTag_GenericResponse;
    responsePacket.commandPacket.flags = 0;
    responsePacket.commandPacket.reserved = 0;
    responsePacket.commandPacket.parameterCount = 2;
    responsePacket.status = commandStatus;
    responsePacket.commandTag = commandTag;

    status_t status = g_bootloaderContext.activePeripheral->packetInterface->writePacket(
        g_bootloaderContext.activePeripheral, (const uint8_t *)&responsePacket, sizeof(responsePacket),
        kPacketType_Command);
    if (status != kStatus_Success)
    {
        debug_printf("Error: writePacket returned status 0x%x\r\n", status);
    }
}

//! @brief Send a get property response packet.
void send_get_property_response(uint32_t commandStatus, uint32_t *value, uint32_t numValues)
{
    get_property_response_packet_t responsePacket;
    responsePacket.commandPacket.commandTag = kCommandTag_GetPropertyResponse;
    responsePacket.commandPacket.flags = 0;
    responsePacket.commandPacket.reserved = 0;
    responsePacket.commandPacket.parameterCount = 1 + numValues; // status + value words
    responsePacket.status = commandStatus;

    for (uint32_t i = 0; i < numValues; ++i)
    {
        responsePacket.propertyValue[i] = value[i];
    }

    uint32_t packetSize =
        sizeof(responsePacket.commandPacket) + (responsePacket.commandPacket.parameterCount * sizeof(uint32_t));

    status_t status = g_bootloaderContext.activePeripheral->packetInterface->writePacket(
        g_bootloaderContext.activePeripheral, (const uint8_t *)&responsePacket, packetSize, kPacketType_Command);
    if (status != kStatus_Success)
    {
        debug_printf("Error: writePacket returned status 0x%x\r\n", status);
    }
}

//! @brief Send a read memory response packet.
void send_read_memory_response(uint32_t commandStatus, uint32_t length)
{
    read_memory_response_packet_t responsePacket;
    responsePacket.commandPacket.commandTag = kCommandTag_ReadMemoryResponse;
    responsePacket.commandPacket.flags = kCommandFlag_HasDataPhase;
    responsePacket.commandPacket.reserved = 0;
    responsePacket.commandPacket.parameterCount = 2;
    responsePacket.status = commandStatus;
    responsePacket.dataByteCount = length;

    status_t status = g_bootloaderContext.activePeripheral->packetInterface->writePacket(
        g_bootloaderContext.activePeripheral, (const uint8_t *)&responsePacket, sizeof(responsePacket),
        kPacketType_Command);
    if (status != kStatus_Success)
    {
        debug_printf("Error: writePacket returned status 0x%x\r\n", status);
    }
}


//! @}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
