/*
 * Copyright (c) 2014-2015 Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

//#include "utilities/fsl_assert.h"
#include <string.h>
#include "bootloader_common.h"
//#include "property/property.h"
//#include "memory/memory.h"
#include "bootloader/bl_context.h"
#include "crc/crc32.h"
#include "bootloader/bl_app_crc_check.h"
//#include "utilities/vector_table_info.h"
#include "target_config.h"
#include "enc_operations.h"
////////////////////////////////////////////////////////////////////////////////
// Declarations
////////////////////////////////////////////////////////////////////////////////
extern AT_QUICKACCESS_SECTION_CODE(status_t flexspi_nor_flash_read(FLEXSPI_Type *base, uint32_t dstAddr, uint32_t *src, uint32_t size));
////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////
#if BL_FEATURE_CRC_ASSERT
//! @brief Toggle a pin to the inverse of default logic state to indicate crc check failure.
static void assert_pin_to_indicate_crc_check_failure(void)
{
    // Initialize pin for GPIO
    CRC_CHECK_FAILURE_PIN_PORT->PCR[CRC_CHECK_FAILURE_PIN_NUMBER] = PORT_PCR_MUX(CRC_CHECK_FAILURE_PIN_GPIO_MODE);
    // Set pin as output
    CRC_CHECK_FAILURE_PIN_GPIO->PDDR |= ( 1 << CRC_CHECK_FAILURE_PIN_NUMBER);
    // Toggle pin (the inverse of default logic state) for crc check failure
    CRC_CHECK_FAILURE_PIN_GPIO->PTOR = 1 << CRC_CHECK_FAILURE_PIN_NUMBER;
}

// See bl_app_crc_check.h for documentation on this function.
void restore_crc_check_failure_pin(void)
{
    // Restore crc check failure pin to default state
    if ((CRC_CHECK_FAILURE_PIN_PORT->PCR[CRC_CHECK_FAILURE_PIN_NUMBER] & PORT_PCR_MUX_MASK) == PORT_PCR_MUX(CRC_CHECK_FAILURE_PIN_GPIO_MODE))
    {
        // Restore pin as input
        CRC_CHECK_FAILURE_PIN_GPIO->PDDR &= ~(1 << CRC_CHECK_FAILURE_PIN_NUMBER);
        // Restore pin as default muxing slot mode
        CRC_CHECK_FAILURE_PIN_PORT->PCR[CRC_CHECK_FAILURE_PIN_NUMBER] = PORT_PCR_MUX(CRC_CHECK_FAILURE_PIN_DEFAULT_MODE);
    }
}
#endif
// See bl_app_crc_check.h for documentation on this function.
void init_crc_check_status(property_store_t *propertyStore)
{
    // Crc check is available if BCA exists and crc parameters are set (not all 0xff bytes)
    if (!((propertyStore->configurationData.crcStartAddress == 0xffffffff) &&
          (propertyStore->configurationData.crcByteCount == 0xffffffff) &&
          (propertyStore->configurationData.crcExpectedValue == 0xffffffff)))
    {
        propertyStore->crcCheckStatus = kStatus_AppCrcCheckInactive;
    }
    else
    {
        propertyStore->crcCheckStatus = kStatus_AppCrcCheckInvalid;
    }
}

//! @brief Calculate crc on a range of flash, specified in the bootloader configuration area.
uint32_t calculate_application_crc32(crc_checksum_header_t *header, uint32_t crcHeaderStart)
{
    uint32_t crc32;

    // Initialize the CRC32 information
    crc32_data_t crcInfo;
    crc32_init(&crcInfo);

    // Run CRC, Considering skip crcExpectedValue address
    uint32_t bypassStartAddress = crcHeaderStart + ((uint32_t)&header->crcExpectedValue - (uint32_t)&header->tag);
    uint32_t bypassEndAddress = bypassStartAddress + sizeof(header->crcExpectedValue);

    if ((header->crcStartAddress >= bypassEndAddress) ||
        (header->crcStartAddress + header->crcByteCount <= bypassStartAddress))
    {
        crc32_update(&crcInfo, (uint8_t *)header->crcStartAddress, header->crcByteCount);
    }
    else
    {
        // Assume that crcExpectedValue address (4 byte) resides in crc addresses completely
        crc32_update(&crcInfo, (uint8_t *)header->crcStartAddress, bypassStartAddress - header->crcStartAddress);
        crc32_update(&crcInfo, (uint8_t *)bypassEndAddress,
                     header->crcStartAddress + header->crcByteCount - bypassEndAddress);
    }

    // Finalize the CRC calculations
    crc32_finalize(&crcInfo, &crc32);

    return crc32;
}

//! @brief Check if crc check addresses reside in the range of valid memory space (Pflash or QSPIflash)
static bool is_crc_check_address_valid(bootloader_configuration_data_t *config)
{
    bool isAddressValid;

    isAddressValid = ((config->crcStartAddress >= FLASH_VALID_START) &&
                          (config->crcStartAddress + config->crcByteCount - 1 <= FLASH_VALID_END));
    return isAddressValid;
}

// See bl_app_crc_check.h for documentation on this function.
bool is_application_crc_check_pass(void)
{
    bool isCrcCheckPassed = true;
    
    crc_info crcInfo;
    memcpy(&crcInfo,(void *)(BL_PARAMETERS_ADDRESS+sizeof(enc_region_hdr_t)),sizeof(crcInfo));
    if(crcInfo.tag == CRC_TAG)
    {
      crc32_data_t crcData;
      uint32_t crc32;
      uint32_t Offset = 0;
      uint32_t flashAddress = BL_APP_VECTOR_TABLE_PHY_ADDRESS;
      uint32_t dataLength = crcInfo.crcConfig.byteCountCrc;
      uint32_t buff[64];
      crc32_init(&crcData);
      while(dataLength)
      {
        if(dataLength > 256)
        {
            flexspi_nor_flash_read(BL_FLEXSPI_BASE_ADDRESS,(uint32_t)(flashAddress+Offset),(uint32_t *)buff,256);
            crc32_update(&crcData, (const uint8_t *)buff, 256);
            dataLength -= 256;
            Offset += 256;
        }
        else
        {
            flexspi_nor_flash_read(BL_FLEXSPI_BASE_ADDRESS,(uint32_t)(flashAddress+Offset),(uint32_t *)buff,dataLength);
            crc32_update(&crcData,(const uint8_t *)buff, dataLength);
            Offset += dataLength;
            dataLength = 0;
        }
      }
      // Finalize the CRC calculations
      crc32_finalize(&crcData, &crc32);
      if(crc32 == crcInfo.crcConfig.currentCrc)
      {
        isCrcCheckPassed = true;
      }
      else
      {
        //isCrcCheckPassed = false;
      }
    }
    else
    {
      // no crc information, directly return
      
    }
    
    return isCrcCheckPassed;
}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
