/*
 * Copyright (c) 2015 Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#if !defined(__FLEXSPINOR_MEMORY_H__)
#define __FLEXSPINOR_MEMORY_H__

#include "memory/memory.h"

#if BL_FEATURE_FLEXSPI_NOR_MODULE
////////////////////////////////////////////////////////////////////////////////
// Definitions
////////////////////////////////////////////////////////////////////////////////
enum
{
    //! @brief Max page size used to create a page buffer
    kFlexSpiNorMemory_MaxPageSize = 1024U,
    //! @brief Max total size of NOR FLASH supported by FlexSPI
    kFlexSpiNorMemory_MaxSize = 256U * 1024 * 1024,
    //! @brief Max Size of SPI NOR alias memory
    kFlexSpiNorAliasAreaMaxSize = 0x04000000U, // 64MB

    kFlexSpiNorProgramConfigBlock_Tag = 0x0F,
};

#define MAGIC_NUMBER_FLEXSPI_NOR_PROGRAM_CFG \
    (((uint32_t)kFlexSpiNorProgramConfigBlock_Tag << 28) | kFlexSpiNorProgramConfigBlock_Tag)
#if 0
#define MAGIC_NUMBER_CONVERTING_KEY_TO_OTPMK	0xE00055aa
#define TAG_NOUNCE_COUNTER						0xE5342100
typedef struct
{
	uint32_t tag;
	uint8_t nounce_counter[16];
	uint32_t reserve[3];
}switching_nounce_counter_t;
#endif

typedef enum _flexspi_nor_property
{
    kFlexspiNorProperty_InitStatus = 0,
    kFlexspiNorProperty_StartAddress = 1,           //!< Tag used to retrieve start address
    kFlexspiNorProperty_TotalFlashSizeInKBytes = 2, //!< Tag used to retrieve total flash size in terms of KByte
    kFlexspiNorProperty_PageSize = 3,               //!< Tag used to retrieve page size in terms of byte
    kFlexspiNorProperty_SectorSize = 4,             //!< Tag used to retrieve sector size in term of byte
    kFlexspiNorProperty_BlockSize = 5,              //!< Tag used to retrieve block size in terms of byte

    kFlexspiNorProperty_TotalFlashSize = 0x10, //!< Tag used to retrieve total flash size in terms of byte
} flexspi_nor_property_t;

/* FlexSPI NOR status */
enum _flexspi_nor_status
{
    kStatus_FLEXSPINOR_ProgramFail = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 0), //!< Status for Page programming failure
    kStatus_FLEXSPINOR_EraseSectorFail = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 1), //!< Status for Sector Erase failure
    kStatus_FLEXSPINOR_EraseAllFail = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 2),    //!< Status for Chip Erase failure
    kStatus_FLEXSPINOR_WaitTimeout = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 3),     //!< Status for timeout
    kStatus_FlexSPINOR_NotSupported = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 4),    // Status for PageSize overflow
    kStatus_FlexSPINOR_WriteAlignmentError = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 5), //!< Status for Alignement error
    kStatus_FlexSPINOR_CommandFailure =
        MAKE_STATUS(kStatusGroup_FLEXSPINOR, 6), //!< Status for Erase/Program Verify Error
    kStatus_FlexSPINOR_SFDP_NotFound = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 7), //!< Status for SFDP read failure
    kStatus_FLEXSPINOR_Unsupported_SFDP_Version =
        MAKE_STATUS(kStatusGroup_FLEXSPINOR, 8), //!< Status for Unrecognized SFDP version
    kStatus_FLEXSPINOR_Flash_NotFound = MAKE_STATUS(kStatusGroup_FLEXSPINOR, 9), //!< Status for Flash detection failure
    kStatus_FLEXSPINOR_DTRRead_DummyProbeFailed =
        MAKE_STATUS(kStatusGroup_FLEXSPINOR, 10), //!< Status for DDR Read dummy probe failure
};

////////////////////////////////////////////////////////////////////////////////
// Prototypes
////////////////////////////////////////////////////////////////////////////////
#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

//! @name FlexSPI NOR memory
//@{

//! @brief Initialize SPI NOR memory
status_t flexspi_nor_mem_init(void);

//! @brief Read SPI NOR memory.
status_t flexspi_nor_mem_read(uint32_t address, uint32_t length, uint8_t *buffer);

//! @brief Write SPI NOR memory.
status_t flexspi_nor_mem_write(uint32_t address, uint32_t length, const uint8_t *buffer);

//! @brief Fill SPI NOR memory with a word pattern.
status_t flexspi_nor_mem_fill(uint32_t address, uint32_t length, uint32_t pattern);

//! @brief Erase SPI NOR memory
status_t flexspi_nor_mem_erase(uint32_t address, uint32_t length);

//! @brief flush cached data to SPI NOR memory
status_t flexspi_nor_mem_flush(void);


//@}

#if defined(__cplusplus)
}
#endif // __cplusplus

#endif // BL_FEATURE_FLEXSPI_NOR_MODULE
#endif // __FLEXSPINOR_MEMORY_INTERFACE_H__
////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
