/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "clock_config.h"
#include <assert.h>
#include "fsl_debug_console.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
AT_QUICKACCESS_SECTION_CODE(extern void BOARD_SetRunMode(
    SCG_Type *scg, uint32_t scgRunConfig, QuadSPI_Type *qspi, clock_ip_name_t qspiClock, uint32_t qspiClockConfig));
extern const scg_sys_clk_config_t g_sysClkConfigNormalRun;
extern const scg_sys_clk_config_t g_sysClkConfigVlprFirc;
extern const scg_sys_clk_config_t g_sysClkConfigVlprSirc;
extern const scg_sys_clk_config_t g_sysClkConfigVlprFircAN12573;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
extern bool BOARD_IsRunOnQSPI(void);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*
 * Set the clock configuration for HSRUN mode.
 */
void APP_SetClockHsrun(void)
{
}

/*
 * Set the clock configuration for RUN mode from HSRUN mode.
 */
void APP_SetClockRunFromHsrun(void)
{
}

/*
 * Set the clock configuration for RUN mode from VLPR mode.
 */
void APP_SetClockRunFromVlpr(void)
{
    QuadSPI_Type *qspi = BOARD_IsRunOnQSPI() ? QuadSPI0 : NULL;
    uint32_t *config   = (uint32_t *)(&g_sysClkConfigNormalRun);

    PRINTF("HW_OCOTP_PDN = 0x%x\r\n", OCOTP_CTRL->HW_OCOTP_PDN);
    PRINTF("SPLLCSR = 0x%x\r\n", SCG0->SPLLCSR);            
    PRINTF("APLLCSR = 0x%x\r\n", SCG0->APLLCSR);
    PRINTF("SOSCCSR = 0x%x\r\n", SCG0->SOSCCSR);            
    
    if(OCOTP_CTRL->HW_OCOTP_PDN == 1)
    {
        /* Power down OCOTP. */
        OCOTP_CTRL->HW_OCOTP_PDN = 1;                   //4.8mW improvement
    }

    /* Enable SPLL. */
    SCG0->SPLLCSR |= SCG_SPLLCSR_SPLLEN_MASK;          //no relevant power improvement
    /* Enable APLL. APLL is not initialized in clock_config.c */
    //SCG0->APLLCSR |= SCG_APLLCSR_APLLEN_MASK;          //0.2mW improvement
    /* Enable SOSC. */
    SCG0->SOSCCSR |= SCG_SOSCCSR_SOSCEN_MASK;          //0.8mW improvement
    while (!(SCG->SOSCCSR & SCG_SOSCCSR_SOSCVLD_MASK))
    {
    }    
    /* Enable SIRC */
    //SCG0->SIRCCSR |= SCG_SIRCCSR_SIRCEN_MASK; 

    /* Recover SPLL */
    SCG0->SPLLCSR |= SCG_SPLLCSR_SPLLEN_MASK;
    while (!CLOCK_IsSysPllValid())
    {
    }

    /* When switching from VLPR to RUN, switch RUN clock source back to RUN source */
    BOARD_SetRunMode(SCG0, *config, qspi, kCLOCK_Qspi,
                     PCC1_PCC_QSPI_OTFAD_CGC_MASK | PCC1_PCC_QSPI_OTFAD_PCS(3)); /* QSPI source: 48M FIRC Async */
}

/*
 * Set the clock configuration for VLPR mode.
 */
void APP_SetClockVlpr(uint32_t VlprTargetClockConfig)
{
    QuadSPI_Type *qspi = BOARD_IsRunOnQSPI() ? QuadSPI0 : NULL;
    uint32_t *config   = NULL;

    /* When switching from RUN to VLPR, first switch RUN clock source to VLPR source */
    if (VlprTargetClockConfig == 2) 
    {
        /* Set VLPR frequency to 16Mhz - Run from SIRC */
        config = (uint32_t *)(&g_sysClkConfigVlprSirc);
        CLOCK_SetVlprModeSysClkConfig((scg_sys_clk_config_t *)config);
        BOARD_SetRunMode(SCG0, *config, qspi, kCLOCK_Qspi,
                     PCC1_PCC_QSPI_OTFAD_CGC_MASK | PCC1_PCC_QSPI_OTFAD_PCS(2)); /* QSPI source: 16M SIRC Async */
    }
    else if (VlprTargetClockConfig == 1)
    {
        /* Set VLPR frequency to 48Mhz with optimized clock dividers config - Run from FIRC */
        config = (uint32_t *)(&g_sysClkConfigVlprFircAN12573);
        CLOCK_SetVlprModeSysClkConfig((scg_sys_clk_config_t *)config);
        BOARD_SetRunMode(SCG0, *config, qspi, kCLOCK_Qspi,
                     PCC1_PCC_QSPI_OTFAD_CGC_MASK | PCC1_PCC_QSPI_OTFAD_PCS(2)); /* QSPI source: 16M SIRC Async */
    }
    else
    {
        /* Set VLPR frequency to 48Mhz with Standard clock dividers config - Run from FIRC */
        config = (uint32_t *)(&g_sysClkConfigVlprFirc);
        CLOCK_SetVlprModeSysClkConfig((scg_sys_clk_config_t *)config);
        BOARD_SetRunMode(SCG0, *config, qspi, kCLOCK_Qspi,
                     PCC1_PCC_QSPI_OTFAD_CGC_MASK | PCC1_PCC_QSPI_OTFAD_PCS(2)); /* QSPI source: 16M SIRC Async */    
    }
    
    /* Disable SPLL to work around hardware issue */
    SCG0->SPLLCSR &= ~SCG_SPLLCSR_SPLLEN_MASK;
}
