/*
 * Copyright 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#define _GNU_SOURCE
/*=========================================================================

Include Files

===========================================================================*/
#include <arpa/inet.h>
#include <assert.h>
#include <linux/if_tun.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>   
#include <unistd.h>

// netlink includes
#include <asm/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <net/if.h>

#include "Framer.h"
#include "FSCIFrame.h"
#include "PhysicalDevice.h"
#include "UARTConfiguration.h"
#include "Zigbee_BlackBox_HSDK.h"

/*=========================================================================

Public type definitions

===========================================================================*/
static volatile uint8_t nodeHasJoined_g = FALSE; 
static volatile bool_t stopProgram_g = FALSE;
static volatile bool_t binding_done_g = FALSE;
static uint8_t shortAddr[] = {0xFF, 0xFF};

/* FSCI standard payloads */
static uint8_t set_ch_buf[]             = {0x00, 0x00, 0x00, 0x00}; /* Channel value has to be placed in last position in hexadecimal format*/
static uint8_t set_epid_buf[]           = {0xFA, 0xCE, 0xFA, 0xCE, 0xFA, 0xCE, 0xFA, 0xCE,};
static uint8_t permit_join_buf[]		= {0xFF, 0xFC, 0xFE, 0x00}; /* First two bytes: 0x0000 for Coordinator
																						0xFFFC for Router/Coordinator boards
																		Last two bytes: Time in seconds for ´permit join´ to be active 
																		*/
uint8_t * read_dynamic_attributes_buf; 	

static uint8_t read_attributes_buf[]    = {UNICAST_ADD_MODE, 0xFF, 0xFF, DEFAULT_ENDPOINT, DEFAULT_ENDPOINT, CLUSTER_ID_HIGH, 
											LVL_CTRL_CLUSTER_ID_LOW, SERVER_TO_CLIENT, 0x00, 0x00, 0x00, NUMBER_OF_ATTRIBUTES, 0x00, 0x00
											};/* Fisrt byte: Select address mode (Unicast)
												 Next two bytes: Short address of the required node.
												 Next two bytes: Cluster ID
												 Next byte: Direction (server to client)
												 Next tree bytes: Manufacturer (0x00 - not required)
												 Next byte: Number of attributes to be read (By default: 0x01)
												 Rest of bytes: Attribute list (Depends of the number of attributes)
												*/
static uint8_t on_off_cmd_buf[]      	= {UNICAST_ADD_MODE, 0xFF, 0xFF, DEFAULT_ENDPOINT, DEFAULT_ENDPOINT, TOGGLE_CMD};
											/* First byte: Select address mode (Unicast)
											   Next two bytes (0xFF): To be replaced with the short address of the target node
											   Next byte: Source endpoint.
											   Next byte: Destination endpoint.
											   Last byte: Command.  
											  */

static uint8_t bind_buf[]               = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, DEFAULT_ENDPOINT, CLUSTER_ID_HIGH,
											ON_OFF_CLUSTER_ID_LOW, 0x03, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, DEFAULT_ENDPOINT
											};/* First 8 bytes (0xFF): To be replaced with the Target Extended Address
												 Next endpoint: Target endpoint
												 Next two bytes: Cluster ID
												 Next byte (0x03): Specify to use the coordinator extended address
												 Next 8 bytes (0xFF): To be replaced with the Local Extended Address
												 Last byte: Destination endpoint  
												*/
/*static uint8_t config_report_buf[]		= {UNICAST_ADD_MODE, 0xFF, 0xFF, DEFAULT_ENDPOINT, DEFAULT_ENDPOINT, CLUSTER_ID_HIGH,
											ON_OFF_CLUSTER_ID_LOW, CLIENT_TO_SERVER, 0x00, 0x00, 0x00, NUMBER_OF_ATTRIBUTES, 0x00
											};*/
static uint8_t match_descriptor_buf[]   = {0xFF, 0xFF, HOME_AUTO_APP_PROFILE_HIGH, HOME_AUTO_APP_PROFILE_LOW, INPUT_CLUSTERS, CLUSTER_ID_HIGH,
											BASIC_CLUSTER_ID_LOW, CLUSTER_ID_HIGH, GROUPS_CLUSTER_ID_LOW, CLUSTER_ID_HIGH, ON_OFF_CLUSTER_ID_LOW,
											OUTPUT_CLUSTERS, CLUSTER_ID_HIGH, IDENTIFY_CLUSTER_ID_LOW
											};/* First two bytes: Short address of the required node. 
												 Requested Profile ID:   0x0104 (Lighting & Occupacy devices)
												 Input cluster list(3):  0x0000 (Basic Cluster ID)
												 					  	 0x0004 (Groups Cluster ID)
												 					     0x0006 (OnOff Cluster ID)
												 Output cluster list(1): 0x0003 (Identify Cluster ID)  
												*/

static uint8_t simple_descriptor_buf[] 	= {0xFF, 0xFF, DEFAULT_ENDPOINT}; /* First two bytes: Short address of the required node. 
																			 Last byte: endpoint*/

static uint8_t install_code_buf[]		= {0x75, 0xc5, 0x2c, 0x60, 0xf1, 0x2a, 0x03, 0xa8, 0x75, 0xc5, 0x2c, 0x60, 0xf1, 0x2a, 0x03, 0xa8,
											0x75, 0xc5, 0x2c, 0x60, 0xf1, 0x2a, 0x03, 0xa8};/*{0x47, 0x6F, 0xB4, 0x46, 0x80, 0x88, 0x83, 0x2B, 0x47, 0x6F, 0xB4, 0x46, 0x80, 0x88, 0x83, 0x2B, 
											0x47, 0x6F, 0xB4, 0x46, 0x80, 0x88, 0x83, 0x2B,
											};*//* First 8 bytes: MAC address of the joiner device
												 By default:
												 Last 16 bytes: MAC address of the joiner device repeated once
												 Refer to chapter 13. Install Code, on "Host SDK on Linux OS Application Note" */	

/*=========================================================================

Public global variables declarations

===========================================================================*/
/*! ************************************************************************
\fn uint8_t IntToHexChannel(uint8_t value)

\brief  converts an integet channel value to hexadecimal.

\param      value   integer value

\retval     uint8_t     hexadecimal
 ************************************************************************* */
static uint8_t IntToHexChannel(uint8_t value)
{
   	uint8_t hex = 0x0B;
   	hex += (value - hex);
   	return hex;
}

/*! ************************************************************************
\fn void SigHandler(int signo)

\brief  Ctrl-C signal handler, only used to stop program. 
 ************************************************************************* */
static void SigHandler(int signo)
{
    if (signo == SIGINT) {
        stopProgram_g = TRUE;
    }
}

/*! ************************************************************************
\fn int GetBaudrate(int value)

\brief  converts baudrate value to enumeration.

\param      value   selected baudrate value by the user

\retval     int     enumaration baudrate value
 ************************************************************************* */
static int GetBaudrate(int value)
{

    if (value == 115200)
        return BR115200;

    return -1;
}

/*! ************************************************************************
\fn void ReverseByteArray(uint8_t *buf, int size)

\brief  This function reverses an array positions. 
        [0x11, 0x22, 0x33, 0x44] -> [0x44, 0x33, 0x22, 0x11]

\param      buf     pointer to the buffer that wants to be reversed.
\param      size    size of the buffer.
 ************************************************************************* */
static void ReverseByteArray(uint8_t *buf, int size)
{
    int i;
    uint8_t *temp = malloc(size);

    memcpy(temp, buf, size);

    for (i = 0; i < size; i++) {
        buf[i] = temp[size - i - 1];
    }

    free(temp);
}

/*! ************************************************************************
\fn void FrameReceived(void *callee, void *response)

\brief  Executes on every RX packet.

\param      response    data with the received frame. 
 ************************************************************************* */
static void FrameReceived(void *callee, void *response)
{
    int16_t i;
    uint8_t opCode;
    uint8_t rsp = 0;
    FSCIFrame *frame = (FSCIFrame *)response;               /* Cast the received frame */ 

    if (frame->opGroup != RX_OG && frame->opGroup != RX_DISC_OG && frame->opGroup != TX_OG && frame->opGroup != RX_RSP_OG) {
    	printf("Destoy frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, frame->opCode);
        DestroyFSCIFrame(frame);
        return;
    }

    opCode = frame->opCode;
    if(opCode == 0x00 && frame->opGroup == RX_RSP_OG){
    }
    else if(opCode == 0x01 && frame->opGroup == RX_RSP_OG){ /* This is a On/Off command response */
    	opCode = ON_OFF_CMD_OC;
    }
    else if(opCode == 0x02 && frame->opGroup == RX_RSP_OG){ /* This is a Read Individual Attribute response */
    	opCode = READ_ATT_OC;
    }
    else if(opCode == 0x00){ /* The OpCode response is placed in the data frame */
    	opCode = frame->data[frame->length - 1];
    	rsp = 1;
    }
    

	switch (opCode) {

	case FACTORY_RST_OC:
		printf("RX: FactoryReset.Status");
		if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case FACTORY_NEW_RESTART_OC:
		printf("RX: FactoryNewRestart\n");
		if (frame->data[0] == 0x00) {
            printf("\t STARTUP\n");
        }
		break;

	case NODE_CMD_ID_LIST_OC:
		printf("RX: NodeCommandIDList\n");
		break;

	case NODE_CLUSTER_ATT_LIST_OC:
		printf("RX: NodeClusterAttributeList:\n");
#if SEE_ATTRIBUTE_LIST
		printf("\t Endpoint: %02x\n",frame->data[0]);
		printf("\t Profile ID: %02x%02x\n",frame->data[1],frame->data[2]);
		printf("\t Cluster ID: %02x%02x\n",frame->data[3],frame->data[4]);
		printf("\t Attribute List: ");
		for(i = 5; i < frame->length ; i+= 2){
			printf("%02x%02x,",frame->data[i],frame->data[i+1]);
		}
		printf("\n");
#endif
		break;

	case NODE_CLUSTER_LIST_OC:
		printf("RX: NodeClusterList\n");
		break;

	case SET_CHANNEL_MASK_OC:
		printf("RX: SetChannel.Status");
        if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case SET_EPID_OC:
		if(frame->opGroup == RX_OG){
			printf("RX: Status");
			if (frame->data[0] == 0x00) {
	            printf(" -> Success\n");
	        }
		}
        else if(frame->opGroup == RX_RSP_OG){						/* SET EPID and CONFIGURE REPORTING has the same OpCode but different OpGroup*/
			printf("RX: ConfigureReporting.Response\n");
			printf("\t Short Addres: [%02x%02x]\n",frame->data[1],frame->data[2]);
			printf("\t Endpoint: [%02x]\n",frame->data[3]);
			printf("\t Cluster ID: [%02x%02x]\n",frame->data[4],frame->data[5]);
		}
		break;

	case CREATE_NWK_OC:
		if(!rsp){ /* Network response */
			printf("RX: NetworkJoinedFormed");
			if (frame->data[0] == 0x01) {
	            printf(" -> Formed New Network\n");
	            printf("\t Short Address: 0x%02x%02x\n",frame->data[1],frame->data[2]);
	            printf("\t Extended Address: ");
	            for(i = 3 ; i < (frame->length - 1) ; i++){			/* Print coordinator extended address */
	            	printf("%02x",frame->data[i]);
	            	bind_buf[9 + i] = frame->data[i];				/* Copy coordinator extended address to binding buffer */
	            }
	            printf("\n\t Channel: %d",frame->data[i]);
	            printf("\n");
        	}
        	else if(frame->data[0] == 0x00){
        		printf("-> Joined to Existing Network\n");
        	}
		}else{/* Board response */
			printf("RX: CreateNetwork.Status");
			if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        	}
		}
		break;

	case PERMIT_JOIN_OC:
		printf("RX: PermitJoin.Status");
		if (frame->data[0] == 0x00) {
			printf(" -> Success\n");
		}
		break;

	case DEVICE_ANNOUNCE_OC:									/* Joiner Request */
		printf("RX: Device Announce\n");
		printf("\t Short Address: %02x%02x\n",frame->data[0],frame->data[1]);
		printf("\t Extended Address: ");
		for(i = 2 ; i < (frame->length - 1) ; i++){				/* Print joiner extended address */
			printf("%02x", frame->data[i]);
			bind_buf[i-2] = frame->data[i];						/* Copy joiner extended address to binding buffer */
		}
		printf("\n");
		shortAddr[0] = frame->data[0];							/* Copy joiner short address */
		shortAddr[1] = frame->data[1];
		break;

	case ROUTER_DISCOVERY_OC:
		printf("RX: RouterDiscovery.Confirm\n");
		printf("\t Status");
		if (frame->data[0] == 0x00) {
			printf(" -> Success\n");
			nodeHasJoined_g = TRUE;								/* A node has been accepted in the network */
		}
		printf("\t NwkStatus");
		if (frame->data[1] == 0x00) {
			printf(" -> Success\n");
		}
		break;

	case READ_ATT_OC:
		if(frame->opGroup == RX_OG){
			printf("RX: ReadAttribute\n");
			printf("\t Status");
			if (frame->data[0] == 0x00) {
			printf(" -> Success\n");
			}
		}else if(frame->opGroup == RX_RSP_OG){
			printf("RX: ReadIndividualAttribute.Response\n");
			printf("\t Source Addres: [%02x%02x]\n",frame->data[1],frame->data[2]);
			printf("\t Endpoint: [%02x]\n",frame->data[3]);
			printf("\t Cluster ID: [%02x%02x]\n",frame->data[4],frame->data[5]);
			printf("\t Attribute ID: [%02x%02x] ",frame->data[7],frame->data[6]);
			if(frame->data[6] == 0x00 && frame->data[7] == 0x00)	/* Print attribute characteristics */
				printf(" On/Off\n");
			else
				printf("\n");	
			
			printf("\t Attribute Data Type: [%02x] ",frame->data[9]);
			if(frame->data[9] == 0x10) 
				printf(" Boolean\n");
			else if(frame->data[9] == 0xff)
				printf(" Hexadecimal\n");
			else						/* Add more attribute data types */
				printf("\n");

			printf("\t Value: [%02x]",frame->data[12]);
			if(frame->data[12] == 0x00){
				printf(" Off\n");
			}else if(frame->data[12] == 0x01){
				printf(" On\n");
			}
		}
		break;

	case ON_OFF_CMD_OC:
		if(frame->opGroup == RX_OG){
			printf("RX: OnOffWithNoEffects\n");
			printf("\t Status");
			if (frame->data[0] == 0x00) {
			printf(" -> Success\n");
			}
		}else if(frame->opGroup == RX_RSP_OG){
			printf("RX: OnOffWithNoEffectsDefault.Response\n");
			printf("\t Endpoint: [%02x]\n",frame->data[1]);
			printf("\t Cluster ID: [%02x%02x]\n",frame->data[2],frame->data[3]);
			printf("\t Command ID: [%02x]\n",frame->data[4]);
			printf("\t StatusCode: [%02x]\n",frame->data[5]);
		}
		break;

	case BIND_OC:
		if(!rsp){ /* Joiner response */
			printf("RX: Bind.Response\n");
			printf("\t Status");
			if (frame->data[1] == 0x00) {
           		printf(" -> Success\n");
           		binding_done_g = TRUE;
        	}
		}else{ /* Board response */
			printf("RX: Bind.Status");
			if (frame->data[0] == 0x00) {
            	printf(" -> Success\n");
        	}
		}
		break;

	case MATCH_DESCRIPTOR_OC:
		if(!rsp){ /* Joiner response */
			printf("RX: MatchDescriptor.Response\n");
			printf("\t Status");
			if (frame->data[1] == 0x00) {
           		printf(" -> Success\n");
        	}
        	printf("\t Source Addres: [%02x%02x]\n",frame->data[2],frame->data[3]);
        	if(frame->data[4] == 0x00){
        		printf("\t No matched Endpoints\n");
        	}
        	else{
        		for(i = 0; i < frame->data[4]; i++){
        			printf("\t Matched in Endpoint: %02X\n", frame->data[5 + i]);
        		}
        	}
		}else{ /* Board response */
			printf("RX: MatchDescriptor.Status");
			if (frame->data[0] == 0x00) {
            	printf(" -> Success\n");
        	}
		}
		break;

	case SIMPLE_DESCRIPTOR_OC:
		if(!rsp){ /* Joiner response */
				printf("RX: SimpleDescriptor.Response");
				if (frame->data[1] == 0x00) {
	           		printf(" -> Success\n");
	        	}
	        	printf("\t Source Addres: [%02x%02x]\n",frame->data[2],frame->data[3]);
	        	printf("\t Endpoint: [%02x]\n",frame->data[5]);
	        	printf("\t Profile ID: [%02x%02x]\n",frame->data[6],frame->data[7]);
	        	printf("\t Supported InClusters: %02x\n", frame->data[11]);
	        	for(i = 0; i < frame->data[11]; i++){
	        		printf("\t\t [%02x%02x]\n",frame->data[12 + 2*i],frame->data[13 + 2*i]);
	        	}
	        	printf("\t Supported OutClusters: %02x\n", frame->data[(12 + frame->data[11]*2)]);
	        	for(i = 0; i < frame->data[(12 + frame->data[11]*2)]; i++){
	        		printf("\t\t [%02x%02x]\n",frame->data[24 + 2*i],frame->data[24 + 2*i]);
	        	}
			}else{ /* Board response */
				printf("RX: SimpleDescriptor.Status");
				if (frame->data[0] == 0x00) {
	            	printf(" -> Success\n");
	        	}
			}
		break;

	case INSTALL_CODE_OC:
		if(!rsp){ /* Network response */
			printf("RX: New Link Key Added:\n");
			printf("\t");
			for(i = 0; i < frame->length; i++){
        		printf("%02X", frame->data[i]);
        	}
        	printf("\n");
		}else{ /* Board response */
			printf("RX: InstallCode.Status");
			if (frame->data[0] == 0x00) {
		       	printf(" -> Success\n");
		    }
		}
		break;

	case LEAVE_IND_OC:
			printf("RX: Leave\n");
		break;

	default:
        printf("RX: unknown frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, opCode);
        break;
	}

    DestroyFSCIFrame(frame);
}

int main(int argc, char **argv)
{
    /* Check number of arguments. */
    if (argc < 4) {
        printf("Usage UART: # %s {/dev/ttyACMx | /dev/ttymxcx} channel [baudrate bps]\n", argv[0]);
        exit(1);
    } 

    /* Begin UART configuration */
    DeviceType dev_type = UART;
    void *serial_config = defaultConfigurationData();
    int baudrate = GetBaudrate(atoi(argv[3]));

    if (baudrate == -1) {
        printf("Wrong baudrate value.\n");
        exit(1);
    } else {
        setBaudrate(serial_config, baudrate);
    }
    /* End UART configuration */

    /* Get selected channel */
    uint8_t channel = atoi(argv[2]);
    assert(channel >= 11 && channel <= 26);
	set_ch_buf[3] = IntToHexChannel(channel);


    PhysicalDevice *device = InitPhysicalDevice(dev_type, serial_config, argv[1], GLOBAL);              /* Initialize device with UART configuration and the selected port */
    Framer *framer = InitializeFramer(device, FSCI, LENGTH_FIELD_SIZE, CRC_FIELD_SIZE, _LITTLE_ENDIAN); /* Configure framer */
    OpenPhysicalDevice(device);                                                                         /* Open kinetis device */
    AttachToFramer(framer, NULL, FrameReceived);                                                        /* Set callback */

    /* Frames */
    FSCIFrame *factory_reset        = CreateFSCIFrame(framer, TX_OG, FACTORY_RST_OC,               NULL, 0, VIF);
    FSCIFrame *create_network       = CreateFSCIFrame(framer, TX_OG, CREATE_NWK_OC,                NULL, 0, VIF);
    FSCIFrame *set_channel          = CreateFSCIFrame(framer, TX_OG, SET_CHANNEL_MASK_OC,          set_ch_buf, sizeof(set_ch_buf), VIF);
    FSCIFrame *set_epid             = CreateFSCIFrame(framer, TX_OG, SET_EPID_OC,                  set_epid_buf, sizeof(set_epid_buf), VIF);
    FSCIFrame *permit_join          = CreateFSCIFrame(framer, TX_OG, PERMIT_JOIN_OC,               permit_join_buf, sizeof(permit_join_buf), VIF);
    FSCIFrame *install_code 		= CreateFSCIFrame(framer, TX_OG, INSTALL_CODE_OC, 				install_code_buf, sizeof(install_code_buf), VIF);
    FSCIFrame *read_attributes; 
    FSCIFrame *send_on_off_cmd; 
    FSCIFrame *bind;
    /*FSCIFrame *config_reporting;*/
    FSCIFrame *match_descriptor;
    FSCIFrame *simple_descriptor;


    /* Factory reset */
    printf("TX: FactoryReset\n");
    SendFrame(framer, factory_reset);
    sleep(1);

    /* Set channel */
    printf("TX: SetChannelMask\n");
    SendFrame(framer, set_channel);
    sleep(1);

#if USE_SET_XPANID
    /* Set extended PAN ID (OPTIONAL) */
    printf("TX: SetExtendedPANID\n");
    SendFrame(framer, set_epid);
    sleep(1);
#endif

    /* Start network */
    printf("TX: StartNetworkMessage\n");
    SendFrame(framer, create_network);
    sleep(2);

#if USE_INSTALL_CODE
    /* Set Unique Link Key */
    printf("TX: InstallCode.Request\n");
    SendFrame(framer, install_code);
    sleep(1);

#endif
    /* Permit join */
    printf("TX: PermitJoining.Request\n");
    SendFrame(framer, permit_join);
    sleep(1);

    stopProgram_g = FALSE;
    while(!stopProgram_g)
    {
    	/* Wait until a node joins to the Thread network */
        if(nodeHasJoined_g)
        {
            nodeHasJoined_g = FALSE;
            sleep(3);

#if USE_MATCH_DESCRIPTOR
            /* Match Descriptor */
    		printf("\nTX: MatchDescriptor.Request\n");
    		match_descriptor_buf[0] = shortAddr[0];
			match_descriptor_buf[1] = shortAddr[1];
    		match_descriptor = CreateFSCIFrame(framer, TX_OG, MATCH_DESCRIPTOR_OC, match_descriptor_buf, sizeof(match_descriptor_buf), VIF);
		    SendFrame(framer, match_descriptor);
		    sleep(1);
#endif
#if USE_SIMPLE_DESCRIPTOR
		    /* Simple Descriptor */
    		printf("\nTX: SimpleDescriptor.Request\n");
    		simple_descriptor_buf[0] = shortAddr[0];
			simple_descriptor_buf[1] = shortAddr[1];
    		simple_descriptor = CreateFSCIFrame(framer, TX_OG, SIMPLE_DESCRIPTOR_OC, simple_descriptor_buf, sizeof(simple_descriptor_buf), VIF);
		    SendFrame(framer, simple_descriptor);
		    sleep(1);
#endif
#if USE_READ_ATTRIBUTES

		    read_attributes_buf[1] = shortAddr[0];
			read_attributes_buf[2] = shortAddr[1];

		    #if USE_DYNAMIC_ATTRIBUTES /*  Read multiple attributes */
		    	


					uint8_t attribute_count = 0x00;	
		    		read_dynamic_attributes_buf = (uint8_t *) calloc((12 + NUMBER_OF_ATTRIBUTES*2), sizeof(uint8_t)); 
            		for (int i = 0; i < (12 + NUMBER_OF_ATTRIBUTES*2); ++i)
            		{
            			if(i < 12)
            			read_dynamic_attributes_buf[i] = read_attributes_buf[i];
            			else if (i > 13 && (i%2) != 0)
            			{
            				attribute_count++;
            				read_dynamic_attributes_buf[i] = attribute_count;
            			}
            			printf("%x", read_dynamic_attributes_buf[i]);
            		}
            		printf("\n");

            		printf("\nTX: ReadMultipleAttributes.Request\n");
            		read_attributes = CreateFSCIFrame(framer, TX_REQ_OG, READ_ATT_OC, read_dynamic_attributes_buf, (12 + NUMBER_OF_ATTRIBUTES*2) * sizeof(uint8_t), VIF);
				    SendFrame(framer, read_attributes);
				    sleep(2);
				   

            #else
		           	/* Read a single Attribute */
		    		printf("\nTX: ReadAttribute.Request\n");
		    		read_attributes = CreateFSCIFrame(framer, TX_REQ_OG, READ_ATT_OC, read_attributes_buf, sizeof(read_attributes_buf), VIF);
				    SendFrame(framer, read_attributes);
				    sleep(2);

		    #endif

		    

#endif
#if USE_FIND_AND_BIND
		    /* Find & Bind Initiator */
    		printf("\nTX: Find & Bind Initiator\n");
    		bind = CreateFSCIFrame(framer, TX_OG, BIND_OC, bind_buf, sizeof(bind_buf), VIF);
		    SendFrame(framer, bind);
		    sleep(3);
#endif      
#if USE_SET_ATTRIBUTE
		    /* On/Off Cluster command */
    		printf("\nTX: OnOffWithNoEffects\n");
    		on_off_cmd_buf[1] = shortAddr[0];
			on_off_cmd_buf[2] = shortAddr[1];
    		send_on_off_cmd = CreateFSCIFrame(framer, TX_OG, ON_OFF_CMD_OC, on_off_cmd_buf, sizeof(on_off_cmd_buf), VIF);
		    SendFrame(framer, send_on_off_cmd);
		    sleep(2); 
#endif
        }
    }
}