/*
 * Copyright 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#define _GNU_SOURCE
/*=========================================================================

Include Files

===========================================================================*/
#include <arpa/inet.h>
#include <assert.h>
#include <linux/if_tun.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>   
#include <unistd.h>

// netlink includes
#include <asm/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <net/if.h>

#include "Framer.h"
#include "FSCIFrame.h"
#include "PhysicalDevice.h"
#include "UARTConfiguration.h"
#include "Thread_HSDK.h"

/*=========================================================================

Public type definitions

===========================================================================*/
#define URI_PATH_TEMP               "/temp"
#define URI_PATH_LED                "/led"
#define URI_PATH_PAYLOAD_TOGGLE     "toggle"

#define USE_IFCONFIG                1
#define USE_GET_NODES_IP            1
#define USE_PING                    1
#define USE_COAP                    1
#define USE_SOCKET                  1

/*=========================================================================

Public global variables declarations

===========================================================================*/
static bool_t coapInstAlreadyCreated_g = FALSE;
static volatile bool_t stopProgram_g = FALSE;
static volatile uint8_t nodeHasJoined_g = FALSE;  
static char nodeIpStr[INET6_ADDRSTRLEN];
static uint8_t PAN_ID[] = {0xFA, 0xCE}; 

/* FSCI standard payloads */
static uint8_t create_buf[]             = {THR_INST_ID};
static uint8_t set_pskd_buf[]           = {THR_INST_ID, 0x19, 0x00, 0x06, 'T', 'H', 'R', 'E', 'A', 'D'};
static uint8_t set_ch_buf[]             = {THR_INST_ID, 0x04, 0x00, 0x01, 0x00};
static uint8_t get_steering_buf[]       = {THR_INST_ID, 0x31, 0x00};
static uint8_t sync_steering_buf[]      = {THR_INST_ID, 0x01};
static uint8_t add_joiner_default_buf[] = {THR_INST_ID, 0x01, 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x06,
                                           'T', 'H', 'R', 'E', 'A', 'D'
                                          };
static uint8_t ping_buf[37]             = { 0 };    /* placeholder, to be completed later */
static uint8_t getnodesip_buf[20]       = {THR_INST_ID, IN6ADDR_REALMLOCAL_ALLNODES_INIT, 0x02,
                                           gDiagnosticTlv_ShortAddr_c, gDiagnosticTlv_IPv6AddrList_c
                                          };

/* FSCI socket frame payloads */
static uint8_t create_socket_buf[]      = {0x0A, 0x00, 0x11};
static uint8_t bind_socket_buf[]        = {0x07, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB8,
                                           0x0D, 0x00, 0xFD, 0xD1, 0x04, 0x0A
                                          };
static uint8_t connect_socket_buf[]     = {0x07, 0x01, 0x00, 0x00, 0xFE, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB8,
                                           0x0D, 0x00, 0xFD, 0xD2, 0x04, 0x0A
                                          };
static uint8_t send_socket_buf[]        = {0x07, 0x40, 0x04, 0x00, 0x31, 0x32, 0x33, 0x34};
static uint8_t receive_socket_buf[]     = {0x07, 0x04, 0x00, 0x40};
   

/*! ************************************************************************
\fn int GetBaudrate(int value)

\brief  converts baudrate value to enumeration.

\param      value   selected baudrate value by the user

\retval     int     enumaration baudrate value
 ************************************************************************* */
static int GetBaudrate(int value)
{

    if (value == 115200)
        return BR115200;

    return -1;
}

/*! ************************************************************************
\fn void SigHandler(int signo)

\brief  Ctrl-C signal handler, only used to stop program. 
 ************************************************************************* */
static void SigHandler(int signo)
{
    if (signo == SIGINT) {
        stopProgram_g = TRUE;
    }
}

/*! ************************************************************************
\fn void ReverseByteArray(uint8_t *buf, int size)

\brief  This function reverses an array positions. 
        [0x11, 0x22, 0x33, 0x44] -> [0x44, 0x33, 0x22, 0x11]

\param      buf     pointer to the buffer that wants to be reversed.
\param      size    size of the buffer.
 ************************************************************************* */
static void ReverseByteArray(uint8_t *buf, int size)
{
    int i;
    uint8_t *temp = malloc(size);

    memcpy(temp, buf, size);

    for (i = 0; i < size; i++) {
        buf[i] = temp[size - i - 1];
    }

    free(temp);
}

/*! ************************************************************************
\fn void FrameReceived(void *callee, void *response)

\brief  Executes on every RX packet.

\param      response    data with the received frame. 
 ************************************************************************* */
static void FrameReceived(void *callee, void *response)
{
    int16_t i,j,k;
    FSCIFrame *frame = (FSCIFrame *)response;               /* Cast the received frame */

    /* used in ifconfig */
    char ip6str[INET6_ADDRSTRLEN];
    uint8_t temp_ip6[16] = { 0 };

    if (frame->opGroup != THR_RX_OG && frame->opGroup != MWS_RX_OG) {
        DestroyFSCIFrame(frame);
        return;
    }

    switch (frame->opCode) {

            /* Factory reset*/
        case FACTORY_RESET_OC:
            printf("RX: THR_FactoryReset.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> Success\n");
            }

            break;
            /* Reset indication */
        case CPU_RESET_IND_OC:
            printf("RX: THR_McuReset.Indication");

            if (frame->data[0] == 0x01) {
                printf(" -> ResetMcuPending\n");
            } else if (frame->data[0] == 0x00) {
                printf(" -> ResetMcuSuccess\n");
            }

            break;
            /* Received general event */
        case THR_EVT_GENERAL_CNF_OC:
            printf("RX: THR_EventGeneral.Confirm -> ");

            if (frame->data[1] == 0x01 && frame->data[2] == 0x00) {
                printf("Disconnected\n");
            } else if (frame->data[1] == 0x02 && frame->data[2] == 0x00) {  
                printf("Connected\n");                                      
            } else if (frame->data[1] == 0x03 && frame->data[2] == 0x00) {  
                printf("Reset to factory default\n");                       
                coapInstAlreadyCreated_g = FALSE;                             
            } else if (frame->data[1] == 0x0A && frame->data[2] == 0x00) {  
                printf("Device is Leader\n");                               
            } else if (frame->data[1] == 0x0B && frame->data[2] == 0x00) {  
                printf("Device is Router\n");                               
            } else if (frame->data[1] == 0x0C && frame->data[2] == 0x00) {  
                printf("Device is REED\n");                                 
            } else {
                printf("Status %02x\n", frame->data[1]);
            }

            break;
            /* Set network attribute */
        case THR_SET_ATTR_OC:
            printf("RX: THR_SetAttr.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> Success\n");
            }

            break;
            /* Create network response */
        case THR_CREATE_NWK_OC:
            printf("RX: THR_CreateNwk.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> OK\n");
            } else if (frame->data[0] == 0x0B) {
                printf(" -> Already Created\n");
            }

            break;
            /* Network create event */
        case THR_EVT_NWK_CREATE_CNF_OC:
            printf("RX: THR_EventNwkCreate.Confirm");

            if (frame->data[1] == 0x03 && frame->data[2] == 0x00) {
                printf(" -> Select best channel\n");
            } else if (frame->data[1] == 0x04 && frame->data[2] == 0x00) {
                printf(" -> Generate PSKc\n");
            } else if (frame->data[1] == 0x01 && frame->data[2] == 0x00) {
                printf(" -> Success\n");
            }

            break;
            /* Network commissioning event */
        case THR_EVT_NWK_COMM_IND_OC:
            if (frame->data[1] == 0x0D && frame->data[2] == 0x00) {
                /* avoid flooding the console with keep-alive */
                break;
            }

            printf("RX: THR_EventNwkCommissioning.Indication");

            if (frame->data[1] == 0x01 && frame->data[2] == 0x00) {
                printf(" -> Joiner discovery started\n");
            } else if (frame->data[1] == 0x0A && frame->data[2] == 0x00) {  
                printf(" -> Commissioner petition accepted\n");             
            } else if (frame->data[1] == 0x0F && frame->data[2] == 0x00) {  
                printf(" -> Commissioner Joiner DTLS session started\n");   
            } else if (frame->data[1] == 0x10 && frame->data[2] == 0x00) {  
                printf(" -> Joiner DTLS alert\n");                          
            } else if (frame->data[1] == 0x11 && frame->data[2] == 0x00) {  
                printf(" -> Commissioner Joiner accepted\n");               
            } else if (frame->data[1] == 0x1E && frame->data[2] == 0x00) {  
                printf(" -> Providing the security material to the Joiner\n"); 
            } else {
                printf(" -> %d\n", frame->data[1]);
            }

            break;
            /* Commissioning diagnostic response */
        case THR_EVT_COMM_DIAG_IND_OC:
            printf("RX: THR_CommissioningDiagnostic.Indication");

            if (frame->data[0] == 0x01) {
                printf(" -> IN");
            } else if (frame->data[0] == 0x00) {
                printf(" -> OUT");
            }

            if (frame->data[1] == 0x00) {
                printf(" JOIN_FIN_REQ\n");                                  /* New joiner request */
            } else if (frame->data[1] == 0x01) {
                printf(" JOIN_FIN_RSP\n");                                  /* Joiner accepted */
            } else if (frame->data[1] == 0x02) {
                printf(" JOIN_ENT_REQ\n");
            } else if (frame->data[1] == 0x03) {
                printf(" JOIN_ENT_RSP\n");
                nodeHasJoined_g = TRUE;                                       /* A device has been added to the network */
            } 

            break;
          
        case THR_MGMT_DIAG_GET_OC:
            printf("RX: THR_MgmtDiagnosticGet.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf(" -> FAIL!\n");
            }

            break;
            /* ifconfig command response */
        case IFCONFIG_OC:
            printf("RX: NWKU_IfconfigAll.Response -> ");
            int curr = 0;
            uint8_t CountInterfaces = frame->data[curr]; curr++;
            printf("CountInterfaces %d", CountInterfaces);

            for (i = 0; i < CountInterfaces; i++) {

                if (curr == frame->length) {
                    break;
                }

                uint8_t InterfaceID = frame->data[curr]; curr++;
                printf("\n\nInterface ID %d ", InterfaceID);
                uint8_t CountIpAddresses = frame->data[curr]; curr++;
                printf("\nIp Addresses:\n");

                for (j = 0; j < (CountIpAddresses-2); j++) {

                    for (k = 0; k < 16; k++) {
                        temp_ip6[k] = frame->data[curr + 15 - k];
                    }

                    curr += 16;

                    inet_ntop(AF_INET6, temp_ip6, ip6str, INET6_ADDRSTRLEN);
                    printf("\t%s\n", ip6str);
                }
            }

            break;
            /* getnodesip response */
        case THR_MGMT_DIAG_GET_IND_OC:
            printf("RX: THR_MgmtDiagnosticGetRsp.Indication");

            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf(" -> FAIL!\n");
                break;
            }

            uint16_t dataLen = frame->data[3] + (frame->data[4] << 8);
            uint16_t offset = 0;

            while (dataLen > 0) {
                uint8_t tlvType, tlvLength;

                tlvType = frame->data[6 + offset];
                tlvLength = frame->data[7 + offset];
                /*
                printf("tlvType: %x\n",tlvType);
                printf("tlvLength: %x \n",tlvLength);
                */

                if (tlvType != gDiagnosticTlv_ShortAddr_c && tlvType != gDiagnosticTlv_IPv6AddrList_c) {
                    printf("TLV type %d not currently processed!\n", tlvType);
                }
        #if USE_GET_NODES_IP
                if (tlvType == gDiagnosticTlv_ShortAddr_c) {
                    uint16_t shortAddr = frame->data[9 + offset] + (frame->data[8 + offset] << 8);
                    printf("\tNode Short Address: %04X\n", shortAddr);
                }
        #endif
                if (tlvType == gDiagnosticTlv_IPv6AddrList_c) {

                    if (tlvLength % 16 != 0) {
                        printf("Invalid payload!\n");

                    } else {
                        for (i = 0; i < tlvLength / 16; i++) {
                            inet_ntop(AF_INET6, &frame->data[8 + offset + 16 * i], ip6str, INET6_ADDRSTRLEN);
                            inet_ntop(AF_INET6, &frame->data[8 + offset + 16 * i], nodeIpStr, INET6_ADDRSTRLEN);
                        #if USE_GET_NODES_IP
                            printf("\t\t%s\n", ip6str);
                        #endif
                        }
                    }
                }
                offset += (2 + tlvLength);
                dataLen -= (2 + tlvLength);
            }
            break;
            /* Add joiner to Thread network */
        case MESHCOP_ADD_JOINER_OC:
            printf("RX: MESHCOP_AddExpectedJoiner.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf(" -> FAIL!\n");
            }

            break;
            /* Sync steering */
        case MESHCOP_SYNC_STEERING_OC:
            printf("RX: MESHCOP_SyncSteeringData.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf(" -> FAIL!\n");
            }

            break;
            /* ping command response */
        case NWKU_PING_OC:
            if (frame->data[0] == 0x00) {
                uint16_t time = frame->data[1] + (frame->data[2] << 8);
                printf("Reply from %s: bytes=%d time=%dms\n", nodeIpStr, PING_PACKET_SIZE, time);
            } else {
                printf("Request timed out\n");
            }

            break;
            /* Creation of CoAP instance response */
        case COAP_CREATE_INST_OC:
            if (frame->data[0] == 0x00) {
                printf("\nRX: NWKU_CoapCreateInstance.Confirm -> OK!\n");
            } else {
                printf("RX: NWKU_CoapCreateInstance.Confirm -> FAIL!\n");
            }

            break;
        case COAP_REGISTER_OC:
            if (frame->data[0] == 0x00) {
                printf("\nRX: NWKU_CoapRegister.Confirm -> OK!\n");
            } else {
                printf("RX: NWKU_CoapRegister.Confirm -> FAIL!\n");
            }
            break;
            /* Send CoAP message confirm */
        case COAP_SEND_OC:
            //printf("opGroup: %x \n",frame->opGroup);
            if (frame->data[0] != 0x00 && frame->opGroup == THR_RX_OG){
                printf("RX: NWKU_CoapSend.Confirm -> FAIL!\n");
            }else if (frame->data[0] == 0x00 && frame->opGroup == THR_RX_OG){
                printf("RX: NWKU_CoapSend.Confirm -> OK!\n");
            } 
                
            break;

            /* Receive CoAP messages */
        case COAP_MSG_RCVD_OC:
            printf("RX: NWKU_CoapMsgReceived.Indication\n");
            printf("\tStatus -> ");

            if (frame->data[0] == 0x00) {
                printf("Success\n");
            } else if (frame->data[0] == 0x01) {
                printf("Failed\n");
                break;
            } else if (frame->data[0] == 0x02) {
                printf("Duplicate\n");
                break;
            } else if(frame->data[0] == 0x2F){ /* 0x2F = "/" */
                printf("Receive CoAP host request!\n");

                break;
            }

            uint8_t payloadLength = frame->data[51];

            if (payloadLength) {
                char payload[0xFF] = { 0 };
                strncpy(payload, (const char *)&frame->data[52], payloadLength);
                printf("\tPayload -> %s\n", payload);
            }

            break;
            
        case MWS_REQ_ACCESS_OC:
        case MWS_REL_ACCESS_OC:
            if (frame->data[0] == 0x00) {
                printf("Line asserted!\n");
            } else {
                printf("Request denied!\n");
            }

            break;
            /* Get a network attribute */
        case THR_GET_ATTR_OC:
            if (frame->data[1] == 0x31) {
                if (frame->data[4] == 0x00) {
                    printf("RX: THR_GetAttr.Confirm SteeringData is NULL. Please issue `thr create` again!\n");
                } else {
                    printf("RX: THR_GetAttr.Confirm SteeringData -> OK!\n");
                }
            }

            break;
            /* Creation of a socket communication */
        case SOCKET_CREATE_OC:
            printf("RX: Socket-Create.Confirm\n");
            bind_socket_buf[0] = frame->data[0];        /* Copy Socket Index to bind buffer */
            connect_socket_buf[0] = frame->data[0];     /* Copy Socket Index to connect buffer */
            send_socket_buf[0] = frame->data[0];        /* Copy Socket Index to send buffer */
            receive_socket_buf[0] = frame->data[0];     /* Copy Socket Index to receive buffer */
            break;
            /* Socket bind response */
        case SOCKET_BIND_OC:
            printf("RX: Socket-Bind.Confirm");

            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf("-> ERROR!\n");
            }
            break;
            /* Socket connect response */
        case SOCKET_CONNECT_OC:
            printf("RX: Socket-Connect.Confirm");
            if (frame->data[0] == 0x00) {
                printf(" -> OK!\n");
            } else {
                printf("-> ERROR!\n");
            }
            break;
            /* Receive data from a node sent by socket */
        case SOCKET_RCV_FROM_OC:
            
            if (frame->length == 1) {                                                           /* Recevived empty payload */

            } else {
                printf("\nRX: Socket-Connect.Confirm\n");

                uint16_t dataLen = frame->data[19] + (frame->data[20] << 8);                    /* Get received data size */
                printf("\tPayload(%d) -> ",dataLen);                    
                for(i = 0; i < dataLen; i++)
                {   
                    printf("%c",frame->data[21 + i]);                                           /* Print payload data */
                }
                ReverseByteArray(frame->data,frame->length);                                  /* Reverse data frame to obtain IPv6 address */
                inet_ntop(AF_INET6, &frame->data[dataLen + 4], ip6str, INET6_ADDRSTRLEN);       /* Obtain IPv6 address */
                printf("\tFrom IPv6 Address: %s\n", ip6str);                                     
            }
            
            break;
            /* Unknown frame */
        default:
            printf("RX: unknown frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, frame->opCode);
            break;
    }

    DestroyFSCIFrame(frame);
}

/*! ************************************************************************
\fn void ConfigureNetworkAttributes(Framer *framer)

\brief  Set all general configuration attributes needed to create a Thread network
        as the PAN ID, key, ML prefix, etc.

\param      framer    pointer to the used framer. 
 ************************************************************************* */
static void ConfigureNetworkAttributes(Framer *framer)
{
    /* Network Key */
    uint8_t nwk_key[] = {0x00, 0x16, 0x00, 0x10,
                      0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
                     };
    FSCIFrame *set_nwk_key = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_key, sizeof(nwk_key), VIF);

    /* Network Name */
    uint8_t nwk_name[] = {0x00, 0x10, 0x00, 0x03,
                      'G', 'R', 'L'
                     };
    FSCIFrame *set_nwk_name = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_name, sizeof(nwk_name), VIF);

    /* PAN ID */
    uint8_t nwk_PID[] = {0x00, 0x05, 0x00, 0x02,
                      PAN_ID[1], PAN_ID[0]
                     };
                     
    FSCIFrame *set_panid = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_PID, sizeof(nwk_PID), VIF);

    /* Extended PAN ID */
    uint8_t nwk_EPID[] = {0x00, 0x06, 0x00, 0x08,
                      0x00, 0x0D, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00
                     };
    FSCIFrame *set_epanid = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_EPID, sizeof(nwk_EPID), VIF);

    /* ML Prefix */
    uint8_t nwk_ML[] = {0x00, 0x1C, 0x00, 0x11,
                      0xFD, 0x00, 0x0D, 0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                      0x40
                     };
    FSCIFrame *set_ml_prefix = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_ML, sizeof(nwk_ML), VIF);

    /* PSKc */
    uint8_t nwk_PSKc[] = {0x00, 0x18, 0x00, 0x0F,
                      't', 'h', 'r', 'e', 'a', 'd', 'j', 'p', 'a', 'k', 'e', 't', 'e', 's', 't'
                     };
    FSCIFrame *set_pskc = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_PSKc, sizeof(nwk_PSKc), VIF);

    /* Security Policy Rotation Interval */
    uint8_t nwk_policy[] = {0x00, 0x20, 0x00, 0x04,
                      0x10, 0x0E, 0x00, 0x00
                     };
    FSCIFrame *set_policy = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC, nwk_policy, sizeof(nwk_policy), VIF);

    /* Set network key */
    printf("TX: THR_SetAttr.Request Network Key\n");
    SendFrame(framer, set_nwk_key); sleep(1);
    DestroyFSCIFrame(set_nwk_key);
    /* Set network name */
    printf("TX: THR_SetAttr.Request Network Name\n");
    SendFrame(framer, set_nwk_name); sleep(1);
    DestroyFSCIFrame(set_nwk_name);
    /* Set network PAN ID */
    printf("TX: THR_SetAttr.Request PAN ID\n");
    SendFrame(framer, set_panid); sleep(1);
    DestroyFSCIFrame(set_panid);
    /* Set network Extended PAN ID */
    printf("TX: THR_SetAttr.Request Extended PAN ID\n");
    SendFrame(framer, set_epanid); sleep(1);
    DestroyFSCIFrame(set_epanid);
    /* Set network ML Prefix */
    printf("TX: THR_SetAttr.Request ML Prefix\n");
    SendFrame(framer, set_ml_prefix); sleep(1);
    DestroyFSCIFrame(set_ml_prefix);
    /* Set PSKc */
    printf("TX: THR_SetAttr.Request PSKc\n");
    SendFrame(framer, set_pskc); sleep(6);
    DestroyFSCIFrame(set_pskc);
    /* Set network policy security */
    printf("TX: THR_SetAttr.Request Security Policy Rotation Interval\n");
    SendFrame(framer, set_policy); sleep(1);
    DestroyFSCIFrame(set_policy);
}

/*! ************************************************************************
\fn void NWKU_CoapSendRequest(NWKU_CoapSendRequest_t *req, Framer *framer)

\brief  encapsulates, serializes and sends the CoAP message.

\param      req         pointer to the CoAP message configured in CoapHandler function.
\param      framer      pointer to the used framer. 
 ************************************************************************* */
static void NWKU_CoapSendRequest(NWKU_CoapSendRequest_t *req, Framer *framer)
{
    FSCIFrame *coap_req;
    uint8_t *pMsg = NULL;
    uint16_t msgLen = 0, idx = 0;

    /* Sanity check */
    if (!req) {
        return;
    }

    /* Compute the size of the request */
    msgLen += sizeof(uint8_t);                      /* InstanceID */
    msgLen += sizeof(req->DestinationIpAddress);    /* DestinationIpAddress */
    msgLen += sizeof(uint16_t);                     /* UDPPort */
    msgLen += sizeof(uint8_t);                      /* RequestType */
    msgLen += sizeof(uint8_t);                      /* MessageType */
    msgLen += 30;                                   /* URIpath */
    msgLen += sizeof(uint8_t);                      /* PayloadLength */
    msgLen += req->PayloadLength;                   /* Payload */

    /* Allocate memory for the marshalled payload */
    pMsg = malloc(msgLen);

    if (!pMsg) {
        return;
    }

    /* Serialize */
    pMsg[idx] = req->InstanceID; idx++;
    memcpy(pMsg + idx, req->DestinationIpAddress, sizeof(req->DestinationIpAddress)); idx += sizeof(req->DestinationIpAddress);
    memcpy(pMsg + idx, &(req->UDPPort), sizeof(req->UDPPort)); idx += sizeof(req->UDPPort);
    pMsg[idx] = req->RequestType; idx++;
    pMsg[idx] = req->MessageType; idx++;
    memcpy(pMsg + idx, req->URIpath, 30); idx += 30;
    pMsg[idx] = req->PayloadLength; idx++;
    memcpy(pMsg + idx, req->Payload, req->PayloadLength); idx += req->PayloadLength;

    /* Send the request */
    coap_req = CreateFSCIFrame(framer, TX_OG, COAP_SEND_OC, pMsg, msgLen, VIF);
    SendFrame(framer, coap_req);
    free(req->Payload);
    DestroyFSCIFrame(coap_req);
}

/*! ************************************************************************
\fn void CoapCreateInstance(Framer *framer)

\brief  Creates a CoAP instance.

\param      framer    pointer to the used framer. 
 ************************************************************************* */
static void CoapCreateInstance(Framer *framer)
{
    if (!coapInstAlreadyCreated_g) {
        uint8_t NWKU_CoapCreateInstanceReq[3] = {COAP_DEFAULT_PORT & 0x00FF, COAP_DEFAULT_PORT >> 8, AF_INET6};
        FSCIFrame *coap_create_instance_req = CreateFSCIFrame(framer, TX_OG, COAP_CREATE_INST_OC, NWKU_CoapCreateInstanceReq,
                                              sizeof(NWKU_CoapCreateInstanceReq), VIF);
        SendFrame(framer, coap_create_instance_req); sleep(1);
        DestroyFSCIFrame(coap_create_instance_req);
        coapInstAlreadyCreated_g = TRUE;
    }
}

/*! ************************************************************************
\fn void CoapHandler(Framer *framer, NWKU_CoapSendRequest_RequestType_t requestType, NWKU_CoapSendRequest_MessageType_t messageType)

\brief  Configures the CoAP message that wants to be sent.

\param      framer          pointer to the used framer. 
\param      requestType     selects a confirmable or a non-confirmable request type.
\param      messageType     selects the message type (GET or POST).
 ************************************************************************* */
static void CoapHandler(Framer *framer, NWKU_CoapSendRequest_RequestType_t requestType, NWKU_CoapSendRequest_MessageType_t messageType)
{
    int i;
    NWKU_CoapSendRequest_t req = { 0 };

    /* Select the request type of CoAP message */
    if(NWKU_CoapSendRequest_RequestType_CON == requestType){
        req.RequestType = NWKU_CoapSendRequest_RequestType_CON;                     /* Set confirmable request (with ACKNOWLEDGE)*/
    }else{                                                                          
        req.RequestType = NWKU_CoapSendRequest_RequestType_NON;                     /* Set non-confirmable request (with no ACKNOWLEDGE) */
    }

    /* Select the message type of CoAP message */
    switch(messageType)
    {
        case NWKU_CoapSendRequest_MessageType_GET:
            req.MessageType = NWKU_CoapSendRequest_MessageType_GET;                 /* Assign message GET */
            memcpy(req.URIpath, URI_PATH_TEMP, strlen(URI_PATH_TEMP));              /* Add URI path "/temp" */
        break;
        case NWKU_CoapSendRequest_MessageType_POST:
            req.MessageType = NWKU_CoapSendRequest_MessageType_POST;                /* Assign message POST */
            memcpy(req.URIpath, URI_PATH_LED, strlen(URI_PATH_LED));                /* Add URI path "/led"*/
            req.PayloadLength = strlen(URI_PATH_PAYLOAD_TOGGLE);                    /* Assign payload length */
            req.Payload = calloc(req.PayloadLength, sizeof(uint8_t));               /* Request payload memory */
            memcpy(req.Payload, URI_PATH_PAYLOAD_TOGGLE, req.PayloadLength);        /* Assign payload "toggle" */
        break;
        default:
        break;
    }

    uint8_t temp_addr[16];
    if(inet_pton(AF_INET6, nodeIpStr, temp_addr) == 1){

        /* Reverse IPv6 bytes */
        for (i = 0; i < 16; i++) {
            req.DestinationIpAddress[i] = temp_addr[15 - i];                        /* Set ip address */
        }
    }

    CoapCreateInstance(framer);
    sleep(3);
    NWKU_CoapSendRequest(&req, framer);
}

int main(int argc, char **argv)
{
    /* Check number of arguments. */
    if (argc < 4) {
        printf("Usage UART: # %s {/dev/ttyACMx | /dev/ttymxcx} channel [baudrate bps]\n", argv[0]);
        exit(1);
    }

    uint8_t i;
    uint8_t dest_ip6[16] = { 0 };

    /* Begin UART configuration */
    DeviceType dev_type = UART;
    void *serial_config = defaultConfigurationData();
    int baudrate = GetBaudrate(atoi(argv[3]));

    if (baudrate == -1) {
        printf("Wrong baudrate value.\n");
        exit(1);
    } else {
        setBaudrate(serial_config, baudrate);
    }
    /* End UART configuration */

    /* Get selected channel */
    uint8_t channel = atoi(argv[2]);
    assert(channel >= 11 && channel <= 26);
    set_ch_buf[4] = channel;

    PhysicalDevice *device = InitPhysicalDevice(dev_type, serial_config, argv[1], GLOBAL);              /* Initialize device with UART configuration and the selected port */
    Framer *framer = InitializeFramer(device, FSCI, LENGTH_FIELD_SIZE, CRC_FIELD_SIZE, _LITTLE_ENDIAN); /* Configure framer */
    OpenPhysicalDevice(device);                                                                         /* Open kinetis device */
    AttachToFramer(framer, NULL, FrameReceived);                                                        /* Set callback */

    /* Frames */
    FSCIFrame *factory_rst          = CreateFSCIFrame(framer, TX_OG, FACTORY_RESET_OC,          NULL, 0, VIF);
    FSCIFrame *set_channel          = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC,           set_ch_buf, sizeof(set_ch_buf), VIF);
    FSCIFrame *set_pskd             = CreateFSCIFrame(framer, TX_OG, THR_SET_ATTR_OC,           set_pskd_buf, sizeof(set_pskd_buf), VIF);
    FSCIFrame *create_nwk           = CreateFSCIFrame(framer, TX_OG, THR_CREATE_NWK_OC,         create_buf, sizeof(create_buf), VIF);
    FSCIFrame *add_joiner           = CreateFSCIFrame(framer, TX_OG, MESHCOP_ADD_JOINER_OC,     add_joiner_default_buf, sizeof(add_joiner_default_buf), VIF);
    FSCIFrame *sync_steering_data   = CreateFSCIFrame(framer, TX_OG, MESHCOP_SYNC_STEERING_OC,  sync_steering_buf, sizeof(sync_steering_buf), VIF);
    FSCIFrame *get_steering_data    = CreateFSCIFrame(framer, TX_OG, THR_GET_ATTR_OC,           get_steering_buf, sizeof(get_steering_buf), VIF);
    FSCIFrame *getnodesip           = CreateFSCIFrame(framer, TX_OG, THR_MGMT_DIAG_GET_OC,      getnodesip_buf, sizeof(getnodesip_buf), VIF);
    FSCIFrame *create_socket        = CreateFSCIFrame(framer, TX_OG, SOCKET_CREATE_OC,          create_socket_buf, sizeof(create_socket_buf), VIF);
    FSCIFrame *send_data_socket     = CreateFSCIFrame(framer, TX_OG, SOCKET_SEND_OC,            send_socket_buf, sizeof(send_socket_buf), VIF); 
    FSCIFrame *rcv_from_socket      = CreateFSCIFrame(framer, TX_OG, SOCKET_RCV_FROM_OC,        receive_socket_buf, sizeof(receive_socket_buf), VIF);
    FSCIFrame *bind_socket; 
    FSCIFrame *connect_socket;
    FSCIFrame *ping;

    /* Factory reset */
    printf("TX: THR_FactoryReset.Request\n");
    SendFrame(framer, factory_rst);
    sleep(1);

    /* Network characteristics */
    ConfigureNetworkAttributes(framer);
    sleep(1);

    /* Begin: Network creation */
    /* Set channel */
    printf("TX: THR_SetAttr.Request Channel %d\n", channel);
    SendFrame(framer, set_channel); 
    sleep(1);
    /* Set security key: THREAD */
    printf("TX: THR_SetAttr.Request PSKd `THREAD`\n");
    SendFrame(framer, set_pskd); 
    sleep(1);
    /* Create network */
    printf("TX: THR_CreateNwk.Request\n");
    SendFrame(framer, create_nwk); 
    sleep(12);
    /* Add expected joiners */
    printf("TX: MESHCOP_AddExpectedJoiner.Request\n");
    SendFrame(framer, add_joiner); 
    sleep(1);
    /* Sync steering data */
    printf("TX: MESHCOP_SyncSteeringData.Request\n");
    SendFrame(framer, sync_steering_data); 
    sleep(1);
    /* Get steering data */
    printf("TX: THR_GetAttr.Request Steering Data\n");
    SendFrame(framer, get_steering_data);
    sleep(1);
    /* End: Network creation */

#if USE_IFCONFIG
    FSCIFrame *ifconfig = CreateFSCIFrame(framer, TX_OG, IFCONFIG_OC,  NULL, 0, VIF);
    printf("TX: NWKU_IfconfigAll.Request\n");
    SendFrame(framer, ifconfig); 
    sleep(1);
    printf("PAN ID:  0x%02x%02x\n",PAN_ID[0],PAN_ID[1]);
    printf("Channel: %d\n\n",channel);
 #endif

    stopProgram_g = FALSE;

    while(!stopProgram_g) 
    {
        /* Wait until a node joins to the Thread network */
        if(nodeHasJoined_g)
        {
            nodeHasJoined_g = FALSE;

            /* Print node data */
            sleep(3);
            printf("\nTX: THR_MgmtDiagnosticGet.Request\n");
            SendFrame(framer, getnodesip); 
            sleep(3);
#if USE_PING
            /* Begin ping */
            if (inet_pton(AF_INET6, nodeIpStr, dest_ip6) == 1) {
                for (i = 0; i < 16; i++) {
                    ping_buf[i] = dest_ip6[15 - i];
                }

                ping_buf[32] = PING_PACKET_SIZE & 0xFF; /* payload length */
                ping_buf[33] = PING_PACKET_SIZE >> 8;

                ping_buf[34] = 0xD0; /* timeout */
                ping_buf[35] = 0x07; /* timeout (0x07D0 = 2 secs) */
                ping_buf[36] = 0x01; /* secured */

                ping = CreateFSCIFrame(framer, TX_OG, NWKU_PING_OC, ping_buf, sizeof(ping_buf), VIF);
                printf("\nPing to %s with %d bytes of data\n", nodeIpStr, PING_PACKET_SIZE);
                SendFrame(framer, ping);
                sleep(4);
            }
            /* End ping */
#endif
#if USE_COAP
            /* Begin CoAP message */
            printf("TX: Begin CoAP messages \n");
            printf("TX: Send CoAP message LED toggle with no ACK\n");
            CoapHandler(framer, NWKU_CoapSendRequest_RequestType_NON, NWKU_CoapSendRequest_MessageType_POST);  /* Send CoAP message (/led toggle) with no ACK */
            sleep(2);
            printf("TX: Send CoAP message temp with ACK\n");
            CoapHandler(framer, NWKU_CoapSendRequest_RequestType_CON, NWKU_CoapSendRequest_MessageType_GET);   /* Send CoAP message (/temp) with ACK */
            sleep(2);
            /* End CoAP message */
#endif
#if USE_SOCKET
            /* Begin Socket */
            printf("\nTX: Socket-Create.Request\n");
            SendFrame(framer, create_socket);

            bind_socket = CreateFSCIFrame(framer, TX_OG, SOCKET_BIND_OC, bind_socket_buf, sizeof(bind_socket_buf), VIF);
            printf("TX: Socket-Bind.Request\n");
            SendFrame(framer, bind_socket);

            connect_socket = CreateFSCIFrame(framer, TX_OG, SOCKET_CONNECT_OC, connect_socket_buf, sizeof(connect_socket_buf), VIF);
            printf("TX: Socket-Connect.Request\n");
            SendFrame(framer, connect_socket);

            printf("TX: Socket-Send.Request\n");
            SendFrame(framer, send_data_socket);
            /* End Socket */
#endif
        } 

        /* Waits to receive any socket message */
        sleep(1);
        SendFrame(framer, rcv_from_socket);
    }

    return 0;
}