/*
 * Copyright 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#define _GNU_SOURCE
/*=========================================================================

Include Files

===========================================================================*/
#include <pthread.h>
#include <arpa/inet.h>
#include <assert.h>
#include <linux/if_tun.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>   
#include <unistd.h>

// netlink includes
#include <asm/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <net/if.h>

#include "Framer.h"
#include "FSCIFrame.h"
#include "PhysicalDevice.h"
#include "UARTConfiguration.h"
#include "HRS_BLE_HSDK.h"

/*=========================================================================

Public type definitions

===========================================================================*/
typedef enum Uuid_service
{
    heart_rate_service,
    general_service,
    battery_service 
}Uuid_service_t;

/* Finite state machine program status */
typedef enum Connection_Status
{
	setting_parameters,
    start_advertising_service,
	waiting_connection,
	device_connected,
	device_disconnected
}Connection_Status_s;


volatile static Uuid_service_t actual_service_g = general_service;
static uint8_t handle_service[4] = {0x00, 0x00, 0x00, 0x00};/* First two bytes reserved for heart rate service handle
                                                                Last two bytes reserved for battery service handle*/
static uint8_t hr_batter_char_handle[2] = {0x00, 0x00};     /* This buffer will contain heart rate and battery characteristic handle value */
static Connection_Status_s demo_HRs_State;
static uint16_t mLatestHandle[2] = {0x00, 0x00};  		/*  Saves the latest service handle requested to the GATTDB */

/* Arrays to save the characteristics service handle's */
static uint8_t Handle_Write_Notification[]              = {0x01, 0x00, 0x00};
static uint8_t HeartRate_Service_Handle[2]              = {0x00, 0x00};
static uint8_t Battery_Service_Handle[2]                = {0x00, 0x00};
static uint8_t HeartRate_Measurement_Handle[2]          = {0x00, 0x00};
static uint8_t BodySensor_Location_Handle[2]            = {0x00, 0x00};
static uint8_t Battery_Level_Handle[2]                  = {0x00, 0x00};


 static uint8_t counter = 0;

/** FSCI standard payloads buffers**/

/* GATT Service */
static uint8_t add_gatt_psd_buf[]                       = {Uuid16Bits, 0x01, 0x18}; /* Generic Attribute Profile */
static uint8_t add_gatt_cdv_buf[]                       = {Uuid16Bits, 0x05, 0x2A , (gNotify_c | gRead_c), 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, gPermissionNone_c};

/* GAP Service */
static uint8_t add_gap_psd_buf[]                        = {Uuid16Bits, 0x00, 0x18}; /* Generic Acccess Profile */
static uint8_t gap_device_name_cdv_buf[]                = {Uuid16Bits, 0x00, 0x2A , gRead_c, 0x14, 0x00, 0x0C, 0x00, 'K', 'i', 'n', 'e', 't', 'i', 's', ' ', 'B', 'L', 'E', '\0', gPermissionFlagReadable_c};

static uint8_t gap_appearance_cdv_buf[]                 = {Uuid16Bits, 0x01, 0x2A , gRead_c, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, gPermissionFlagReadable_c};
static uint8_t gap_ppcp_cdv_buf[]                       = {Uuid16Bits, 0x04, 0x2A , gRead_c, 0x00, 0x00, 0x08, 0x00, 0x0A, 0x00, 0x10, 0x00, 0x64, 0x00, 0xE2, 0x04, gPermissionFlagReadable_c};

/* Heart Rate Service */
static uint8_t add_hr_psd_buf[]                         = {Uuid16Bits, 0x0D, 0x18};
static uint8_t hr_measure_cdv_buf[]                     = {Uuid16Bits, 0x37, 0x2A , (gNotify_c), 0x00, 0x00, 0x02, 0x00, 0x00, 0xB4, gPermissionNone_c};
static uint8_t hr_sensor_location_cdv_buf[]             = {Uuid16Bits, 0x38, 0x2A , gRead_c, 0x00, 0x00, 0x01, 0x00, 0x01, gPermissionFlagReadable_c};

static uint8_t hr_control_pint_cdv_buf[]                = {Uuid16Bits, 0x39, 0x2A , gWrite_c, 0x00, 0x00, 0x01, 0x00, 0x00, gPermissionFlagWritable_c};

/* Battery Service */
static uint8_t add_battery_psd_buf[]                    = {Uuid16Bits, 0x0F, 0x18};
static uint8_t battery_level_cdv_buf[]                  = {Uuid16Bits, 0x19, 0x2A , (gNotify_c | gRead_c), 0x00, 0x00, 0x01, 0x00, 0x5A, gPermissionFlagReadable_c};
static uint8_t char_pres_format_cd_buf[]                = {Uuid16Bits, 0x04, 0x29 , 0x07, 0x00, 0x04, 0x00, 0xAD, 0x27, 0x01, 0x00, 0x00, gPermissionFlagReadable_c};


/* Device Information Service */
static uint8_t add_device_info_psd_buf[]                = {Uuid16Bits, 0x0A, 0x18};

static uint8_t manufacturer_name_cdv_buf[]              = {Uuid16Bits, 0x29, 0x2A , gRead_c, 0x00, 0x00, 0x04, 0x00, 'N', 'X', 'P', '\0', gPermissionFlagReadable_c}; 
static uint8_t model_number_cdv_buf[]                   = {Uuid16Bits, 0x24, 0x2A , gRead_c, 0x00, 0x00, 0x0C, 0x00, 'K', 'i', 'n', 'e', 't', 'i', 's', ' ', 'B', 'L', 'E', '\0', gPermissionFlagReadable_c};
static uint8_t serial_number_cdv_buf[]                  = {Uuid16Bits, 0x25, 0x2A , gRead_c, 0x00, 0x00, 0x08, 0x00, 'B', 'L', 'E', 'S', 'N', '0', '1', '\0', gPermissionFlagReadable_c};
static uint8_t hardware_revision_cdv_buf[]              = {Uuid16Bits, 0x27, 0x2A , gRead_c, 0x00, 0x00, 0x0B, 0x00, 'F', 'R', 'D', 'M', '-', 'K', 'W', '4', '1', 'Z', '\0', gPermissionFlagReadable_c};
static uint8_t firmware_revision_cdv_buf[]              = {Uuid16Bits, 0x26, 0x2A , gRead_c, 0x00, 0x00, 0x06, 0x00, '1', '.', '1', '.', '1', '\0', gPermissionFlagReadable_c};
static uint8_t software_revision_cdv_buf[]              = {Uuid16Bits, 0x28, 0x2A , gRead_c, 0x00, 0x00, 0x06, 0x00, '1', '.', '1', '.', '2', '\0', gPermissionFlagReadable_c};
static uint8_t system_id_cdv_buf[]                      = {Uuid16Bits, 0x23, 0x2A , gRead_c, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x9F, 0x04, 0x00, gPermissionFlagReadable_c};
static uint8_t ieee_rcdl_cdv_buf[]                      = {Uuid16Bits, 0x2A, 0x2A , gRead_c, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, gPermissionFlagReadable_c};


/* Advertising data */
static uint8_t set_adv_data_buf[]                       = {gAdDataIncluded_c, 0x03, 0x01, gAdFlags_c, 0x06, 0x02, gAdIncomplete16bitServiceList_c, 0x0D, 0x18, 0x07, gAdShortenedLocalName_c, 'N', 'X', 'P', '_', 'H', 'R', 'S', 0x00};
static uint8_t hrs_gatt_notify[]                        = {0x00, 0x00};        
static uint8_t set_adv_param_buf[]                      = {gFastConnMinAdvInterval_c, 0x00, gFastConnMaxAdvInterval_c, 0x00, gAdvConnectableUndirected_c, gBleAddrTypePublic_c, gBleAddrTypePublic_c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, gGapAdvertisingChannelMapDefault_c, gProcessAll_c};
static uint8_t gatt_Send_Notification_Request_buf[]     = {gDeviceId_c, 0x00, 0x00};

/* Buffers for service handles */
static uint8_t batteryService_Handle[]                  = {0x01, 0x00, Uuid16Bits, 0x0F, 0x18};
static uint8_t find_hr_serviceHandleReq[]               = {0x01, 0x00, Uuid16Bits, 0x0D, 0x18};
static uint8_t find_hr_measurementHandleReq[]           = {0x00, 0x00, Uuid16Bits, 0x37, 0x2A};
static uint8_t find_body_sensorHandleReq[]              = {0x00, 0x00, Uuid16Bits, 0x38, 0x2A};
static uint8_t find_battery_levelHandleReq[]            = {0x00, 0x00, Uuid16Bits, 0x19, 0x2A};

/* Buffers to update measurement values */
static uint8_t HR_value = ((uint8_t)gHrs_EnergyExpendedEnabled_c | (uint8_t)gHrs_SensorContactDetected_c | (uint8_t)gHrs_SensorContactSupported_c);
static uint8_t update_HeartRate_Measurement[]           = {0x00, 0x00, 0x02, 0x00, 0x00, 0x00};
static uint8_t update_Battery_Measurement[]             = {0x00, 0x00, 0x01, 0x00, 0x00};

/*! ************************************************************************
\fn int GetBaudrate(int value)

\brief  converts baudrate value to enumeration.

\param      value   selected baudrate value by the user

\retval     int     enumaration baudrate value
 ************************************************************************* */
static int GetBaudrate(int value)
{

    if (value == 115200)
        return BR115200;

    return -1;
}

/*! ************************************************************************
\fn void ReverseByteArray(uint8_t *buf, int size)

\brief  This function reverses an array positions. 
        [0x11, 0x22, 0x33, 0x44] -> [0x44, 0x33, 0x22, 0x11]

\param      buf     pointer to the buffer that wants to be reversed.
\param      size    size of the buffer.
 ************************************************************************* */
static void ReverseByteArray(uint8_t *buf, int size)
{
    int i;
    uint8_t *temp = malloc(size);

    memcpy(temp, buf, size);

    for (i = 0; i < size; i++) {
        buf[i] = temp[size - i - 1];
    }

    free(temp);
}

/*! ************************************************************************
\fn void FrameReceived(void *callee, void *response)

\brief  Executes on every RX packet.

\param      response    data with the received frame. 
 ************************************************************************* */
static void FrameReceived(void *callee, void *response)
{
    FSCIFrame *frame = (FSCIFrame *)response;               /* Cast the received frame */

    if ((frame->opGroup != FSCI_REQ_OG) && (frame->opGroup != FSCI_RSP_IND_CNF_OG) && (frame->opGroup != L2CAP_OG) && (frame->opGroup != GATT_OG) && (frame->opGroup != GATTDB_OG) && (frame->opGroup != GAP_OG)) {
        printf("DestroyFSCIFrame\n");
        DestroyFSCIFrame(frame);
        return;
    }

    uint8_t i;

    switch (frame->opCode) {

        case CONFIRMATION_OC: /* Generic confirmation */
            if(demo_HRs_State != (device_connected))
            {

                if(frame->opGroup == FSCI_RSP_IND_CNF_OG){
                    printf("RX: FSCI.Confirm->");
                }
                else if(frame->opGroup == L2CAP_OG){
                    printf("RX: L2CAP.Confirm->");   
                }
                else if(frame->opGroup == GATT_OG){
                    printf("RX: GATT.Confirm->");   
                }
                else if(frame->opGroup == GATTDB_OG){
                    printf("RX: GATTDB.Confirm->");
                }
                else if(frame->opGroup == GAP_OG){
                    printf("RX: GAP.Confirm->");
                }

                if((frame->data[0] == 0x00) && (frame->data[1] == 0x00)){
                    printf("gBleSuccess\n");
                }
                else if ((frame->data[0] == 0x09) && (frame->data[1] == 0x01)){
                    printf("gGattDBInvalidHandle_c\n");
                }
                else if ((frame->data[0] == 0x05) && (frame->data[1] == 0x01)){
                    printf("gAttInvalidHandle_c\n");
                }
                else{
                    printf("error (%02x%02x)\n", frame->data[1], frame->data[0]);
                }
                /* Check this errors on ble_general.h file*/
            }  
              
        break;

        case ADD_PSD_IND_OC:
            printf("RX: GATTDBDynamic-AddPrimaryServiceDeclaration.Indication\n");
            printf("\tService Handle: %02x%02x\n", frame->data[1], frame->data[0]);
            switch(actual_service_g)
            {
                case heart_rate_service: 
                    handle_service[heart_rate_service] = frame->data[1];
                    handle_service[heart_rate_service + 1] = frame->data[0];
                break;

                case battery_service: 
                    handle_service[battery_service] = frame->data[1];
                    handle_service[battery_service + 1] = frame->data[0];
                break;

                default:
                break;
            }
        break;

        case ADD_CDV_IND_OC: /* ADD_CDV_IND_OC and GAP_INIT_COMPLETE_OC has the same OpCode but different OpGroup */
            if(frame->opGroup == GAP_OG){    /* GAP_INIT_COMPLETE_OC */
                printf("RX: GAP-GenericEventInitializationComplete.Indication\n");
            }
            else if(frame->opGroup == GATTDB_OG){ /* ADD_CDV_IND_OC */
                printf("RX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Indication\n");
                printf("\tCharacteristic Handle: %02x%02x\n", frame->data[1], frame->data[0]);
            }
        break;

        case ADD_CCCD_IND_OC: /* ADD_CCCD_IND_OC and ADV_SETUP_FAILED_OC has the same OpCode but different OpGroup */
            if(frame->opGroup == GAP_OG){ /* ADV_SETUP_FAILED_OC */
                printf("RX: GAP-GenericEventAdvertisingSetupFailed.Indication\n");
                printf("\tSetup fail reason: ");
                if((frame->data[0] == 0x01) && (frame->data[1] == 0x0C)){
                    printf("gHciCommandDisallowed_c\n");
                }
                else{
                    printf("(%02x %02x)\n", frame->data[0], frame->data[1]);
                }
            }
            else if(frame->opGroup == GATTDB_OG){ /* ADD_CCCD_IND_OC */

            }
            printf("RX: GATTDBDynamic-AddCccd.Indication\n");
            printf("\tCccd Handle: %02x%02x\n", frame->data[1], frame->data[0]);
            mLatestHandle[0] = frame->data[0];
            mLatestHandle[1] = frame->data[1];
        break;

        case ADD_CD_IND_OC:
            printf("RX: GATTDBDynamic-AddCharacteristicDescriptor.Indication\n");
            printf("\tDescriptor Handle: %02x%02x\n", frame->data[1], frame->data[0]);
        break;

        case READ_PUBLIC_DEVICE_ADDR_IND_OC:
            if( demo_HRs_State != device_connected)
            {
                printf("RX: GAP-GenericEventPublicAddressRead.Indication\n");
                printf("\tAddress: ");
                for(i = 0; i < 6; i++){
                    printf("%02x", frame->data[i]);
                }
                printf("\n");
            }
        break;

        case ADV_DATA_SETUP_COMPLETE_OC:
            printf("RX: GAP-GenericEventAdvertisingDataSetupComplete.Indication\n");
        break;

        case FindServiceHandleIndication_FSCI_ID:
            if(demo_HRs_State != device_connected)
            {
                printf("RX: GATTDB-FindServiceHandleInService.Indication\n");
                printf("\tService Handle Indication: %02x%02x\n", frame->data[1], frame->data[0]);
            }
    		
			mLatestHandle[0] = frame->data[0];
			mLatestHandle[1] = frame->data[1];

        break;

        case FIND_CHAR_VALUE_HANDLE_SERVICE_IND_OC:
            printf("RX: GATTDB-FindCharacteristicValueHandleInService.Indication\n");
            printf("\tCharacteristic Value Handle: %02x%02x\n", frame->data[1], frame->data[0]);
            hr_batter_char_handle[0] = mLatestHandle[0] = frame->data[0];
			hr_batter_char_handle[0] = mLatestHandle[1] = frame->data[1];
        break;

        case FIND_CCCD_HANDLE_FOR_CHAR_HANDLE_IND_OC:
            if(demo_HRs_State != (device_connected))
            {
                printf("RX: GATTDB-FindCccdHandleForCharacteristicValueHandle.Indication\n");
                printf("\tCccd Handle: %02x%02x\n", frame->data[1], frame->data[0]); 
            }

            mLatestHandle[0] = frame->data[0];
            mLatestHandle[0] = frame->data[1];
        break;

        case ADV_PARAM_SETUP_COMPLETE_OC:
            printf("RX: GAP-GenericEventAdvertisingParametersSetupComplete.Indication\n");
        break;

        case ADV_STATE_CHANGED_OC:
            printf("RX: GAP-AdvertisingEventStateChanged.Indication\n");
        break;

        case ADV_CMD_FAILED_OC:
            printf("RX: AdvertisingEventCommandFailed.Indication\n");
            printf("\tFail reason: ");
            if((frame->data[0] == 0x01) && (frame->data[1] == 0x0C)){
                printf("gHciCommandDisallowed_c\n");
            }
            else{
                printf("(%02x %02x)\n", frame->data[0], frame->data[1]);
            }
        break;

        case Connection_Event_Connected_Indication:
            printf("RX: ConnectionEventConnected.Indication\n");
            sleep(4); 
            demo_HRs_State = device_connected;                           /* Change program status */
        break;    

        case Connection_Event_Pairing_Request:
            printf("RX: ConnectionEventPairingRequest.Indication\n");
        break;    

        case Connection_Event_Disconnected:
            printf("RX: ConnectionEventDisconnected.Indication\n");
            sleep(4);
            demo_HRs_State = device_disconnected;                         /* Change program status */
        break;    

        case FSCI_RSP_IND_CNF_OG:
            printf("\n\n----------------------NOT IMPLEMENTED----------------------------\n\n");
            printf("RX: unknown frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, frame->opCode);
            /*=========================================================================
            Here you can implement your own cases depending on your application behavior,
            just define the appropiate opGroups & opCodes to catch correctly the
            frames received.
            ===========================================================================*/
        break;

        case GATT_SERVER_ERROR:
            if(frame->opGroup == GATT_OG)
            {
                printf("RX: GATT_SERVER_ERROR->Inactive_Connection_Status\n");
                demo_HRs_State = device_disconnected;
            }
        break;

     default:
            printf("RX: unknown frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, frame->opCode);
            break;
    }

    DestroyFSCIFrame(frame);
}


int main(int argc, char **argv)
{
    /* Check number of arguments. */
    if (argc < 3) {
        printf("Usage UART: # %s {/dev/ttyACMx | /dev/ttymxcx} [baudrate bps]\n", argv[0]);
        exit(1);
    }

    /* Begin UART configuration */
    DeviceType dev_type = UART;
    void *serial_config = defaultConfigurationData();
    int baudrate = GetBaudrate(atoi(argv[2]));

    if (baudrate == -1) {
        printf("Wrong baudrate value.\n");
        exit(1);
    } else {
        setBaudrate(serial_config, baudrate);
    }
    /* End UART configuration */

    demo_HRs_State  = setting_parameters; /* Set initial program status */

    PhysicalDevice *device = InitPhysicalDevice(dev_type, serial_config, argv[1], GLOBAL);              /* Initialize device with UART configuration and the selected port */
    Framer *framer = InitializeFramer(device, FSCI, LENGTH_FIELD_SIZE, CRC_FIELD_SIZE, _LITTLE_ENDIAN); /* Configure framer */
    OpenPhysicalDevice(device);                                                                         /* Open kinetis device */
    AttachToFramer(framer, NULL, FrameReceived);                                                        /* Set callback */

    /*=========================================================================
            Creation of FSCI frames that will been sent to the board
     ======================================================================= */

    /** Frames **/
    FSCIFrame *factory_rst                      = CreateFSCIFrame(framer, FSCI_REQ_OG, CPU_RESET_OC,                NULL, 0, VIF);
    FSCIFrame *add_cccd                         = CreateFSCIFrame(framer, GATTDB_OG, ADD_CCCD_REQ_OC,               NULL, 0, VIF);
    FSCIFrame *register_callback                = CreateFSCIFrame(framer, GATT_OG, SERVER_REGISTER_CALLBACK,        NULL, 0, VIF);
    FSCIFrame *start_advertising                = CreateFSCIFrame(framer, GAP_OG, START_ADV_OC,                     NULL, 0, VIF);
    FSCIFrame *read_public_addr                 = CreateFSCIFrame(framer, GAP_OG, READ_PUBLIC_DEVICE_ADDR_REQ_OC,   NULL, 0, VIF);

    /* GATT FSCI commands */
    FSCIFrame *gatt_psd                         = CreateFSCIFrame(framer, GATTDB_OG, ADD_PSD_REQ_OC,          add_gatt_psd_buf, sizeof(add_gatt_psd_buf), VIF);
    FSCIFrame *gatt_service_changed_cdv         = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          add_gatt_cdv_buf, sizeof(add_gatt_cdv_buf), VIF);
    
    /* GAP FSCI commands */
    FSCIFrame *gap_psd                          = CreateFSCIFrame(framer, GATTDB_OG, ADD_PSD_REQ_OC,          add_gap_psd_buf, sizeof(add_gap_psd_buf), VIF);
    FSCIFrame *gap_device_name_cdv              = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          gap_device_name_cdv_buf, sizeof(gap_device_name_cdv_buf), VIF);
    FSCIFrame *gap_appeareance_cdv              = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          gap_appearance_cdv_buf, sizeof(gap_appearance_cdv_buf), VIF);
    FSCIFrame *gap_ppcp_cdv                     = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          gap_ppcp_cdv_buf, sizeof(gap_ppcp_cdv_buf), VIF);
            
    /* Heart Rate FSCI commands */
    FSCIFrame *hr_psd                           = CreateFSCIFrame(framer, GATTDB_OG, ADD_PSD_REQ_OC,          add_hr_psd_buf, sizeof(add_hr_psd_buf), VIF);
    FSCIFrame *hr_measurement_cdv               = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          hr_measure_cdv_buf, sizeof(hr_measure_cdv_buf), VIF);
    FSCIFrame *hr_sensor_location_cdv           = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          hr_sensor_location_cdv_buf, sizeof(hr_sensor_location_cdv_buf), VIF);
    FSCIFrame *hr_control_point_cdv             = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          hr_control_pint_cdv_buf, sizeof(hr_control_pint_cdv_buf), VIF);

    /* Battery FSCI commands */
    FSCIFrame *battery_psd                      = CreateFSCIFrame(framer, GATTDB_OG, ADD_PSD_REQ_OC,          add_battery_psd_buf, sizeof(add_battery_psd_buf), VIF);
    FSCIFrame *bat_level_cdv                    = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          battery_level_cdv_buf, sizeof(battery_level_cdv_buf), VIF);
    FSCIFrame *bat_char_pres_cd                 = CreateFSCIFrame(framer, GATTDB_OG, ADD_CD_REQ_OC,           char_pres_format_cd_buf, sizeof(char_pres_format_cd_buf), VIF);

    /* Device Information FSCI commands */
    FSCIFrame *device_psd                       = CreateFSCIFrame(framer, GATTDB_OG, ADD_PSD_REQ_OC,          add_device_info_psd_buf, sizeof(add_device_info_psd_buf), VIF);
    FSCIFrame *device_manufacturer_cdv          = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          manufacturer_name_cdv_buf, sizeof(manufacturer_name_cdv_buf), VIF);
    FSCIFrame *device_model_cdv                 = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          model_number_cdv_buf, sizeof(model_number_cdv_buf), VIF);
    FSCIFrame *device_serial_cdv                = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          serial_number_cdv_buf, sizeof(serial_number_cdv_buf), VIF);
    FSCIFrame *device_hw_cdv                    = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          hardware_revision_cdv_buf, sizeof(hardware_revision_cdv_buf), VIF);
    FSCIFrame *device_fw_cdv                    = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          firmware_revision_cdv_buf, sizeof(firmware_revision_cdv_buf), VIF);
    FSCIFrame *device_sw_cdv                    = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          software_revision_cdv_buf, sizeof(software_revision_cdv_buf), VIF);
    FSCIFrame *device_system_cdv                = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          system_id_cdv_buf, sizeof(system_id_cdv_buf), VIF);
    FSCIFrame *device_ieee_cdv                  = CreateFSCIFrame(framer, GATTDB_OG, ADD_CDV_REQ_OC,          ieee_rcdl_cdv_buf, sizeof(ieee_rcdl_cdv_buf), VIF);

    /* Set advetising data */
    FSCIFrame *set_adv_data                     = CreateFSCIFrame(framer, GAP_OG, SET_ADV_DATA,               set_adv_data_buf, sizeof(set_adv_data_buf), VIF);  
    FSCIFrame *set_adv_parameters               = CreateFSCIFrame(framer, GAP_OG, SET_ADV_PARAM_OC,           set_adv_param_buf, sizeof(set_adv_param_buf), VIF);

    /* FSCI frames to find handle characteristics */
    FSCIFrame *find_char_value_handle           = CreateFSCIFrame(framer, GATTDB_OG, 0x04, batteryService_Handle, sizeof(batteryService_Handle), VIF); /* This will be redefined to find other handles */
    FSCIFrame *find_body_sl_handle;
    FSCIFrame *find_battery_lvl_handle;

    /* Write Attributes */
    FSCIFrame *write_HR_measurement;
    FSCIFrame *write_body_location;
    FSCIFrame *write_battery_level;

    /* Notification Status */
    FSCIFrame *gatt_notify;                        
    FSCIFrame *gatt_Send_Notification_Request;
    FSCIFrame *register_handles_write_not       = CreateFSCIFrame(framer, GATT_OG, SERVER_REGISTER_HANDLE_WRT_NOT_OC, Handle_Write_Notification, sizeof(Handle_Write_Notification), VIF);

    /*=========================================================================
     Start to send FSCI frames to set services, characteristics and descriptors
     ======================================================================= */

    /* Factory reset */
    printf("\nTX: FSCI-CPUReset.Request\n");
    SendFrame(framer, factory_rst);
    sleep(3);

    /*=========================================================================
                                Add GATT Service 
     ======================================================================= */
    printf("\n\t---- Add GATT Service ----\n");
    printf("\nTX: GATTDBDynamic-AddPrimaryServiceDeclaration.Request\n");
    actual_service_g = general_service;
    SendFrame(framer, gatt_psd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Service Changed\n");
    SendFrame(framer, gatt_service_changed_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCccd.Request\n");
    SendFrame(framer, add_cccd);
    sleep(1);

    /*=========================================================================
                                Add GAP Service 
     ======================================================================= */

    printf("\n\t---- Add GAP Service ----\n");
    printf("\nTX: GATTDBDynamic-AddPrimaryServiceDeclaration.Request\n");
    SendFrame(framer, gap_psd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Device Name\n");
    SendFrame(framer, gap_device_name_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Appeareance\n");
    SendFrame(framer, gap_appeareance_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Ppcp\n");
    SendFrame(framer, gap_ppcp_cdv);
    sleep(1);
    SendFrame(framer, add_cccd);
    sleep(1);

    /*=========================================================================
                                Add Heart Rate Service 
     ======================================================================= */

    printf("\n\t---- Add Heart Rate Service ----\n");
    printf("\nTX: GATTDBDynamic-AddPrimaryServiceDeclaration.Request\n");
    actual_service_g = heart_rate_service;
    SendFrame(framer, hr_psd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> HR Measurement\n");
    SendFrame(framer, hr_measurement_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCccd.Request\n");
    SendFrame(framer, add_cccd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Body Sensor Location\n");
    SendFrame(framer, hr_sensor_location_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Control Point\n");
    SendFrame(framer, hr_control_point_cdv);               
    sleep(1);

    /*=========================================================================
                                Add Battery Service 
     ======================================================================= */

    printf("\n\t---- Add Battery Service ----\n");
    printf("\nTX: GATTDBDynamic-AddPrimaryServiceDeclaration.Request\n");
    actual_service_g = battery_service;
    SendFrame(framer, battery_psd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Battery Level\n");
    SendFrame(framer, bat_level_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDescription.Request -> Char Format\n");
    SendFrame(framer, bat_char_pres_cd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCccd.Request\n");
    SendFrame(framer, add_cccd);
    sleep(1);

    /*=========================================================================
                            Add Device Information Service 
     ======================================================================= */

    printf("\n\t---- Add Device Information Service ----\n");
    printf("\nTX: GATTDBDynamic-AddPrimaryServiceDeclaration.Request\n");
    actual_service_g = general_service;
    SendFrame(framer, device_psd);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Manufacturer Name\n");
    SendFrame(framer, device_manufacturer_cdv);
    sleep(1);//GATTDBDynamicAddCccdIndication
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Model Number\n");
    SendFrame(framer, device_model_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Serial Number\n");
    SendFrame(framer, device_serial_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Hardware Revision\n");
    SendFrame(framer, device_hw_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Firmware Revision\n");
    SendFrame(framer, device_fw_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Software Revision\n");
    SendFrame(framer, device_sw_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> System Id\n");
    SendFrame(framer, device_system_cdv);
    sleep(1);
    printf("\nTX: GATTDBDynamic-AddCharacteristicDeclarationAndValue.Request -> Ieee Rcdl\n");
    SendFrame(framer, device_ieee_cdv);
    sleep(1);

    /*=========================================================================
                            Read Public Device Address
     ======================================================================= */

    printf("\n\t---- Read Public Device Address  ----\n");
    printf("\nTX: GAP-ReadPublicdeviceAddress.Request\n");
    SendFrame(framer, read_public_addr);
    sleep(1);

    /*=========================================================================
                               Set Advertising Data
     ======================================================================= */

    printf("\n\t---- Set Advertising Data  ----\n");
    printf("\nTX: GAP-SetAdvertisingData.Request\n");
    SendFrame(framer, set_adv_data);
    sleep(1);

    
    /*=========================================================================
                        Find Service Characteristic Handles
     ======================================================================= */
    printf("\n\t---- Handles for write notification  ----\n");

    /* Find Battery Service Characteristic handle value */
    printf("\nTX: GATTDB-WriteAttribute.Request -> Find Battery Service Handle\n");
    SendFrame(framer, find_char_value_handle);
    sleep(1);

    /* Save the previous handle */
    Battery_Service_Handle[0] = mLatestHandle[0];
    Battery_Service_Handle[1] = mLatestHandle[1];

   
    printf("\nTX: GATTServer-RegisterHandlesForWriteNotifications.Request\n");
    SendFrame(framer, register_handles_write_not);
    sleep(1);
    printf("\nTX: GATTServer-RegisterCallback.Request\n");
    SendFrame(framer, register_callback);
    sleep(1);

    /* Find Heart Rate Service handle */
    printf("\nTX: GATTDB-FindCharValueHandleInService.Request -> Find HR service handle\n");
    find_char_value_handle = CreateFSCIFrame(framer, GATTDB_OG, GATTDB_FIND_SERVICE_HANDLE , find_hr_serviceHandleReq, sizeof(find_hr_serviceHandleReq), VIF); // define 0x04
    SendFrame(framer, find_char_value_handle);
    sleep(1);

    /* Find Heart Rate Measurement handle */
    printf("\nTX: GATTDB-WriteAttribute.Request -> Find Heart Rate Measurement Handle\n");
    /* Save the previous handle */
    find_hr_measurementHandleReq[0] = HeartRate_Service_Handle[0] = mLatestHandle[0];   /* Save HR Service handle*/
    find_hr_measurementHandleReq[1] = HeartRate_Service_Handle[1] = mLatestHandle[1];
    find_char_value_handle = CreateFSCIFrame(framer, GATTDB_OG, FIND_CHAR_VALUE_HANDLE_SERVICE_REQ_OC, find_hr_measurementHandleReq, sizeof(find_hr_measurementHandleReq), VIF);
    SendFrame(framer, find_char_value_handle);
    sleep(1);

    /* Find Heart Rate Body Sensor Location Characteristic handle value */
    printf("\nTX: GATTDB-FindCharValueHandleInService.Request -> Find Body Sensor Location\n");
    find_body_sensorHandleReq[0] = HeartRate_Service_Handle[0];
    find_body_sensorHandleReq[1] = HeartRate_Service_Handle[1];
    /* Save the previous handle */
    HeartRate_Measurement_Handle[0] = hrs_gatt_notify[0] = mLatestHandle[0];    /* Save HR Measurement handle*/
    HeartRate_Measurement_Handle[1] = hrs_gatt_notify[1] = mLatestHandle[1]; 
    /* Create FSCI frames with the found handles */
    gatt_notify             = CreateFSCIFrame(framer, GATTDB_OG, FIND_CCCD_HANDLE_FOR_CHAR_HANDLE_REQ_OC, hrs_gatt_notify, sizeof(hrs_gatt_notify),VIF);
    find_body_sl_handle     = CreateFSCIFrame(framer, GATTDB_OG, FIND_CHAR_VALUE_HANDLE_SERVICE_REQ_OC, find_body_sensorHandleReq, sizeof(find_body_sensorHandleReq), VIF);
    SendFrame(framer, find_body_sl_handle);
    sleep(1);

    /* Find Battery Level Characteristic handle value */
    printf("\nTX: GATTDB-FindCharValueHandleInService.Request -> Find Battery Level Handle\n");
    find_battery_levelHandleReq[0] = Battery_Service_Handle[0];
    find_battery_levelHandleReq[1] = Battery_Service_Handle[1];
    /* Save the previous handle */
    BodySensor_Location_Handle[0] = mLatestHandle[0];
    BodySensor_Location_Handle[1] = mLatestHandle[1];
    find_battery_lvl_handle       = CreateFSCIFrame(framer, GATTDB_OG, FIND_CHAR_VALUE_HANDLE_SERVICE_REQ_OC, find_battery_levelHandleReq, sizeof(find_battery_levelHandleReq), VIF);
    SendFrame(framer, find_battery_lvl_handle);
    sleep(1);

    /* Save the previous handle */
    Battery_Level_Handle[0] = mLatestHandle[0];
    Battery_Level_Handle[1] = mLatestHandle[1];

    /*=========================================================================
                    Set  Characteristics initial configurations
     ======================================================================= */

    /* Write attribute to Heart Rate Measurement characteristic */
    printf("\nTX: GATTDB-WriteAttribute.Request -> Heart Rate Measurement\n");

    update_HeartRate_Measurement[0] = HeartRate_Measurement_Handle[0];
    update_HeartRate_Measurement[1] = HeartRate_Measurement_Handle[1];
    update_HeartRate_Measurement[4] = gHrs_8BitHeartRateFormat_c;
    update_HeartRate_Measurement[5] = HR_value;

    write_HR_measurement = CreateFSCIFrame(framer, GATTDB_OG, WRITE_ATT_OC, update_HeartRate_Measurement, sizeof(update_HeartRate_Measurement), VIF);
    SendFrame(framer, write_HR_measurement);
    sleep(2);

    /* Write attribute to Heart Rate Body Sensor Location characteristic */
    printf("\nTX: GATTDB-WriteAttribute.Request -> Body Sensor Location\n");
    
    update_Battery_Measurement[0] = BodySensor_Location_Handle[0];
    update_Battery_Measurement[1] = BodySensor_Location_Handle[1];
    update_Battery_Measurement[4] =  gHrs_BodySensorLocChest_c;

    write_body_location = CreateFSCIFrame(framer, GATTDB_OG, WRITE_ATT_OC, update_Battery_Measurement, sizeof(update_Battery_Measurement),VIF);
    SendFrame(framer, write_body_location);
    sleep(2);

    /* Write attribute to Battery Level characteristic */
    printf("\nTX: GATTDB-WriteAttribute.Request -> Battery Level\n");
    update_Battery_Measurement[0] = Battery_Level_Handle[0];
    update_Battery_Measurement[1] = Battery_Level_Handle[1];
    update_Battery_Measurement[4] =  0x63;      /* Battery level = 99% */
    write_battery_level = CreateFSCIFrame(framer, GATTDB_OG, WRITE_ATT_OC, update_Battery_Measurement, sizeof(update_Battery_Measurement),VIF);
    SendFrame(framer, write_battery_level);
    sleep(2);

    /*=========================================================================
        All configurations done,        Start Advertising
     ======================================================================= */

    printf("\n\t----------- Start Advertising -----------\n");
    printf("\nTX: GAP-SetAdvertisingParameters.Request\n");
    SendFrame(framer, set_adv_parameters);
    sleep(1);
    printf("\nTX: StartAdvertising.Request\n");
    SendFrame(framer, start_advertising);
    sleep(1);

    demo_HRs_State = waiting_connection; /* Change program status */

    printf("===================================================================================\n");
	printf("----Please open NXP IoT Toolbox and select Heart Rate, and then NXP_HRS device.----\n");    
    printf("===================================================================================\n");
    
    while(1)
    {
        switch(demo_HRs_State)
        {

            case waiting_connection:

                printf("=========== waiting for connection ===========\n");
                sleep(10);
                break;

            case device_connected:

                printf("=========== Started to send Heart Rate measurements ===========\n");

                counter = 0;
                SendFrame(framer, write_body_location); /* Refresh sensor body location */
                sleep(2);
                while ( demo_HRs_State == device_connected) 
                {
                    sleep(1);
                    counter++;
                    if(counter >= 0xFF)
                        counter = 0x00;

                    /* check counter value to see what we need to notify */
                    if (counter % HRS_HEART_RATE_REPORT_INTERVAL == 0) 
                    {
                        // sequence to refresh heart rate value
                        update_HeartRate_Measurement[5] = 100 + (rand() % 51);  /* Random heart rate measurement*/
                        write_HR_measurement = CreateFSCIFrame(framer, GATTDB_OG, WRITE_ATT_OC, update_HeartRate_Measurement, sizeof(update_HeartRate_Measurement), VIF);
                        SendFrame(framer, write_HR_measurement);                /* Refresh heart rate measurement */
                        sleep(1);

                        SendFrame(framer, gatt_notify);                       /* Notify the new data to GATT Layer */
                        sleep(1);
                        gatt_Send_Notification_Request_buf[1] = HeartRate_Measurement_Handle[0];
                        gatt_Send_Notification_Request_buf[2] = HeartRate_Measurement_Handle[1];
                        gatt_Send_Notification_Request = CreateFSCIFrame(framer, GATT_OG, SEND_NOTIFICATION_REQUEST, gatt_Send_Notification_Request_buf, sizeof(gatt_Send_Notification_Request_buf), VIF);
                        SendFrame(framer, gatt_Send_Notification_Request);   /* Notify the new data to GATT Layer */
                        sleep(1);                       
                    }

                    if (counter % HRS_BATTERY_LEVEL_REPORT_INTERVAL == 0) 
                    {
                        update_Battery_Measurement[0] = Battery_Level_Handle[0];
                        update_Battery_Measurement[1] = Battery_Level_Handle[1];
                        update_Battery_Measurement[4] = rand() % 101; /* Random battery level */
                        write_battery_level = CreateFSCIFrame(framer, GATTDB_OG, WRITE_ATT_OC, update_Battery_Measurement, sizeof(update_Battery_Measurement),VIF);
                        SendFrame(framer, write_battery_level);             /* Refresh battery level measurement */      
                        SendFrame(framer, write_body_location);             /* Refresh sensor body location */
                        sleep(1);
                    }
                }

                break;

            case device_disconnected:

                demo_HRs_State =  start_advertising_service;
                sleep(2);

                break;

            case start_advertising_service:

                printf("\nTX: StartAdvertising.Request\n");
                SendFrame(framer, set_adv_data);
                sleep(1);
                SendFrame(framer, set_adv_parameters);
                sleep(1);
                SendFrame(framer, start_advertising);
                sleep(1);
                demo_HRs_State = waiting_connection;

                printf("===================================================================================\n");
                printf("----Please open NXP IoT Toolbox and select Heart Rate, and then NXP_HRS device.----\n");    
                printf("===================================================================================\n");

                break;       

            default:
                break;
        }        
    }

    while(1)
    {

    }

}                                                                                   