/* xymodem.c */
#include "xymodem.h"
//#include "xymodem_port.h"

#define APP_ENABLE_DEBUG_INFO  0

unsigned int buf_filelen(unsigned char *ptr);

/*****************************************************************************************
** 函数名称： ymodem_init
** 函数功能： 该函数自动填充mblock配置信息，但是缓冲区仍需要用户在传入mblock之前预先分配并传入指针
** 入口参数：
** 返 回 值: 0: 传输成功
**           1: 传输错误
**           2: 取消传输
**          -1: 传输超时
******************************************************************************************/
/* This function would fill the structure mblock. */
int ymodem_init(
    modem_struct *mblock /* [OUT] setup the mblock */
)
{
    int stat;
    int max_tries = MODEM_MAX_RETRIES;
    int crc_tries = MODEM_CRC_RETRIES;
    unsigned char *bufptr = mblock->buf;
    unsigned char *namptr = mblock->filename;

    mblock->nxt_num = 0;
    mblock->modemtype = 2; /*  */
    mblock->rec_err = 0;
    mblock->crc_mode = 1;

    while (max_tries-- > 0)
    {
        /* try to get one frame of data from host. */
        stat = modem_recvdata(mblock);

        if (0 == stat)  //接收成功
        {
            /* 提取文件名字符串 */
            while (*bufptr != '\0')
            {
                *namptr++ = *bufptr++;
            }
            *namptr = '\0'; /* 文件名以\0结尾 */
#if APP_ENABLE_DEBUG_INFO
            PRINTF("FileName: %s\r\n", mblock->filename);
#endif /* APP_ENABLE_DEBUG_INFO */
            /* 提取文件长度符串 */
            bufptr++;
            while (*bufptr == ' ')
            {
                bufptr++;
            }

            /* 提取文件长度 */
            mblock->filelen = buf_filelen(bufptr);
#if APP_ENABLE_DEBUG_INFO
            PRINTF("FileName: %d\r\n", mblock->filelen);
#endif
            // other data;
            XYModem_Uart_SendByte(MODEM_ACK);
            return 0;
        }
        else if (2 == stat)  //取消传输
        {
            return 2;
        }
        else if (-3 == stat)
        {
            if (mblock->cur_num == 1)
            {
                mblock->modemtype = 1;
                mblock->nxt_num = 2;
                return 1;
            }
        }
        else  //超时或校验方式不对
        {
            if (crc_tries-- <= 0)
            {
                crc_tries = MODEM_CRC_RETRIES;
                mblock->crc_mode = (mblock->crc_mode + 1) & 1;
            }
        }
    }
    return -1;
}

/*****************************************************************************************
** 函数名称：modem_recvdata
** 函数功能：获取一包数据包
** 入口参数：
** 返 回 值： 0:成功接收数据
**          -1:接收超时
**          -2:帧错误(校验错误)
**          -3:帧序号序号错误(严重错误)
**           1:消息结束
**           2:取消发送
******************************************************************************************/
int modem_recvdata(modem_struct *mblock)
{
    int stat, hd_found = 0, i;
    int can_counter = 0, eot_counter = 0;
    unsigned char *in_ptr = mblock->buf;
    int cksum;
    unsigned char ch, blk, cblk, crch, crcl;

    //Uart_RxEmpty();  /* 清空接收缓冲区, 准备接收数据包 */

    /* 发送请求信号 */
    if (mblock->nxt_num == 0) /* 请求第一个数据包 */
    {
        if (mblock->crc_mode) /* 如果上一个包的校验成功，则请求新包 */
        {
            XYModem_Uart_SendByte(MODEM_C);
        }
        else /* 如果上一个包的校验失败，则请求重发 */
        {
            XYModem_Uart_SendByte(MODEM_NAK);
        }
    }
    else /* 连续传输数据中 */
    {
        if (mblock->rec_err) /* 如果传输错误，则请求重传上一个数据包 */
        {
            XYModem_Uart_SendByte(MODEM_NAK);
        }
        else
        {
            if (mblock->nxt_num == 1)
            {
                if (mblock->crc_mode) /* 如果上一个包校验成功，则请求新包 */
                {
                    XYModem_Uart_SendByte(MODEM_C);
                }
                else  /* 如果上一个包的校验失败，则请求重发 */
                {
                    XYModem_Uart_SendByte(MODEM_NAK);
                }
            }
            else /* 应答发送成功 */
            {
                XYModem_Uart_SendByte(MODEM_ACK);
            }
        }
    }

    /* Wait for the header byte of the package
    * 等待接收包头字节，包头字节表示了本包的属性
     */
    hd_found = 0;
    while (!hd_found)
    {
        stat = XYModem_Uart_RecvByteTimeout(&ch);
        if (stat == 0)
        {
            switch (ch)
            {
            case MODEM_SOH: /* 128字节数据的包 */
                hd_found = 1;
                mblock->len = 128;
#if APP_ENABLE_DEBUG_INFO
                PRINTF("Get SOH.\r\n");
#endif
                break;
            case MODEM_STX: /* 1024字节数据的包 */
                hd_found = 1;
                mblock->len = 1024;
#if APP_ENABLE_DEBUG_INFO
                PRINTF("Get STX.\r\n");
#endif
                break;
            case MODEM_CAN: /* 取消发送 */
                can_counter++;
                if ((can_counter) >= MODEM_CAN_COUNT)
                {
                    return 2;
                }
                break;
            case MODEM_EOT:  /* 文件传输结束 */
#if APP_ENABLE_DEBUG_INFO
                PRINTF("Get EOF.\r\n");
#endif
                eot_counter++;
                if ((eot_counter) >= MODEM_EOT_COUNT)
                {
                    XYModem_Uart_SendByte(MODEM_ACK);
                    if (mblock->modemtype == 2)  // Ymodem协议为批文件传输协议
                    {
                        XYModem_Uart_SendByte(MODEM_C);  //单个文件需 C ACK C 后会自动停止传输
                        XYModem_Uart_SendByte(MODEM_ACK);
                        XYModem_Uart_SendByte(MODEM_C);
                        modem_cancle();  //多个文件强制停止传输
                    }
                    return 1;
                }
                break;
            default:
                break;
            }
        }
        else
        {
            return -1;
        }
    }

    /* Fetch the value of the data package index.
    * 获取包编号
    */
    stat = XYModem_Uart_RecvByteTimeout(&blk);
#if APP_ENABLE_DEBUG_INFO
    PRINTF("Frame Index: 0x%X.\r\n", blk);
#endif
    if (stat != 0)
    {
        return -1;
    }


    /* Fetch the complementary value of data package index.
    * 获取包编号的补码
    */
    stat = XYModem_Uart_RecvByteTimeout(&cblk);
#if APP_ENABLE_DEBUG_INFO
    PRINTF("FRAME Index N: 0x%X.\r\n", cblk);
#endif
    if (stat != 0)
    {
        return -1;
    }

    /* Fetch the 128 data byte.
    * 获取数据包的128字节
    */
#if APP_ENABLE_DEBUG_INFO
    PRINTF("FRAME Data: \r\n");
#endif
    for (i = 0; i < mblock->len; i++)
    {
#if APP_ENABLE_DEBUG_INFO
        if (0 == (i % 16))
        {
            PRINTF("\r\n");
        }
#endif /* APP_ENABLE_DEBUG_INFO */
        stat = XYModem_Uart_RecvByteTimeout(in_ptr);
#if APP_ENABLE_DEBUG_INFO
        PRINTF("0x%2X ", *in_ptr);
#endif
        in_ptr++;
        if (stat != 0)
        {
            return -1;
        }
    }

    /* Fetch the 2 CRC byte.
    * 获取2字节校验字
    */
    stat = XYModem_Uart_RecvByteTimeout(&crch); /* CRC0, CRC_HIGH. */
#if APP_ENABLE_DEBUG_INFO
    PRINTF("\r\nCRC_H: 0x%X\r\n", crch);
#endif
    if (stat != 0)
    {
        return -1;
    }
    if (mblock->crc_mode)
    {
        stat = XYModem_Uart_RecvByteTimeout(&crcl); /* CRC1, CRC_LOW */
#if APP_ENABLE_DEBUG_INFO
        PRINTF("CRC_L: 0x%X\r\n", crcl);
#endif
        if (stat != 0)
        {
            return -1;
        }
    }

    /* Validate the date in the package. */

    /* index value error. */
    if ((blk ^ cblk) != 0xff)
    {
#if APP_ENABLE_DEBUG_INFO
        PRINTF("Frame Number Error.\r\n");
#endif
        return (-2);
    }

    /* Check the CRC. */
    if (mblock->crc_mode) /* CRC XOR. */
    {
        in_ptr = mblock->buf;
        cksum = 0;

        for (stat = mblock->len; stat > 0; stat--)
        {
            cksum = cksum ^ (int)(*in_ptr++) << 8;
            for (i = 8; i != 0; i--)
            {
                if (cksum & 0x8000)
                {
                    cksum = cksum << 1 ^ 0x1021;
                }
                else
                {
                    cksum = cksum << 1;
                }
            }
        }

        cksum &= 0xffff;
#if APP_ENABLE_DEBUG_INFO
        PRINTF("Calc CRC value: 0x%X\r\n", cksum);
#endif
        if (cksum != (crch << 8 | crcl))
        {
            mblock->rec_err = 1;
#if APP_ENABLE_DEBUG_INFO
            PRINTF("Calc CRC Error.\r\n");
#endif
            return (-2);
        }
    }
    else /* CRC Checksum. */
    {
        for (i = 0; i < mblock->len; i++)  //和校验方式
        {
            cksum += mblock->buf[i];
        }
        if ((cksum & 0xff) != crch)
        {
            mblock->rec_err = 1;
            return (-2);
        }
    }

    /* Updat the static information. */
    mblock->cur_num = blk;
    if (blk != mblock->nxt_num)  /* blk检查 */
    {
        /* mblock->nxt_num存放的是上一个数据包的下一个包id，
        * 如果经历了前面那么多检验，最后发现包id没有连续上，即使数据对了也是白玩。
        * 这种情况下，需要让主机重新发送期望的，能跟当前包连上的next包。
        */
        return (-3);
    }

    //mblock->nxt_num++;
    mblock->nxt_num = mblock->cur_num + 1;
    mblock->rec_err = 0;
    return 0;
}

unsigned int buf_filelen(unsigned char *ptr)
{
  int datatype = 10, result = 0;

  if (ptr[0] == '0' && (ptr[1] == 'x' || ptr[1] == 'X')) /* 修改原来为(ptr[1] == 'x' && ptr[1] == 'X') */
  {
    datatype = 16; /* 16进制 */
    ptr += 2;
  }

    for (; *ptr !='\0'; ptr++)
    {
        if (*ptr >= '0' && *ptr <= '9')
        {
            result = result * datatype + *ptr - '0';
        }
        else
        {
            if (datatype == 10)
            {
                return result; /* 若是十进制，一旦遇到非数字，就该返回了 */
            }
            else
            {
                if (*ptr >= 'A' && *ptr <= 'F')
                {
                    result = result * 16 + *ptr - 55;  // 55 = 'A'-10
                }
                else if (*ptr >= 'a' && *ptr <= 'f')
                {
                    result = result * 16 + *ptr - 87;  // 87 = 'a'-10
                }
                else /* 哪怕是十六进制，一旦遇到非数字字符，也该返回了 */
                {
                    return result;
                }
            }
        }
    }
    return result;
}


void modem_cancle(void)
{
    XYModem_Uart_SendByte(0x18);
    XYModem_Uart_SendByte(0x18);
    XYModem_Uart_SendByte(0x18);
    XYModem_Uart_SendByte(0x18);
    XYModem_Uart_SendByte(0x18);
}

/* EOF. */

