/*
 * @brief Secondary loader common
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#ifndef SL_COMMON_H
#define SL_COMMON_H

#include "chip.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "sl_protocol.h"
#include "sl_pinsetup.h"
#include "sl_hostif.h"

extern IMG_HEADER_T *pImageHeader1;
extern IMG_HEADER_T *pImageHeader2;

/* Verify that application's CRC is correct. Used to determine if a valid
   application is located in FLASH before jumping to it. Returns 0 if
   the check is valid or computed CRC32 value. */
uint32_t checkAppCRC(uint8_t app_number);

uint32_t computeAppCRC(uint8_t app_number);

uint32_t computeCrc32(uint32_t *data, uint32_t len);

/* It is recommended to call this function on system boot or transfer to
   the app. */
void RestoreSystem(void);

/* Overlay pin config structure on work area. This defines the detected or
   app-configured host interface. When booted from FLASH, this structure is
   not configured and is autodetected. When booted from the app, this
   structure is populated. */
extern SL_PINSETUP_T *pinCfgData;

/* Save IOCON states of all pins this loader can use except the host IRQ
   pin state */
void saveStatesIOCON(void);

/* Restore original IOCON states of all pins this loader can use except
   the host IRQ pin state */
void restoreStatesIOCON(void);

/* Setup SPI clock pin autodetect to determine which SPI clock pin is active */
void setupSpiClkPinAutoDetect(void);

/* SPI clock loop detection */
bool spiClkPinAutoDetectLoop(void);

/* Disable SPI clock detection */
void stopSpiClkPinAutoDetect(void);

/* Setup SPI MOSI pin autodetect to determine which SPI MOSI pin is active */
void setupSpiMosiPinAutoDetect(void);

/* SPI MOSI loop detection */
bool spiMosiPinAutoDetectLoop(void);

/* Disable SPI MOSI detection */
void stopSpiMosiPinAutoDetect(void);

/* Setup SPI SSEL pin autodetect to determine which SPI SSEL pin is active */
void setupSpiSselPinAutoDetect(void);

/* SPI SSEL loop detection */
bool spiSselPinAutoDetectLoop(void);

/* Disable SPI SSEL detection */
void stopSpiSselPinAutoDetect(void);

/* Cleanup function, should be called before booting App */
extern void CleanUpSystem(void);

/* App boot function (loads SP and jumper to reset vector for app) */
extern void bootValidApp(uint32_t appBootAddr);

extern void doCleanBoot(uint32_t appBootAddr);

/* Determine number of bytes transferred on a DMA channel */


#ifdef __cplusplus
}
#endif

#endif /* SL_COMMON_H */
